/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.valueset;

import ca.uhn.fhir.context.FhirVersionEnum;
import de.fhlintstone.accessors.IAccessorProvider;
import de.fhlintstone.accessors.implementations.IFrameworkTypeLocator;
import de.fhlintstone.accessors.model.IValueSetAccessor;
import de.fhlintstone.fhir.dependencies.IDependencyGraph;
import de.fhlintstone.generator.GeneratorException;
import de.fhlintstone.generator.IGeneratedTypeNameRegistry;
import de.fhlintstone.packages.FhirResourceType;
import de.fhlintstone.packages.IPackageRegistry;
import de.fhlintstone.process.config.ProcessConfiguration;
import java.net.URI;
import javax.inject.Inject;
import javax.inject.Named;
import lombok.extern.slf4j.XSlf4j;

/**
 * Default implementation of {@link IValueSetGenerator}.
 */
@Named
@XSlf4j
public class ValueSetGenerator implements IValueSetGenerator {

    private final IPackageRegistry packageRegistry;
    private final IAccessorProvider accessorProvider;
    private final IFrameworkTypeLocator frameworkTypeLocator;
    private final IGeneratedTypeNameRegistry generatedTypeNameRegistry;
    private final IConstantGenerator constantGenerator;
    private final ICodeEmitter codeEmitter;

    /**
     * Constructor for dependency injection.
     *
     * @param packageRegistry           the {@link IPackageRegistry} to use
     * @param accessorProvider          the {@link IAccessorProvider} to use
     * @param frameworkTypeLocator      the {@link IFrameworkTypeLocator} to use
     * @param generatedTypeNameRegistry the {@link IGeneratedTypeNameRegistry} to
     *                                  use
     * @param constantGenerator         the {@link IConstantGenerator} to use
     * @param codeEmitter               the {@link ICodeEmitter} to use
     */
    @Inject
    public ValueSetGenerator(
            IPackageRegistry packageRegistry,
            IAccessorProvider accessorProvider,
            IFrameworkTypeLocator frameworkTypeLocator,
            IGeneratedTypeNameRegistry generatedTypeNameRegistry,
            IConstantGenerator constantGenerator,
            ICodeEmitter codeEmitter) {
        super();
        this.packageRegistry = packageRegistry;
        this.accessorProvider = accessorProvider;
        this.frameworkTypeLocator = frameworkTypeLocator;
        this.generatedTypeNameRegistry = generatedTypeNameRegistry;
        this.constantGenerator = constantGenerator;
        this.codeEmitter = codeEmitter;
    }

    @Override
    public boolean generate(ProcessConfiguration globalConfiguration, IDependencyGraph dependencyGraph) {
        logger.entry(globalConfiguration);
        var result = true;
        final var valueSetEnumConfigurations = globalConfiguration.getValueSetEnums();
        if (valueSetEnumConfigurations.isEmpty()) {
            logger.info("No ValueSet / enum generation rules configured.");
        } else {
            // The ValueSet enums do not (currently) have any dependencies, so we can simply iterate over the configured
            // values in any order.
            for (final var valueSetEnumConfiguration : valueSetEnumConfigurations) {
                try {
                    generate(
                            valueSetEnumConfiguration.getValueSet(),
                            globalConfiguration.getFhirVersion(),
                            globalConfiguration.getOutputPath());
                } catch (final GeneratorException e) {
                    logger.error(
                            "Unable to generate an enum to represent the value set {}",
                            valueSetEnumConfiguration.getValueSet(),
                            e);
                    result = false;
                }
            }
        }
        return logger.exit(result);
    }

    /**
     * Generates the code for a single ValueSet
     * @param valueSetURI the canonical URL of the valueSet
     * @param fhirVersion the FHIR version to use
     * @param outputPath the path to write the generated source to
     * @throws GeneratorException
     */
    private void generate(URI valueSetURI, FhirVersionEnum fhirVersion, String outputPath) throws GeneratorException {
        logger.entry(valueSetURI);
        final var qualifiedName = this.generatedTypeNameRegistry.getEnumName(valueSetURI);
        if (qualifiedName.isPresent()) {
            final var accessor = createValueSetAccessor(fhirVersion, valueSetURI);
            logger.info(
                    "Generating enum {} for ValueSet {}", qualifiedName.get().canonicalName(), valueSetURI);
            final var enumData = EnumData.builder()
                    .withClassName(qualifiedName.get())
                    .withValueSetName(accessor.getName().orElseThrow())
                    .withValueSetURL(accessor.getUrl().orElseThrow())
                    .withDescription(accessor.getDescription())
                    .withCodingType(this.frameworkTypeLocator.getCodingType())
                    .withOutputPath(outputPath)
                    .withEnumConstants(this.constantGenerator.generateEnumConstants(accessor))
                    .build();
            this.codeEmitter.generate(enumData);
        } else {
            logger.info("No configuration to generate an enum for ValueSet {}", valueSetURI);
        }
        logger.exit();
    }

    /**
     * Creates the wrapper that provides release-independent access to the ValueSet
     * information.
     *
     * @param valueSetURI the canonical URL of the valueSet
     * @param fhirVersion the FHIR version to use
     * @return
     * @throws GeneratorException
     */
    private IValueSetAccessor createValueSetAccessor(FhirVersionEnum fhirVersion, URI valueSetURI)
            throws GeneratorException {
        logger.entry(fhirVersion, valueSetURI);
        final var valueSetResource = this.packageRegistry.getUniqueResource(FhirResourceType.VALUE_SET, valueSetURI);
        if (valueSetResource.isEmpty()) {
            throw logger.throwing(new GeneratorException(
                    String.format("Unable to locate resource for ValueSet URI %s", valueSetURI)));
        }
        if (!fhirVersion.equals(valueSetResource.get().getStructureFhirVersionEnum())) {
            throw logger.throwing(
                    new GeneratorException(String.format("FHIR version mismatch for ValueSet URI %s", valueSetURI)));
        }
        final var result = this.accessorProvider.provideValueSetAccessor(valueSetResource.get());
        return logger.exit(result);
    }
}
