/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.valueset;

import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.model.FilterOperator;
import de.fhlintstone.accessors.model.ICodeSystemAccessor;
import de.fhlintstone.accessors.model.IConceptDefinitionComponentAccessor;
import de.fhlintstone.accessors.model.IConceptPropertyComponentAccessor;
import de.fhlintstone.accessors.model.IConceptSetFilterComponentAccessor;
import de.fhlintstone.accessors.model.IPrimitiveTypeAccessor;
import de.fhlintstone.accessors.model.PrimitiveDatatype;
import de.fhlintstone.accessors.model.StringifiedValue;
import de.fhlintstone.fhir.FhirUtilities;
import de.fhlintstone.generator.GeneratorException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import javax.inject.Inject;
import javax.inject.Named;
import lombok.extern.slf4j.XSlf4j;

/**
 * Default implementation of {@link IFilterCalculator}.
 */
@Named
@XSlf4j
public class FilterCalculator implements IFilterCalculator {

    private static final String CODE_NAME = "code";
    private static final String CONCEPT_NAME = "concept";
    private static final String DESIGNATION_NAME = "designation";
    private static final Pattern SEARCH_SEMANTICS_PREFIX = Pattern.compile("^(eq|ne|gt|lt|ge|le|sa|eb|ap)");
    private static final Pattern COMMA_BLANK_SPLIT = Pattern.compile("\\s*,\\s*", Pattern.UNICODE_CHARACTER_CLASS);

    /**
     * Default constructor for dependency injection.
     */
    @Inject
    public FilterCalculator() {
        super();
    }

    @Override
    public ImmutableList<IConceptDefinitionComponentAccessor> getMatchingConcepts(
            String valueSetUrl,
            IConceptSetFilterComponentAccessor filterAccessor,
            ICodeSystemAccessor codeSystemAccessor)
            throws GeneratorException {
        logger.entry();
        final var optFilterAccessorOp = filterAccessor.getOp();
        final var optFilterAccessorProperty = filterAccessor.getProperty();
        final var optFilterAccessorValue = filterAccessor.getValue();
        if (optFilterAccessorOp.isEmpty() || optFilterAccessorProperty.isEmpty() || optFilterAccessorValue.isEmpty()) {
            throw logger.throwing(new GeneratorException(
                    String.format("Filter %s on ValueSet %s is invalid", filterAccessor, valueSetUrl)));
        }

        final var filterProperty = optFilterAccessorProperty.get();
        final var filterValue = optFilterAccessorValue.get();
        // Flatten the code system concepts for iterating through them in the filters
        // By default a hierarchical code system is only returning the first layer
        final var conceptAccessors = FhirUtilities.flattenConceptHierarchy(codeSystemAccessor.getConcept());
        return logger.exit(
                switch (optFilterAccessorOp.get()) {
                    case EQUAL -> calculateEqualFilter(valueSetUrl, filterProperty, filterValue, conceptAccessors);
                    case ISA -> calculateIsAFilter(valueSetUrl, filterProperty, filterValue, conceptAccessors);
                    case DESCENDENTOF -> calculateDescendentOfFilter(
                            valueSetUrl, filterProperty, filterValue, conceptAccessors);
                    case ISNOTA -> calculateIsNotAFilter(valueSetUrl, filterProperty, filterValue, conceptAccessors);
                    case REGEX -> calculateRegexFilter(valueSetUrl, filterProperty, filterValue, conceptAccessors);
                    case IN -> calculateInFilter(valueSetUrl, filterProperty, filterValue, conceptAccessors);
                    case NOTIN -> calculateNotInFilter(valueSetUrl, filterProperty, filterValue, conceptAccessors);
                    case GENERALIZES -> calculateGeneralizesFilter(
                            valueSetUrl, filterProperty, filterValue, conceptAccessors);
                    case CHILDOF -> calculateChildOfFilter(valueSetUrl, filterProperty, filterValue, conceptAccessors);
                    case DESCENDENTLEAF -> calculateDescendentLeafFilter(
                            valueSetUrl, filterProperty, filterValue, conceptAccessors);
                    case EXISTS -> calculateExistsFilter(valueSetUrl, filterProperty, filterValue, conceptAccessors);
                });
    }

    /**
     * Filter out all code system concepts according to the equal (=) filter.
     *
     * @param valueSetUrl    The value set getting processed
     * @param filterProperty The name of the property to filter on. For equal
     *                       filters every defined code system property and
     *                       designation is allowed.
     * @param filterValue    The value to filter on. Search semantics for integer,
     *                       datetime, decimals can be prefixex
     * @param concepts       The list of all concepts of the code system.
     * @return A list of concepts matching the filter.
     * @see <a href=
     *      "https://www.hl7.org/fhir/codesystem.html#filters">https://www.hl7.org/fhir/codesystem.html#filters</a>
     * @see <a href=
     *      "https://www.hl7.org/fhir/valueset-filter-operator.html">https://www.hl7.org/fhir/valueset-filter-operator.html</a>
     */
    private ImmutableList<IConceptDefinitionComponentAccessor> calculateEqualFilter(
            String valueSetUrl,
            String filterProperty,
            String filterValue,
            ImmutableList<IConceptDefinitionComponentAccessor> concepts)
            throws GeneratorException {
        logger.entry(filterProperty, filterValue);
        if (filterProperty.equals(DESIGNATION_NAME)) {
            // TODO #33 Implement filter on designation
            throw logger.throwing(new GeneratorException(String.format(
                    "ValueSet %s: Equal filter on designations are currently not supported", valueSetUrl)));
        }

        final var result = new ArrayList<IConceptDefinitionComponentAccessor>();
        for (final var concept : concepts) {
            for (final var property : concept.getProperty()) {
                if (checkPropertyEquality(property, valueSetUrl, filterProperty, filterValue)) {
                    result.add(concept);
                }
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    private boolean checkPropertyEquality(
            IConceptPropertyComponentAccessor property, String valueSetUrl, String filterProperty, String filterValue)
            throws GeneratorException {
        final var propertyCode = property.getCode();
        final var propertyValue = property.getValue();
        if (propertyValue.isPresent() && propertyValue.get() instanceof final IPrimitiveTypeAccessor primitiveValue) {
            if (propertyCode.isPresent() && propertyCode.get().equals(filterProperty)) {
                if (!primitiveValue.isType(PrimitiveDatatype.STRING)
                        && !primitiveValue.isType(PrimitiveDatatype.INTEGER)) {
                    // TODO #44 Add support for all data types
                    throw logger.throwing(new GeneratorException(String.format(
                            "ValueSet %s: Filter on %s: Currently only StringType and IntegerType are "
                                    + "supported for filter",
                            valueSetUrl, filterProperty)));
                }
                if (primitiveValue.isType(PrimitiveDatatype.INTEGER)
                        && SEARCH_SEMANTICS_PREFIX.matcher(propertyCode.get()).matches()) {
                    // TODO #34 Search semantics for int, datetime, decimal
                    throw logger.throwing(new GeneratorException(String.format(
                            "ValueSet %s: Filter on %s: Search semantics are currently not supported",
                            valueSetUrl, filterProperty)));
                }

                final Optional<StringifiedValue> stringifiedValue = primitiveValue.getStringifiedValue();
                return stringifiedValue.isPresent()
                        && stringifiedValue.get().getValue().equals(filterValue);
            }
        } else {
            throw logger.throwing(new GeneratorException(String.format(
                    "ValueSet %s: Filter on %s: Non-primitive values are not supported", valueSetUrl, filterProperty)));
        }
        return false;
    }

    /**
     * Filter out all code system concepts according to the is-a filter.
     *
     * @param valueSetUrl    The value set getting processed
     * @param filterProperty The name of the property to filter on. Must be
     *                       'concept'.
     * @param filterValue    The value to filter on. Represents the code of the
     *                       concept.
     * @param concepts       The list of all concepts of the code system.
     * @return A list of concepts matching the filter. The mentioned concepts and
     *         its transitive children.
     * @see <a href=
     *      "https://www.hl7.org/fhir/codesystem.html#filters">https://www.hl7.org/fhir/codesystem.html#filters</a>
     * @see <a href=
     *      "https://www.hl7.org/fhir/valueset-filter-operator.html">https://www.hl7.org/fhir/valueset-filter-operator.html</a>
     */
    private ImmutableList<IConceptDefinitionComponentAccessor> calculateIsAFilter(
            String valueSetUrl,
            String filterProperty,
            String filterValue,
            ImmutableList<IConceptDefinitionComponentAccessor> concepts)
            throws GeneratorException {
        logger.entry(filterProperty, filterValue);
        checkFilterPropertyIsOnlyConcept(valueSetUrl, filterProperty, FilterOperator.ISA.toString());

        final var result = new ArrayList<IConceptDefinitionComponentAccessor>();
        for (final var concept : concepts) {
            final var conceptCode = concept.getCode();
            if (conceptCode.isPresent() && conceptCode.get().equals(filterValue)) {
                result.add(concept);
                result.addAll(calculateTransitiveChildren(concept));
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    /**
     * Filter out all code system concepts according to the descendent-of filter.
     *
     * @param valueSetUrl    The value set getting processed
     * @param filterProperty The name of the property to filter on. Must be
     *                       'concept'.
     * @param filterValue    The value to filter on. Represents the code of the
     *                       concept.
     * @param concepts       The list of all concepts of the code system
     * @return A list of concepts matching the filter. Only the transitive children
     *         of the mentioned concept.
     * @see <a href=
     *      "https://www.hl7.org/fhir/codesystem.html#filters">https://www.hl7.org/fhir/codesystem.html#filters</a>
     * @see <a href=
     *      "https://www.hl7.org/fhir/valueset-filter-operator.html">https://www.hl7.org/fhir/valueset-filter-operator.html</a>
     */
    private ImmutableList<IConceptDefinitionComponentAccessor> calculateDescendentOfFilter(
            String valueSetUrl,
            String filterProperty,
            String filterValue,
            ImmutableList<IConceptDefinitionComponentAccessor> concepts)
            throws GeneratorException {
        logger.entry(filterProperty, filterValue);
        checkFilterPropertyIsOnlyConcept(valueSetUrl, filterProperty, FilterOperator.DESCENDENTOF.toString());

        final var result = new ArrayList<IConceptDefinitionComponentAccessor>();
        for (final var concept : concepts) {
            final var conceptCode = concept.getCode();
            if (conceptCode.isPresent() && conceptCode.get().equals(filterValue)) {
                result.addAll(calculateTransitiveChildren(concept));
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    /**
     * Filter out all code system concepts according to the is-not-a filter.
     *
     * @param valueSetUrl    The value set getting processed
     * @param filterProperty The name of the property to filter on. Must be
     *                       'concept'.
     * @param filterValue    The value to filter on. Represents the code of the
     *                       concept.
     * @param concepts       The list of all concepts of the code system.
     * @return A list of concepts matching the filter. Contains all concepts except
     *         the mentioned concept and it's children.
     * @see <a href=
     *      "https://www.hl7.org/fhir/codesystem.html#filters">https://www.hl7.org/fhir/codesystem.html#filters</a>
     * @see <a href=
     *      "https://www.hl7.org/fhir/valueset-filter-operator.html">https://www.hl7.org/fhir/valueset-filter-operator.html</a>
     */
    private ImmutableList<IConceptDefinitionComponentAccessor> calculateIsNotAFilter(
            String valueSetUrl,
            String filterProperty,
            String filterValue,
            ImmutableList<IConceptDefinitionComponentAccessor> concepts)
            throws GeneratorException {
        logger.entry(filterProperty, filterValue);
        checkFilterPropertyIsOnlyConcept(valueSetUrl, filterProperty, FilterOperator.ISNOTA.toString());

        final var result = new ArrayList<>(concepts);
        for (final var concept : concepts) {
            final var conceptCode = concept.getCode();
            if (conceptCode.isPresent() && conceptCode.get().equals(filterValue)) {
                result.remove(concept);
                result.removeAll(calculateTransitiveChildren(concept));
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    /**
     * Filter out all code system concepts according to the regex filter.
     *
     * @param valueSetUrl    The value set getting processed
     * @param filterProperty The name of the property to filter on. Can be 'code',
     *                       'designation' or any code system property.
     * @param filterValue    The value to filter on. Must be a valid regex.
     * @param concepts       The list of all concepts of the code system.
     * @return A list of concepts matching the filter. Contains all concepts those
     *         codes or property values match the provided regex.
     * @see <a href=
     *      "https://www.hl7.org/fhir/codesystem.html#filters">https://www.hl7.org/fhir/codesystem.html#filters</a>
     * @see <a href=
     *      "https://www.hl7.org/fhir/valueset-filter-operator.html">https://www.hl7.org/fhir/valueset-filter-operator.html</a>
     */
    private ImmutableList<IConceptDefinitionComponentAccessor> calculateRegexFilter(
            String valueSetUrl,
            String filterProperty,
            String filterValue,
            ImmutableList<IConceptDefinitionComponentAccessor> concepts)
            throws GeneratorException {
        logger.entry(filterProperty, filterValue);
        final Pattern regex;
        try {
            regex = Pattern.compile(filterValue);
        } catch (final PatternSyntaxException e) {
            throw logger.throwing(new GeneratorException(
                    String.format(
                            "ValueSet %s: The provided value %s for the regex filter on %s is not a valid regex",
                            valueSetUrl, filterValue, filterProperty),
                    e));
        }

        final var result = new ArrayList<IConceptDefinitionComponentAccessor>();
        for (final var concept : concepts) {
            // Case 1: filter on designation
            if (filterProperty.equals(DESIGNATION_NAME)) {
                // TODO #33 Implement filter on designation
                throw logger.throwing(new GeneratorException(String.format(
                        "ValueSet %s: Regex filter on designations are currently not supported", valueSetUrl)));
                // Case 2: filter on concept code
            } else if (filterProperty.equals(CODE_NAME)) {
                final var conceptCode = concept.getCode();
                if (conceptCode.isPresent() && regex.matcher(conceptCode.get()).matches()) {
                    result.add(concept);
                }
                // Case 3: Filter on property
            } else {
                for (final var property : concept.getProperty()) {
                    if (checkPropertyRegexMatch(property, valueSetUrl, filterProperty, regex)) {
                        result.add(concept);
                    }
                }
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    private boolean checkPropertyRegexMatch(
            IConceptPropertyComponentAccessor property, String valueSetUrl, String filterProperty, Pattern regex)
            throws GeneratorException {
        final var propertyCode = property.getCode();
        final var propertyValue = property.getValue();
        if (propertyValue.isPresent() && propertyValue.get() instanceof final IPrimitiveTypeAccessor primitiveValue) {
            if (propertyCode.isPresent() && propertyCode.get().equals(filterProperty)) {
                if (!primitiveValue.isType(PrimitiveDatatype.STRING)
                        && !primitiveValue.isType(PrimitiveDatatype.INTEGER)) {
                    // TODO #44 Add support for all data types
                    throw logger.throwing(new GeneratorException(String.format(
                            "ValueSet %s: Filter on %s: Currently only StringType and IntegerType "
                                    + "are supported for filter",
                            valueSetUrl, filterProperty)));
                }

                final Optional<StringifiedValue> stringifiedValue = primitiveValue.getStringifiedValue();
                return stringifiedValue.isPresent()
                        && regex.matcher(stringifiedValue.get().getValue()).matches();
            }
        } else {
            throw logger.throwing(new GeneratorException(String.format(
                    "ValueSet %s: Filter on %s: Non-primitive values are not supported", valueSetUrl, filterProperty)));
        }
        return false;
    }

    /**
     * Filter out all code system concepts according to the in filter.
     *
     * @param filterProperty The name of the property to filter on. Can be any code
     *                       system property.
     * @param filterValue    The value to filter on. Comma seperated string list.
     * @param concepts       The list of all concepts of the code system.
     * @return A list of concepts matching the filter. Contains all concepts those
     *         property values are in the provided value list.
     * @see <a href=
     *      "https://www.hl7.org/fhir/codesystem.html#filters">https://www.hl7.org/fhir/codesystem.html#filters</a>
     * @see <a href=
     *      "https://www.hl7.org/fhir/valueset-filter-operator.html">https://www.hl7.org/fhir/valueset-filter-operator.html</a>
     */
    private ImmutableList<IConceptDefinitionComponentAccessor> calculateInFilter(
            String valueSetUrl,
            String filterProperty,
            String filterValue,
            ImmutableList<IConceptDefinitionComponentAccessor> concepts)
            throws GeneratorException {
        logger.entry(filterProperty, filterValue);
        // split with removing all blank around the comma
        final var valueList =
                Arrays.stream(COMMA_BLANK_SPLIT.split(filterValue.trim())).toList();

        final var result = new ArrayList<IConceptDefinitionComponentAccessor>();
        for (final var concept : concepts) {
            for (final var property : concept.getProperty()) {
                if (checkPropertyInValueList(property, valueSetUrl, filterProperty, valueList)) {
                    result.add(concept);
                }
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    /**
     * Filter out all code system concepts according to the not-in filter.
     *
     * @param filterProperty The name of the property to filter on. Can be any code
     *                       system property.
     * @param filterValue    The value to filter on. Comma seperated string list.
     * @param concepts       The list of all concepts of the code system.
     * @return A list of concepts matching the filter. Contains all concepts except
     *         where property values are in the provided value list.
     * @see <a href=
     *      "https://www.hl7.org/fhir/codesystem.html#filters">https://www.hl7.org/fhir/codesystem.html#filters</a>
     * @see <a href=
     *      "https://www.hl7.org/fhir/valueset-filter-operator.html">https://www.hl7.org/fhir/valueset-filter-operator.html</a>
     */
    private ImmutableList<IConceptDefinitionComponentAccessor> calculateNotInFilter(
            String valueSetUrl,
            String filterProperty,
            String filterValue,
            ImmutableList<IConceptDefinitionComponentAccessor> concepts)
            throws GeneratorException {
        logger.entry(filterProperty, filterValue);
        // split with removing all blank around the comma
        final var valueList =
                Arrays.stream(COMMA_BLANK_SPLIT.split(filterValue.trim())).toList();

        final var result = new ArrayList<IConceptDefinitionComponentAccessor>();
        for (final var concept : concepts) {
            for (final var property : concept.getProperty()) {
                if (!checkPropertyInValueList(property, valueSetUrl, filterProperty, valueList)) {
                    result.add(concept);
                }
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    private boolean checkPropertyInValueList(
            IConceptPropertyComponentAccessor property,
            String valueSetUrl,
            String filterProperty,
            List<String> valueList)
            throws GeneratorException {
        final var propertyCode = property.getCode();
        final var propertyValue = property.getValue();
        if (propertyValue.isPresent() && propertyValue.get() instanceof final IPrimitiveTypeAccessor primitiveValue) {
            if (propertyCode.isPresent() && propertyCode.get().equals(filterProperty)) {
                if (!primitiveValue.isType(PrimitiveDatatype.STRING)
                        && !primitiveValue.isType(PrimitiveDatatype.INTEGER)) {
                    // TODO #44 Add support for all data types
                    throw logger.throwing(new GeneratorException(String.format(
                            "ValueSet %s: Filter on %s: Currently only StringType and IntegerType "
                                    + "are supported for filter",
                            valueSetUrl, filterProperty)));
                }

                final Optional<StringifiedValue> stringifiedValue = primitiveValue.getStringifiedValue();
                return stringifiedValue.isPresent()
                        && valueList.contains(stringifiedValue.get().getValue());
            }
        } else {
            throw logger.throwing(new GeneratorException(String.format(
                    "ValueSet %s: Filter on %s: Non-primitive values are not supported", valueSetUrl, filterProperty)));
        }
        return false;
    }

    /**
     * Filter out all code system concepts according to the generalizes filter.
     *
     * @param valueSetUrl    The value set getting processed
     * @param filterProperty The name of the property to filter on. Must be
     *                       'concept'.
     * @param filterValue    The value to filter on. Represents the code of the
     *                       concept.
     * @param concepts       The list of all concepts of the code system.
     * @return A list of concepts matching the filter. The mentioned concept and its
     *         transitive parents.
     * @see <a href=
     *      "https://www.hl7.org/fhir/codesystem.html#filters">https://www.hl7.org/fhir/codesystem.html#filters</a>
     * @see <a href=
     *      "https://www.hl7.org/fhir/valueset-filter-operator.html">https://www.hl7.org/fhir/valueset-filter-operator.html</a>
     */
    private ImmutableList<IConceptDefinitionComponentAccessor> calculateGeneralizesFilter(
            String valueSetUrl,
            String filterProperty,
            String filterValue,
            ImmutableList<IConceptDefinitionComponentAccessor> concepts)
            throws GeneratorException {
        logger.entry(filterProperty, filterValue);
        checkFilterPropertyIsOnlyConcept(valueSetUrl, filterProperty, FilterOperator.GENERALIZES.toString());

        final var result = new ArrayList<IConceptDefinitionComponentAccessor>();
        for (final var concept : concepts) {
            final var conceptCode = concept.getCode();
            if (conceptCode.isPresent() && conceptCode.get().equals(filterValue)) {
                result.add(concept);
                result.addAll(calculateParentsForChild(concepts, concept));
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    /**
     * Filter out all code system concepts according to the exists filter.
     *
     * @param valueSetUrl    The value set getting processed
     * @param filterProperty The name of the property to filter on. Can be any code
     *                       system property.
     * @param filterValue    The value to filter on. true or false.
     * @param concepts       The list of all concepts of the code system.
     * @return A list of concepts matching the filter. Contains all concepts that
     *         have the mentioned property.
     * @see <a href=
     *      "https://www.hl7.org/fhir/codesystem.html#filters">https://www.hl7.org/fhir/codesystem.html#filters</a>
     * @see <a href=
     *      "https://www.hl7.org/fhir/valueset-filter-operator.html">https://www.hl7.org/fhir/valueset-filter-operator.html</a>
     */
    private ImmutableList<IConceptDefinitionComponentAccessor> calculateExistsFilter(
            String valueSetUrl,
            String filterProperty,
            String filterValue,
            ImmutableList<IConceptDefinitionComponentAccessor> concepts)
            throws GeneratorException {
        logger.entry(filterProperty, filterValue);
        final boolean shouldExist;
        if ("true".equalsIgnoreCase(filterValue)) {
            shouldExist = true;
        } else if ("false".equalsIgnoreCase(filterValue)) {
            shouldExist = false;
        } else {
            throw logger.throwing(new GeneratorException(String.format(
                    "ValueSet %s: The provided value for the exists filter on %s must be 'true' or 'false', was '%s'",
                    valueSetUrl, filterProperty, filterValue)));
        }

        final var result = new ArrayList<IConceptDefinitionComponentAccessor>();
        for (final var concept : concepts) {
            for (final var property : concept.getProperty()) {
                if (checkPropertyMatchesExists(property, filterProperty, shouldExist)) {
                    result.add(concept);
                }
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    private boolean checkPropertyMatchesExists(
            IConceptPropertyComponentAccessor property, String filterProperty, boolean shouldExist) {
        final var propertyCode = property.getCode();
        if (shouldExist && propertyCode.isPresent() && propertyCode.get().equals(filterProperty)) {
            return true;
        }
        return !shouldExist && (propertyCode.isEmpty() || (!propertyCode.get().equals(filterProperty)));
    }

    /**
     * Filter out all code system concepts according to the child-of filter.
     *
     * @param valueSetUrl    The value set getting processed
     * @param filterProperty The name of the property to filter on. Must be
     *                       'concept'.
     * @param filterValue    The value to filter on. Represents the code of the
     *                       concept.
     * @param concepts       The list of all concepts of the code system.
     * @return A list of concepts matching the filter. Only the direct children of
     *         the mentioned concept.
     * @see <a href=
     *      "https://www.hl7.org/fhir/codesystem.html#filters">https://www.hl7.org/fhir/codesystem.html#filters</a>
     * @see <a href=
     *      "https://www.hl7.org/fhir/valueset-filter-operator.html">https://www.hl7.org/fhir/valueset-filter-operator.html</a>
     */
    private ImmutableList<IConceptDefinitionComponentAccessor> calculateChildOfFilter(
            String valueSetUrl,
            String filterProperty,
            String filterValue,
            ImmutableList<IConceptDefinitionComponentAccessor> concepts)
            throws GeneratorException {
        logger.entry(filterProperty, filterValue);
        checkFilterPropertyIsOnlyConcept(valueSetUrl, filterProperty, FilterOperator.CHILDOF.toString());

        final var result = new ArrayList<IConceptDefinitionComponentAccessor>();
        for (final var concept : concepts) {
            final var conceptCode = concept.getCode();
            if (conceptCode.isPresent() && conceptCode.get().equals(filterValue)) {
                result.addAll(concept.getChildConcept());
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    /**
     * Filter out all code system concepts according to the descendent-of filter.
     *
     * @param valueSetUrl    The value set getting processed
     * @param filterProperty The name of the property to filter on. Must be
     *                       'concept'.
     * @param filterValue    The value to filter on. Represents the code of the
     *                       concept.
     * @param concepts       The list of all concepts of the code system
     * @return A list of concepts matching the filter. Only the transitive children
     *         of the mentioned concept.
     * @see <a href=
     *      "https://www.hl7.org/fhir/codesystem.html#filters">https://www.hl7.org/fhir/codesystem.html#filters</a>
     * @see <a href=
     *      "https://www.hl7.org/fhir/valueset-filter-operator.html">https://www.hl7.org/fhir/valueset-filter-operator.html</a>
     */
    private ImmutableList<IConceptDefinitionComponentAccessor> calculateDescendentLeafFilter(
            String valueSetUrl,
            String filterProperty,
            String filterValue,
            ImmutableList<IConceptDefinitionComponentAccessor> concepts)
            throws GeneratorException {
        logger.entry(filterProperty);
        checkFilterPropertyIsOnlyConcept(valueSetUrl, filterProperty, FilterOperator.DESCENDENTLEAF.toString());

        final var result = new ArrayList<IConceptDefinitionComponentAccessor>();
        for (final var concept : concepts) {
            final var conceptCode = concept.getCode();
            if (conceptCode.isPresent() && conceptCode.get().equals(filterValue)) {
                result.addAll(calculateLeafChildren(concept));
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    /**
     * Collects all transitive child concepts (in an is-a hierarchy) of the given
     * parent concept.
     *
     * @param parent The parent concepts to collect children of.
     * @return A list of all transitive child concepts excluding the parent.
     */
    private ImmutableList<IConceptDefinitionComponentAccessor> calculateTransitiveChildren(
            IConceptDefinitionComponentAccessor parent) {
        logger.entry(parent);
        final var result = new ArrayList<IConceptDefinitionComponentAccessor>();
        final var children = new ArrayDeque<>(parent.getChildConcept());
        while (!children.isEmpty()) {
            final var current = children.pop();
            result.add(current);
            children.addAll(current.getChildConcept());
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    /**
     * Collects all leaf child concepts (i.e. children that don't have children on
     * their own) of the given parent concept.
     *
     * @param parent The parent concepts to collect children of.
     * @return A list of all leaf child concepts excluding the parent.
     */
    private ImmutableList<IConceptDefinitionComponentAccessor> calculateLeafChildren(
            IConceptDefinitionComponentAccessor parent) {
        logger.entry(parent);
        final var result = new ArrayList<IConceptDefinitionComponentAccessor>();
        final var children = new ArrayDeque<>(parent.getChildConcept());
        while (!children.isEmpty()) {
            final var current = children.pop();
            if (current.getChildConcept().isEmpty()) {
                result.add(current);
            } else {
                children.addAll(current.getChildConcept());
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    /**
     * Collects all transitive parent concepts (in an is-a hierarchy) of the given
     * concept.
     *
     * @param concepts     A list of all concepts in a hierarchy.
     * @param childConcept The concept to collect parents of.
     * @return A list of all transitive parent concepts excluding the child.
     */
    private ImmutableList<IConceptDefinitionComponentAccessor> calculateParentsForChild(
            ImmutableList<IConceptDefinitionComponentAccessor> concepts,
            IConceptDefinitionComponentAccessor childConcept) {
        logger.entry(childConcept);
        final var result = new ArrayList<IConceptDefinitionComponentAccessor>();
        for (final var parentConcept : concepts) {
            if (parentConcept.getChildConcept().contains(childConcept)) {
                result.add(parentConcept);
                result.addAll(calculateParentsForChild(concepts, parentConcept));
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    /**
     * Checks if a filter property is only a concept
     *
     * @param valueSetUrl    The value set getting processed
     * @param filterProperty The filter property that should match concept
     * @param filterName     The name of the filter operator
     * @throws GeneratorException If the filter property doesn't match concept
     */
    private void checkFilterPropertyIsOnlyConcept(String valueSetUrl, String filterProperty, String filterName)
            throws GeneratorException {
        logger.entry(filterProperty);
        if (!filterProperty.equals(CONCEPT_NAME)) {
            throw logger.throwing(new GeneratorException(String.format(
                    "ValueSet %s: %s filter are only allowed for property '%s', but provided '%s'",
                    valueSetUrl, filterName, CONCEPT_NAME, filterProperty)));
        }
        logger.exit();
    }
}
