/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.valueset;

import java.util.Optional;
import lombok.Builder;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.ToString;
import lombok.extern.slf4j.XSlf4j;

/**
 * An enum constant representing a ValueSet value.
 */
@EqualsAndHashCode
@Builder(setterPrefix = "with")
@ToString
@XSlf4j
public final class EnumConstant {

    /**
     * The name of the constant.
     */
    @Getter
    private final String constantName;

    /**
     * The code according to the FHIR definition.
     */
    @Getter
    private final String code;

    /**
     * The coding system according to the FHIR definition.
     */
    @Getter
    private final String system;

    /**
     * The display text according to the FHIR definition.
     */
    @Getter
    @Builder.Default
    private final Optional<String> display = Optional.empty();

    /**
     * Determines whether two enum constants are equal with regards to the identifying portions system and code.
     * This method will issue a warning if system and code are equal, but display or constantName are different.
     * @param other the EnumConstant to compare to
     * @return <code>true</code> if the two enum constants are equal in system and code
     */
    public boolean equalsOnSystemAndCode(EnumConstant other) {
        final var equalOnSystemAndCode = this.system.equals(other.system) && this.code.equals(other.code);
        if (equalOnSystemAndCode
                && (!this.display.equals(other.display) || !this.constantName.equals(other.constantName))) {
            logger.warn(
                    "EnumConstant {} and EnumConstant {} equal in system and code but differ in display or constantName",
                    this,
                    other);
        }
        return equalOnSystemAndCode;
    }
}
