/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.valueset;

import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.IAccessorProvider;
import de.fhlintstone.accessors.model.ICodeSystemAccessor;
import de.fhlintstone.accessors.model.IConceptDefinitionComponentAccessor;
import de.fhlintstone.accessors.model.IConceptReferenceComponentAccessor;
import de.fhlintstone.accessors.model.IConceptSetComponentAccessor;
import de.fhlintstone.accessors.model.IValueSetAccessor;
import de.fhlintstone.fhir.FhirUtilities;
import de.fhlintstone.generator.GeneratorException;
import de.fhlintstone.packages.FhirResourceType;
import de.fhlintstone.packages.IPackageRegistry;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Named;
import lombok.extern.slf4j.XSlf4j;
import org.hl7.fhir.instance.model.api.IBaseResource;

/**
 * Default implementation of {@link IConstantGenerator}.
 */
@Named
@XSlf4j
public class ConstantGenerator implements IConstantGenerator {

    private static final Pattern HYPHEN_DOT_PATTERN = Pattern.compile("[-.]");

    private final IPackageRegistry packageRegistry;
    private final IAccessorProvider accessorProvider;
    private final IFilterCalculator filterCalculator;

    /**
     * Constructor for dependency injection.
     *
     * @param packageRegistry  the {@link IPackageRegistry} to use
     * @param accessorProvider the {@link IAccessorProvider} to use
     * @param filterCalculator the {@link IFilterCalculator} to use
     */
    @Inject
    public ConstantGenerator(
            IPackageRegistry packageRegistry, IAccessorProvider accessorProvider, IFilterCalculator filterCalculator) {
        super();
        this.packageRegistry = packageRegistry;
        this.accessorProvider = accessorProvider;
        this.filterCalculator = filterCalculator;
    }

    @Override
    public ImmutableList<EnumConstant> generateEnumConstants(IValueSetAccessor valueSetAccessor)
            throws GeneratorException {
        logger.entry(valueSetAccessor);
        final var valueSetUrl = valueSetAccessor.getUrl().orElseThrow();
        Set<EnumConstant> result = Set.of();
        // TODO #19 consider using expansions if present (the current use cases
        // don't...)
        final var compose = valueSetAccessor.getCompose();
        // see https://www.hl7.org/fhir/valueset.html#compositions on how to interpret
        // the values

        if (compose.isPresent()) {
            final var includes = compose.get().getInclude();
            if (!includes.isEmpty()) {
                result = generateEnumConstantsForIncludes(valueSetAccessor, includes);
            }

            final var excludes = compose.get().getExclude();
            if (!excludes.isEmpty()) {
                // TODO #18 support exclude statements for value sets
                throw logger.throwing(new GeneratorException(String.format(
                        "The ValueSet %s uses exclusions, which are currently not supported", valueSetUrl)));
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    /**
     * Processes the include statements and generates the corresponding constants.
     *
     * @see <a href=
     *      "https://www.hl7.org/fhir/valueset.html#compositions">https://www.hl7.org/fhir/valueset.html#compositions</a>
     *      for the rules implemented.
     *
     * @param accessor The value set to process
     * @param includes The list of includes of the value set
     * @throws GeneratorException If referenced code system or value sets cannot be
     *                            loaded
     * @return A set of the collected enums
     */
    private Set<EnumConstant> generateEnumConstantsForIncludes(
            IValueSetAccessor accessor, ImmutableList<IConceptSetComponentAccessor> includes)
            throws GeneratorException {
        logger.entry();
        final var valueSetUrl = accessor.getUrl().orElseThrow();
        final var constants = new HashSet<EnumConstant>();

        for (final var include : includes) {
            final Set<EnumConstant> includeConstants;
            final var valueSetConstants = collectReferencedValueSetConstants(valueSetUrl, include);
            final var codeSystemConstants = collectCodeSystemConstants(valueSetUrl, include);

            final var valueSetReferences = include.getValueSet();
            final var codeSystemReference = include.getSystem();
            // If referenced value sets AND a code system are present, an intersection with
            // the code system must be created
            if (!valueSetReferences.isEmpty() && codeSystemReference.isPresent()) {
                includeConstants = calculateConstantIntersections(valueSetConstants, codeSystemConstants);
            } else if (codeSystemReference.isPresent()) {
                includeConstants = codeSystemConstants;
            } else {
                includeConstants = valueSetConstants;
            }
            constants.addAll(includeConstants);
        }
        return logger.exit(constants);
    }

    /**
     * Collect all enum constants in a concept set by referenced value sets
     *
     * @param valueSetUrl        The URI of the value set being processed
     * @param conceptSetAccessor The conceptSet (i.e. include/exclude) for the value
     *                           set
     * @return A list of enum constants, empty if no referenced value sets are
     *         present
     * @throws GeneratorException If at least one referenced value set cannot be
     *                            loaded from the package registry
     */
    private Set<EnumConstant> collectReferencedValueSetConstants(
            String valueSetUrl, IConceptSetComponentAccessor conceptSetAccessor) throws GeneratorException {
        logger.entry(conceptSetAccessor);
        Set<EnumConstant> result = new HashSet<>();

        final var valueSetReferences = conceptSetAccessor.getValueSet();
        if (!valueSetReferences.isEmpty()) {
            var isFirst = true;
            for (final var valueSetReference : valueSetReferences) {
                final var referencedValueSet =
                        loadValueSet(valueSetUrl, valueSetReference.getValue().orElseThrow());
                final var valueSetEnumConstants = new HashSet<>(generateEnumConstants(referencedValueSet));
                // If multiple value sets are present, an intersection must be created
                if (isFirst) {
                    result = valueSetEnumConstants;
                    isFirst = false;
                } else {
                    result = calculateConstantIntersections(result, valueSetEnumConstants);
                }
            }
        }
        return logger.exit(result);
    }

    /**
     * Collect all enum constants in a concept set by the referenced code system.
     * This can either be values defined by filters, concepts, or the whole code
     * system
     *
     * @param valueSetUrl        The URI of the value set being processed
     * @param conceptSetAccessor The conceptSet (i.e. include/exclude) for the value
     *                           set
     * @return A list of enum constants defined by the code system section of the
     *         value set
     * @throws GeneratorException If the referenced code system cannot be loaded
     *                            from the package registry
     */
    private Set<EnumConstant> collectCodeSystemConstants(
            String valueSetUrl, IConceptSetComponentAccessor conceptSetAccessor) throws GeneratorException {
        logger.entry(conceptSetAccessor);
        Set<EnumConstant> result = new HashSet<>();

        final var codeSystemUrl = conceptSetAccessor.getSystem();
        if (codeSystemUrl.isPresent()) {
            final var codeSystem = loadCodeSystem(valueSetUrl, codeSystemUrl.get());

            // Only one filters OR concept can be present
            final var filters = conceptSetAccessor.getFilter();
            if (!filters.isEmpty()) {
                var isFirst = true;
                for (final var filter : filters) {
                    final var filteredConcepts =
                            this.filterCalculator.getMatchingConcepts(valueSetUrl, filter, codeSystem);
                    final var filterEnumConstants =
                            getConstantsFromCodeSystemConcepts(filteredConcepts, codeSystemUrl.get());
                    // If multiple filters are present, an intersection must be created
                    if (isFirst) {
                        result = filterEnumConstants;
                        isFirst = false;
                    } else {
                        result = calculateConstantIntersections(result, filterEnumConstants);
                    }
                }
            } else {
                final var valueSetConcepts = conceptSetAccessor.getConcept();
                // Flatten the code system concepts for enum generation
                // By default a hierarchical code system is only returning the first layer
                final var codeSystemConcepts = FhirUtilities.flattenConceptHierarchy(codeSystem.getConcept());

                if (!valueSetConcepts.isEmpty()) {
                    // concept: Only the enumerated codes are selected
                    result =
                            getConstantsFromValueSetConcepts(valueSetConcepts, codeSystemConcepts, codeSystemUrl.get());
                } else {
                    // no concept or filter: All codes defined by the code system, independent of
                    // code status, are included
                    result = getConstantsFromCodeSystemConcepts(codeSystemConcepts, codeSystemUrl.get());
                }
            }
        }
        return logger.exit(result);
    }

    /**
     * Generate enum constants for all value setconcepts provided
     *
     * @param valueSetConcepts   The list of value set concepts
     * @param codeSystemConcepts The list of related code system concepts. Must belong to the code system of the value set concepts.
     *                           Used for alternative display texts, if they are not provided in the value set concepts.
     *                           The concepts must be flattened.
     * @param codeSystemUrl      The URI of the code system of the value set
     * @return A set of the collected constants
     */
    private Set<EnumConstant> getConstantsFromValueSetConcepts(
            List<IConceptReferenceComponentAccessor> valueSetConcepts,
            List<IConceptDefinitionComponentAccessor> codeSystemConcepts,
            String codeSystemUrl) {
        logger.entry(valueSetConcepts, codeSystemConcepts);
        final Set<EnumConstant> constants = new HashSet<>();

        for (final var valueSetConcept : valueSetConcepts) {
            constants.add(EnumConstant.builder()
                    .withConstantName(
                            convertCodeToConstantName(valueSetConcept.getCode().orElseThrow()))
                    .withCode(valueSetConcept.getCode().orElseThrow())
                    .withDisplay(Optional.of(getDisplayForValueSetConcept(valueSetConcept, codeSystemConcepts)
                            .orElse("(no description available)")))
                    .withSystem(codeSystemUrl)
                    .build());
        }
        return logger.exit(constants);
    }

    /**
     * Gets the display of a value set concept. If the value set concept doesn't hold a display value,
     * the display value of the related code system concept is taken, if available.
     *
     * @param valueSetConcept The value set concept to get the display value for.
     * @param codeSystemConcepts A list of related code system concepts used to get the display value,
     *                           if the value set concept's display value is empty.
     * @return An Optional filled with the display value of the value set concept,
     * if present in the value set concept or the related code system concept, an empty Optional otherwise,
     */
    private Optional<String> getDisplayForValueSetConcept(
            IConceptReferenceComponentAccessor valueSetConcept,
            List<IConceptDefinitionComponentAccessor> codeSystemConcepts) {
        var description = valueSetConcept.getDisplay();
        if (description.isEmpty()) {
            final var relatedCodeSystemConcept =
                    findRelatedCodeSystemConceptForValueSetConcept(valueSetConcept, codeSystemConcepts);

            if (relatedCodeSystemConcept.isPresent()) {
                description = relatedCodeSystemConcept.get().getDisplay();
            }
        }
        return description;
    }

    /**
     * Finds a related code system concept for a given value set concept from a list of provided code system concepts by code.
     * The value set concept and code system concepts must belong to the same code system.
     *
     * @param valueSetConcept The value set concept to find the related code system concept for.
     * @param codeSystemConcepts The list of code system concepts to search through.
     * @return An Optional filled with the related code system concept, if found, an empty Optional otherwise
     */
    private Optional<IConceptDefinitionComponentAccessor> findRelatedCodeSystemConceptForValueSetConcept(
            IConceptReferenceComponentAccessor valueSetConcept,
            List<IConceptDefinitionComponentAccessor> codeSystemConcepts) {
        return codeSystemConcepts.stream()
                .filter(codeSytemConcept -> codeSytemConcept.getCode().isPresent()
                        && codeSytemConcept
                                .getCode()
                                .get()
                                .equals(valueSetConcept.getCode().orElseThrow()))
                .findFirst();
    }

    /**
     * Generate enum constants for all code system concepts provided
     *
     * @param codeSystemConcepts The list of code system concepts. The concepts must
     *                           be flattened.
     * @param codeSystemUrl      The URI of the code system of the value set
     * @return A set of the collected constants
     */
    private Set<EnumConstant> getConstantsFromCodeSystemConcepts(
            List<IConceptDefinitionComponentAccessor> codeSystemConcepts, String codeSystemUrl) {
        logger.entry(codeSystemConcepts);
        final Set<EnumConstant> constants = new HashSet<>();

        for (final var concept : codeSystemConcepts) {
            constants.add(EnumConstant.builder()
                    .withConstantName(
                            convertCodeToConstantName(concept.getCode().orElseThrow()))
                    .withCode(concept.getCode().orElseThrow())
                    .withDisplay(Optional.of(concept.getDisplay().orElse("(no description available)")))
                    .withSystem(codeSystemUrl)
                    .build());
        }
        return logger.exit(constants);
    }

    /**
     * Obtains a code system from the package registry.
     *
     * @param valueSetUrl   The URI of the value set in processing
     * @param codeSystemUrl The URI of the code system to load
     * @return The loaded code system
     * @throws GeneratorException If the code system cannot be loaded
     */
    private ICodeSystemAccessor loadCodeSystem(String valueSetUrl, String codeSystemUrl) throws GeneratorException {
        logger.entry(codeSystemUrl);
        final var codeSystem = loadResource(valueSetUrl, codeSystemUrl, FhirResourceType.CODE_SYSTEM);
        return logger.exit(this.accessorProvider.provideCodeSystemAccessor(codeSystem));
    }

    /**
     * Obtains a value set from the package registry.
     *
     * @param valueSetUrl           The URI of the value set in processing
     * @param referencedValueSetUrl The URI of the referenced value set to load
     * @return The loaded value set
     * @throws GeneratorException If the referenced value set cannot be loaded
     */
    private IValueSetAccessor loadValueSet(String valueSetUrl, String referencedValueSetUrl) throws GeneratorException {
        logger.entry(referencedValueSetUrl);
        final var valueSet = loadResource(valueSetUrl, referencedValueSetUrl, FhirResourceType.VALUE_SET);
        return logger.exit(this.accessorProvider.provideValueSetAccessor(valueSet));
    }

    /**
     * Loads a generic resource from the package registry.
     *
     * @param valueSetUrl  The URI of the value set in processing
     * @param resourceUrl  The URI of the resource to load
     * @param resourceType The type of the resource to load
     * @return The loaded resource
     * @throws GeneratorException If the resource cannot be loaded
     */
    private IBaseResource loadResource(String valueSetUrl, String resourceUrl, FhirResourceType resourceType)
            throws GeneratorException {
        Optional<IBaseResource> resource;
        try {
            resource = this.packageRegistry.getUniqueResource(resourceType, new URI(resourceUrl));
            if (resource.isEmpty()) {
                throw logger.throwing(new GeneratorException(String.format(
                        "The ValueSet %s references the %s %s which is not available",
                        valueSetUrl, resourceType, resourceUrl)));
            }
        } catch (final URISyntaxException e) {
            throw logger.throwing(new GeneratorException(
                    String.format(
                            "The ValueSet %s uses the invalid %s reference %s", valueSetUrl, resourceType, resourceUrl),
                    e));
        }
        return logger.exit(resource.orElseThrow());
    }

    /**
     * Determines the constant name for the value.
     *
     * @param code
     * @return
     */
    private String convertCodeToConstantName(String code) {
        // TODO #9 add option to generate constant from display text instead
        // (see e.g. MaritalStatus / http://hl7.org/fhir/ValueSet/marital-status)
        logger.entry(code);
        var constantName = HYPHEN_DOT_PATTERN.matcher(code).replaceAll("_").toUpperCase();
        final char c = constantName.charAt(0);
        if (c >= '0' && c <= '9') {
            constantName = "_" + constantName;
        }
        return logger.exit(constantName);
    }

    private Set<EnumConstant> calculateConstantIntersections(Set<EnumConstant> set1, Set<EnumConstant> set2) {
        return set1.stream()
                .filter(constant1 -> set2.stream().anyMatch(constant2 -> constant2.equalsOnSystemAndCode(constant1)))
                .collect(Collectors.toSet());
    }
}
