/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.valueset;

import com.palantir.javapoet.AnnotationSpec;
import com.palantir.javapoet.CodeBlock;
import com.palantir.javapoet.JavaFile;
import com.palantir.javapoet.MethodSpec;
import com.palantir.javapoet.TypeSpec;
import com.palantir.javapoet.TypeSpec.Builder;
import de.fhlintstone.generator.GeneratorException;
import jakarta.annotation.Generated;
import java.io.File;
import java.io.IOException;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Optional;
import javax.inject.Named;
import javax.lang.model.element.Modifier;
import lombok.extern.slf4j.XSlf4j;
import org.hl7.fhir.exceptions.FHIRException;

/**
 * Default implementation of {@link ICodeEmitter}.
 */
@Named
@XSlf4j
@SuppressWarnings("java:S1192") // String literals should not be duplicated
public class CodeEmitter implements ICodeEmitter {

    /** Create a new instance of CodeEmitter */
    public CodeEmitter() {
        // Default constructor with generated JavaDoc
    }

    @Override
    public void generate(EnumData enumData) throws GeneratorException {
        final var qualifiedName = enumData.getClassName();
        logger.debug(
                "Generating code of enum {} for ValueSet {}",
                qualifiedName.canonicalName(),
                enumData.getValueSetName());
        final var sortedConstants = enumData.getEnumConstants().stream()
                .sorted((c1, c2) -> c1.getConstantName().compareTo(c2.getConstantName()))
                .toList();
        final var enumBuilder = initializeBuilder(enumData);
        addConstants(enumBuilder, sortedConstants);
        generateMethodFromCode(enumBuilder, enumData, sortedConstants);
        generateMethodToCode(enumBuilder, sortedConstants);
        generateMethodFromCoding(enumBuilder, enumData, sortedConstants);
        generateMethodToCoding(enumBuilder, enumData, sortedConstants);
        generateMethodGetSystem(enumBuilder, sortedConstants);
        generateMethodGetDisplay(enumBuilder, sortedConstants);
        writeEnumToFile(enumBuilder, enumData);
        logger.exit();
    }

    /**
     * Initializes the enum builder.
     */
    private TypeSpec.Builder initializeBuilder(EnumData enumData) {
        logger.entry(enumData);
        final var enumBuilder = TypeSpec.enumBuilder(enumData.getClassName())
                .addModifiers(Modifier.PUBLIC)
                .addAnnotation(AnnotationSpec.builder(Generated.class)
                        .addMember("value", "$S", "fhlintstone")
                        .addMember("date", "$S", ZonedDateTime.now().format(DateTimeFormatter.ISO_INSTANT))
                        .build())
                .addJavadoc(
                        "Representation of the HL7 FHIR ValueSet $L.\n($L)\n",
                        enumData.getValueSetName(),
                        enumData.getValueSetURL());
        final Optional<String> valueSetDescription = enumData.getDescription();
        if (valueSetDescription.isPresent()) {
            enumBuilder.addJavadoc(valueSetDescription.get());
        }
        return logger.exit(enumBuilder);
    }

    /**
     * Generates the enum constants.
     * @param sortedConstants
     */
    private void addConstants(Builder enumBuilder, List<EnumConstant> sortedConstants) {
        logger.entry();
        for (final var constant : sortedConstants) {
            enumBuilder.addEnumConstant(
                    constant.getConstantName(),
                    TypeSpec.anonymousClassBuilder(CodeBlock.of(""))
                            .addJavadoc(
                                    "$L ($L)",
                                    constant.getCode(),
                                    constant.getDisplay().orElse("?"))
                            .build());
        }
        logger.exit();
    }

    /**
     * Generates the static method fromCode.
     * @param sortedConstants
     */
    private void generateMethodFromCode(Builder enumBuilder, EnumData enumData, List<EnumConstant> sortedConstants) {
        logger.entry();
        final var methodBuilder = MethodSpec.methodBuilder("fromCode")
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                .returns(enumData.getClassName())
                .addParameter(String.class, "codeString")
                .addException(FHIRException.class)
                .addJavadoc(
                        """
                        Attempts to determine the {@link $1T} value from the code provided.
                        @param codeString the code to interpret
                        @return the {@link $1T} value if the code could be interpreted
                        @throws FHIRException if the code does not match any of the known values
                        """,
                        enumData.getClassName())
                .beginControlFlow("if (codeString == null || \"\".equals(codeString))")
                .addStatement("return null")
                .endControlFlow();
        for (final var constant : sortedConstants) {
            methodBuilder
                    .beginControlFlow("if ($S.equals(codeString))", constant.getCode())
                    .addStatement("return $L", constant.getConstantName())
                    .endControlFlow();
        }
        methodBuilder.addStatement(
                "throw new $T(String.format($S, codeString))",
                FHIRException.class,
                String.format("Unknown %s code %%s", enumData.getValueSetName()));
        enumBuilder.addMethod(methodBuilder.build());
        logger.exit();
    }

    /**
     * Generates the method toCode.
     *
     * @param enumBuilder
     * @param enumConstants
     */
    private void generateMethodToCode(Builder enumBuilder, List<EnumConstant> sortedConstants) {
        logger.entry();
        final var methodBuilder = MethodSpec.methodBuilder("toCode")
                .addModifiers(Modifier.PUBLIC)
                .returns(String.class)
                .addJavadoc(
                        """
                        Returns the code associated with the enum value.
                        @return the code associated with the enum value
                        """)
                .beginControlFlow("switch(this)");
        for (final var constant : sortedConstants) {
            methodBuilder.addStatement("case $L: return $S", constant.getConstantName(), constant.getCode());
        }
        methodBuilder.addStatement("default: return \"?\"");
        methodBuilder.endControlFlow();
        enumBuilder.addMethod(methodBuilder.build());
        logger.exit();
    }

    /**
     * Generates the static method fromCoding.
     *
     * @param enumBuilder
     * @param accessor
     * @param enumConstants
     * @param qualifiedName
     */
    private void generateMethodFromCoding(Builder enumBuilder, EnumData enumData, List<EnumConstant> sortedConstants) {
        logger.entry();
        final var methodBuilder = MethodSpec.methodBuilder("fromCoding")
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                .returns(enumData.getClassName())
                .addParameter(enumData.getCodingType(), "coding")
                .addException(FHIRException.class)
                .addJavadoc(
                        """
                        Attempts to determine the {@link $1T} value from the Coding instance provided.
                        @param coding the Coding to interpret
                        @return the {@link $1T} value if the Coding could be interpreted
                        @throws FHIRException if the Coding system or code does not match any of the known values
                        """,
                        enumData.getClassName());
        for (final var constant : sortedConstants) {
            methodBuilder
                    .beginControlFlow("if (coding.is($S, $S))", constant.getSystem(), constant.getCode())
                    .addStatement("return $L", constant.getConstantName())
                    .endControlFlow();
        }
        methodBuilder.addStatement(
                "throw new $T(String.format($S, coding))",
                FHIRException.class,
                String.format("Unknown %s coding %%s", enumData.getValueSetName()));
        enumBuilder.addMethod(methodBuilder.build());
        logger.exit();
    }

    /**
     * Generates the method toCoding.
     *
     * @param enumBuilder
     * @param accessor
     * @param enumConstants
     */
    private void generateMethodToCoding(Builder enumBuilder, EnumData enumData, List<EnumConstant> sortedConstants) {
        logger.entry();
        final var methodBuilder = MethodSpec.methodBuilder("toCoding")
                .addModifiers(Modifier.PUBLIC)
                .returns(enumData.getCodingType())
                .addJavadoc(
                        """
                        Returns a Coding instance representing the enum value.
                        @return a Coding instance representing the enum value
                        """)
                .beginControlFlow("switch(this)");
        for (final var constant : sortedConstants) {
            methodBuilder.addStatement(
                    "case $L: return new $T($S, $S, $S)",
                    constant.getConstantName(),
                    enumData.getCodingType(),
                    constant.getSystem(),
                    constant.getCode(),
                    constant.getDisplay().orElse("?"));
        }
        methodBuilder.addStatement("default:throw new $T(\"Unexpected enum value\")", FHIRException.class);
        methodBuilder.endControlFlow();
        enumBuilder.addMethod(methodBuilder.build());
        logger.exit();
    }

    /**
     * Generates the method getSystem.
     *
     * @param enumBuilder
     * @param enumConstants
     */
    private void generateMethodGetSystem(Builder enumBuilder, List<EnumConstant> sortedConstants) {
        logger.entry();
        final var methodBuilder = MethodSpec.methodBuilder("getSystem")
                .addModifiers(Modifier.PUBLIC)
                .returns(String.class)
                .addJavadoc(
                        """
                        Returns the code system associated with the enum value.
                        @return the code system associated with the enum value
                        """)
                .beginControlFlow("switch(this)");
        for (final var constant : sortedConstants) {
            methodBuilder.addStatement("case $L: return $S", constant.getConstantName(), constant.getSystem());
        }
        methodBuilder.addStatement("default: return \"?\"");
        methodBuilder.endControlFlow();
        enumBuilder.addMethod(methodBuilder.build());
        logger.exit();
    }

    /**
     * Generates the method getDisplay.
     *
     * @param enumBuilder
     * @param enumConstants
     */
    private void generateMethodGetDisplay(Builder enumBuilder, List<EnumConstant> sortedConstants) {
        logger.entry();
        final var methodBuilder = MethodSpec.methodBuilder("getDisplay")
                .addModifiers(Modifier.PUBLIC)
                .returns(String.class)
                .addJavadoc(
                        """
                        Returns the display text associated with the enum value.
                        @return the display text associated with the enum value
                        """)
                .beginControlFlow("switch(this)");
        for (final var constant : sortedConstants) {
            methodBuilder.addStatement(
                    "case $L: return $S",
                    constant.getConstantName(),
                    constant.getDisplay().orElse("?"));
        }
        methodBuilder.addStatement("default: return \"?\"");
        methodBuilder.endControlFlow();
        enumBuilder.addMethod(methodBuilder.build());
        logger.exit();
    }

    /**
     * Writes the generated code to the output file.
     *
     * @param enumBuilder
     * @param className
     * @param outputPath
     * @throws GeneratorException
     */
    private void writeEnumToFile(TypeSpec.Builder enumBuilder, EnumData enumData) throws GeneratorException {
        logger.entry();
        final var enumSpec = enumBuilder.build();
        final JavaFile javaFile = JavaFile.builder(enumData.getClassName().packageName(), enumSpec)
                .skipJavaLangImports(true)
                .indent("\t")
                .build();
        try {
            javaFile.writeTo(new File(enumData.getOutputPath()));
        } catch (final IOException e) {
            throw logger.throwing(new GeneratorException(
                    String.format(
                            "Unable to write generated implementation %s to output path %s",
                            enumData.getClassName().canonicalName(), enumData.getOutputPath()),
                    e));
        }
        logger.exit();
    }
}
