/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.intermediate;

import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.accessors.implementations.IFrameworkTypeLocator;
import de.fhlintstone.accessors.implementations.IMappedType;
import de.fhlintstone.accessors.implementations.ITypeSpecification;
import de.fhlintstone.accessors.implementations.ITypeSpecification.TypeReference;
import de.fhlintstone.accessors.implementations.MappedType;
import de.fhlintstone.accessors.implementations.TypeSpecification;
import de.fhlintstone.fhir.ClassAnnotation;
import de.fhlintstone.fhir.FhirException;
import de.fhlintstone.fhir.IStructureDefinitionIntrospector;
import de.fhlintstone.generator.GeneratorException;
import de.fhlintstone.generator.IGeneratedTypeNameRegistry;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import javax.inject.Inject;
import javax.inject.Named;
import lombok.extern.slf4j.XSlf4j;

/**
 * Default implementation of {@link ITypeMapper}.
 */
@Named
@XSlf4j
public class TypeMapper implements ITypeMapper {

    private final IFrameworkTypeLocator frameworkTypeLocator;
    private final IGeneratedTypeNameRegistry generatedTypeNameRegistry;
    private final ITypeInformationStore typeInformationStore;
    private final IStructureDefinitionIntrospector structureDefinitionIntrospector;

    /**
     * Constructor for dependency injection.
     *
     * @param frameworkTypeLocator            the {@link IFrameworkTypeLocator} to use
     * @param generatedTypeNameRegistry       the {@link IGeneratedTypeNameRegistry} to use
     * @param typeInformationStore            the {@link ITypeInformationStore} to use
     * @param structureDefinitionIntrospector the {@link IStructureDefinitionIntrospector} to use
     */
    @Inject
    public TypeMapper(
            IFrameworkTypeLocator frameworkTypeLocator,
            IGeneratedTypeNameRegistry generatedTypeNameRegistry,
            ITypeInformationStore typeInformationStore,
            IStructureDefinitionIntrospector structureDefinitionIntrospector) {
        super();
        this.frameworkTypeLocator = frameworkTypeLocator;
        this.generatedTypeNameRegistry = generatedTypeNameRegistry;
        this.typeInformationStore = typeInformationStore;
        this.structureDefinitionIntrospector = structureDefinitionIntrospector;
    }

    @Override
    public IMappedType determineSuperclass(ITypeInformation typeInformation) throws GeneratorException {
        logger.entry(typeInformation);

        // This method is used by the TypeInformationConverter to determine the superclass of a top-level class to be
        // generated. The superclass might be a generated type (if so, then it is configured indirectly by adding a
        // configuration entry for the baseDefinition canonical) or a framework type.

        IMappedType result = null;

        // check whether a top-level type is configured for the type
        final var baseDefinition = typeInformation.getBaseDefinition();
        final var generatedType = this.generatedTypeNameRegistry.getClassName(baseDefinition);
        if (generatedType.isPresent()) {
            logger.debug(
                    "Generated class {} is configured as superclass of type {}",
                    generatedType.getClass(),
                    typeInformation);
            result = MappedType.builder()
                    .withTypeCode(baseDefinition.toString())
                    .withTypeCodeURI(baseDefinition)
                    .withType(generatedType.get())
                    .build();
        } else {
            // no configured type present - check for framework types
            final var classAnnotation = typeInformation.getClassAnnotation();
            if (classAnnotation.isPresent() && classAnnotation.get() == ClassAnnotation.BLOCK) {
                // for @Block classes, we always have to use BackboneElement
                logger.debug(
                        "Framework class BackboneElement is used as superclass of @Block type {}", typeInformation);
                result = MappedType.builder()
                        .withTypeCode(baseDefinition.toString())
                        .withTypeCodeURI(baseDefinition)
                        .withType(this.frameworkTypeLocator.getBackboneElementType())
                        .build();
            } else {
                // for other classes, use the baseDefinition canonical to look up the framework type
                result = mapFrameworkType(baseDefinition);
                logger.debug("Superclass of type {} is {}", typeInformation, result);
            }
        }
        return logger.exit(result);
    }

    /**
     * Tries to determine a framework type for a canonical URI. This is essentially a wrapper around
     * {@link IFrameworkTypeLocator#determineFrameworkTypes(ITypeSpecification)} that wraps the result into an
     * {@link IMappedType} instance.
     *
     * @param canonicalURI the canonical URI to search a framework type for
     * @return an {@link IMappedType} object that refers to the framework type found.
     * @throws GeneratorException if the canonical URI could not be resolved or is ambiguous
     */
    private IMappedType mapFrameworkType(final URI canonicalURI) throws GeneratorException {
        final var typeSpecification = TypeSpecification.builder()
                .withType(canonicalURI.toString(), canonicalURI)
                .build();
        final var frameworkTypes = this.frameworkTypeLocator.determineFrameworkTypes(typeSpecification);
        switch (frameworkTypes.size()) {
            case 0:
                throw logger.throwing(new GeneratorException(String.format(
                        "The canonical URI %s is neither a framework type nor configured for class generation",
                        canonicalURI)));
            case 1:
                return logger.exit(frameworkTypes.iterator().next());
            default:
                throw logger.throwing(new GeneratorException(String.format(
                        "The canonical URI %s cannot be resolved to a type name unambiguously", canonicalURI)));
        }
    }

    @Override
    public ImmutableCollection<IMappedType> determineAttributeTypes(
            ITypeInformation typeInformation, ITypeAttribute sourceAttribute) {
        logger.entry(sourceAttribute);

        // This method is used by the TypeInformationConverter to determine the type or types used to represent an
        // attribute. The result may contain both generated and framework types.

        // First check whether a nested type is available to represent the attribute - if so, that type takes
        // precedence over all types.
        var result = mapNestedClassName(typeInformation, sourceAttribute);

        if (result.isEmpty()) {
            // No nested type - check whether a top-level type is configured for the type - if so, that type again takes
            // precedence over possible framework types-
            result = mapTopLevelClassName(sourceAttribute.getTypes());
        }

        if (result.isEmpty()) {
            // No configured type at all - check for framework types and use these.
            result.addAll(this.frameworkTypeLocator.determineFrameworkTypes(sourceAttribute.getTypes()));
        }

        return logger.exit(ImmutableList.copyOf(result));
    }

    /**
     * Tries to determine a configured nested class name for the attribute. The nested classes are configured for an
     * element of the surrounding StructureDefinition.
     *
     * @param typeInformation information about the surrounding type
     * @param sourceAttribute information about the attribute that type information is required for
     * @return a list of the types that were determined to, or an empty list
     */
    private List<IMappedType> mapNestedClassName(ITypeInformation typeInformation, ITypeAttribute sourceAttribute) {
        logger.entry(sourceAttribute);
        final var result = new ArrayList<IMappedType>();
        final String elementID = sourceAttribute.getElementId();
        final var nestedType =
                this.generatedTypeNameRegistry.getNestedClassName(typeInformation.getCanonicalURI(), elementID);
        if (nestedType.isPresent()) {
            logger.debug(
                    "Nested type {} is used to represent element {} of type {} as configured",
                    nestedType.get().reflectionName(),
                    elementID,
                    typeInformation);
            for (final var typeSpecification : sourceAttribute.getTypes()) {
                final TypeReference typeCode = typeSpecification.getTypeCode();

                var typeMapped = false;
                // determine if there are profiles to be considered
                for (final var profile : typeSpecification.getProfiles()) {
                    result.add(MappedType.builder()
                            .withTypeCode(typeCode.code())
                            .withTypeCodeURI(typeCode.canonical())
                            .withProfile(Optional.of(profile.code()))
                            .withProfileURI(Optional.of(profile.canonical()))
                            .withType(nestedType.get())
                            .build());
                    typeMapped = true;
                }
                if (!typeMapped) {
                    // no profiles were specified --> map to the type code
                    result.add(MappedType.builder()
                            .withTypeCode(typeCode.code())
                            .withTypeCodeURI(typeCode.canonical())
                            .withType(nestedType.get())
                            .build());
                }
            }
        } else {
            logger.debug("No nested type is configured for element {} of type {}", elementID, typeInformation);
        }
        return logger.exit(result);
    }

    /**
     * Tries to determine a configured top-level class for set of type specifications.
     *
     * @param typeSpecifications a list of {@link ITypeSpecification}s describing a type
     * @return a list of the types that were determined to, or an empty list
     */
    private List<IMappedType> mapTopLevelClassName(ImmutableCollection<ITypeSpecification> typeSpecifications) {
        logger.entry(typeSpecifications);
        final var result = new ArrayList<IMappedType>();
        for (final var typeSpecification : typeSpecifications) {
            final TypeReference typeCode = typeSpecification.getTypeCode();
            var typeMapped = false;
            // determine if there are profiles to be considered
            for (final var profile : typeSpecification.getProfiles()) {
                final var profileType = this.generatedTypeNameRegistry.getClassName(profile.canonical());
                if (profileType.isPresent()) {
                    logger.debug("Framework type {} can be used to represent profile {}", profileType.get(), profile);
                    result.add(MappedType.builder()
                            .withTypeCode(typeCode.code())
                            .withTypeCodeURI(typeCode.canonical())
                            .withProfile(Optional.of(profile.code()))
                            .withProfileURI(Optional.of(profile.canonical()))
                            .withType(profileType.get())
                            .build());
                    typeMapped = true;
                }
            }
            if (!typeMapped) {
                // no types were mapped using the profiles (or no profiles were specified) --> try the type code
                final var typeCodeType = this.generatedTypeNameRegistry.getClassName(typeCode.canonical());
                if (typeCodeType.isPresent()) {
                    logger.debug(
                            "Framework type {} can be used to represent type code {}", typeCodeType.get(), typeCode);
                    result.add(MappedType.builder()
                            .withTypeCode(typeCode.code())
                            .withTypeCodeURI(typeCode.canonical())
                            .withType(typeCodeType.get())
                            .build());
                }
            }
        }
        return logger.exit(result);
    }

    @Override
    public IMappedType determineFrameworkSuperclass(ITypeInformation typeInformation) throws GeneratorException {
        logger.entry(typeInformation);

        TypeName frameworkType = null;

        var currentTypeInformation = typeInformation;

        // If this is a nested type, it has a configured supertype
        if (currentTypeInformation.getNestedConfiguration().isPresent()) {
            final var nestedConfiguration =
                    currentTypeInformation.getNestedConfiguration().get();
            final TypeName nestedSuperclass = nestedConfiguration
                    .getSuperTypeName()
                    .orElseThrow(() -> new GeneratorException(String.format(
                            "No or invalid supertype configured for nested element %s",
                            nestedConfiguration.getElementId())));
            logger.debug("Nested class for type {} has configured superclass {}", typeInformation, nestedSuperclass);
            if (this.frameworkTypeLocator.isFrameworkType(nestedSuperclass)) {
                frameworkType = nestedSuperclass;
            } else {
                currentTypeInformation = this.typeInformationStore
                        .getByType(nestedSuperclass)
                        .orElseThrow(() -> new GeneratorException(String.format(
                                "No type information available for nested element %s (class %s)",
                                nestedConfiguration.getElementId(), nestedSuperclass)));
            }
        }

        if (frameworkType == null) {
            // if still necessary, follow the chain of base types until we find a generated type
            while (currentTypeInformation.isGenerated()) {
                final var canonicalURI = currentTypeInformation.getCanonicalURI();
                logger.debug(
                        "Following type {} to base type {}", canonicalURI, currentTypeInformation.getBaseDefinition());
                currentTypeInformation = currentTypeInformation
                        .getBaseTypeInformation()
                        .orElseThrow(() -> new GeneratorException(
                                String.format("Base type information of type %s is missing", canonicalURI)));
            }
        }
        // We need a single type specification of the generated type at this point, and we don't expect it to
        // have multiple profile specifications.
        final var typeSpecifications = currentTypeInformation.getTypeSpecifications();
        if (typeSpecifications.size() != 1) {
            throw new GeneratorException(String.format(
                    "Base type %s has zero or multiple type specifications (%s)",
                    currentTypeInformation.getCanonicalURI(), typeSpecifications));
        }
        final var typeSpecification = typeSpecifications.iterator().next();
        if (typeSpecification.hasProfiles() && typeSpecification.getProfiles().size() > 1) {
            throw new GeneratorException(String.format(
                    "Base type %s has multiple profile specifications (%s)",
                    currentTypeInformation.getCanonicalURI(), typeSpecification.getProfiles()));
        }

        // if necessary, determine the type via the canonical URI
        if (frameworkType == null) {
            final var canonicalURI = currentTypeInformation.getCanonicalURI();
            frameworkType = currentTypeInformation
                    .getFrameworkType()
                    .orElseThrow(() -> new GeneratorException(
                            String.format("Framework type for non-generated type %s is missing", canonicalURI)));
        }

        // instead of Extension, we have to use BackboneElement as superclass
        if (frameworkType.equals(this.frameworkTypeLocator.getExtensionType())) {
            frameworkType = this.frameworkTypeLocator.getBackboneElementType();
        }

        logger.debug("Framework superclass of type {} is {}", typeInformation, frameworkType);
        final var result = MappedType.forTypeSpecification(typeSpecification, frameworkType);
        return logger.exit(result.iterator().next());
    }

    @Override
    public ImmutableCollection<IMappedType> determineFrameworkSuperclasses(
            ITypeInformation typeInformation, ITypeAttribute sourceAttribute) throws GeneratorException {
        logger.entry(typeInformation, sourceAttribute);

        final var elementID = sourceAttribute.getElementId();
        final var canonicalURI = typeInformation.getCanonicalURI();
        logger.debug("Searching for framework superclass of element {} of type {}", elementID, canonicalURI);

        final var result = new ArrayList<IMappedType>();

        // check for a nested type for this specific attribute first
        final var nestedTypeInformation = typeInformation.getNestedTypes().stream()
                .filter(t -> t.getElementId().orElse("").equals(elementID))
                .findAny();
        if (nestedTypeInformation.isPresent()) {
            logger.debug(
                    "Attribute {} of StructureDefinition {} uses nested type {}",
                    elementID,
                    canonicalURI,
                    nestedTypeInformation.get());
            result.add(determineFrameworkSuperclass(nestedTypeInformation.get()));
        } else {
            // no configured nested types found - use the type codes to search for framework classes
            logger.debug("No nested type found for element {} of StructureDefinition {}", elementID, canonicalURI);
            try {
                result.addAll(
                        this.structureDefinitionIntrospector.getFrameworkSuperclasses(sourceAttribute.getTypes()));
            } catch (final FhirException e) {
                throw new GeneratorException(e.getMessage(), e);
            }
        }

        logger.debug("Framework superclasses of attribute {} of type {} are {}", elementID, typeInformation, result);
        return logger.exit(ImmutableList.copyOf(result));
    }

    @Override
    public boolean isDerivedFromPrimitiveType(ITypeInformation typeInformation) throws GeneratorException {
        logger.entry(typeInformation);
        final var frameworkSupertype = determineFrameworkSuperclass(typeInformation);
        final var result = this.frameworkTypeLocator.isDerivedFromPrimitiveType(frameworkSupertype.getType());
        return logger.exit(result);
    }

    @Override
    public boolean isDerivedFromPrimitiveType(ITypeInformation typeInformation, ITypeAttribute attributeInformation)
            throws GeneratorException {
        logger.entry(attributeInformation);
        final var frameworkSupertypes = determineFrameworkSuperclasses(typeInformation, attributeInformation);
        switch (frameworkSupertypes.size()) {
            case 0:
                // no supertypes should never happen, but we treat it like a non-primitive descendant
                return logger.exit(false);
            case 1:
                final var frameworkSupertype = frameworkSupertypes.iterator().next();
                final var result = this.frameworkTypeLocator.isDerivedFromPrimitiveType(frameworkSupertype.getType());
                return logger.exit(result);
            default:
                // multiple supertypes will be handled using a generic Type, so not a descendant of PrimitiveType
                return logger.exit(false);
        }
    }
}
