/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.intermediate;

import com.palantir.javapoet.TypeName;
import de.fhlintstone.generator.IGeneratedTypeNameRegistry;
import java.net.URI;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;
import lombok.extern.slf4j.XSlf4j;

/**
 * Default implementation of {@link ITypeInformationStore}.
 */
@Named
@Singleton
@XSlf4j
public class TypeInformationStore implements ITypeInformationStore {

    private final IGeneratedTypeNameRegistry generatedTypeNameRegistry;

    private final Map<URI, ITypeInformation> canonicalIndex = new HashMap<>();
    private final Map<TypeName, ITypeInformation> typeNameIndex = new HashMap<>();

    /**
     * Constructor for dependency injection.
     * @param generatedTypeNameRegistry the {@link IGeneratedTypeNameRegistry} to use
     */
    @Inject
    public TypeInformationStore(IGeneratedTypeNameRegistry generatedTypeNameRegistry) {
        this.generatedTypeNameRegistry = generatedTypeNameRegistry;
    }

    @Override
    public void clear() {
        logger.entry();
        this.canonicalIndex.clear();
        this.typeNameIndex.clear();
        logger.exit();
    }

    @Override
    public void add(ITypeInformation typeInformation) {
        logger.entry(typeInformation);
        checkAndAdd(typeInformation);
        for (final var nestedType : typeInformation.getNestedTypes()) {
            checkAndAdd(nestedType);
        }
        logger.exit();
    }

    private void checkAndAdd(ITypeInformation typeInformation) {
        logger.entry(typeInformation);

        // only record the type via canonical URI if it is not a nested type
        if (typeInformation.getNestedConfiguration().isEmpty()) {
            final var canonicalURI = typeInformation.getCanonicalURI();
            logger.debug("Recorded canonical URI is {}", canonicalURI);
            if (this.canonicalIndex.containsKey(canonicalURI)) {
                throw new IllegalArgumentException(String.format(
                        "A type information object with the canonical URI %s has already been stored.", canonicalURI));
            } else {
                this.canonicalIndex.put(canonicalURI, typeInformation);
            }
        }

        // always record the type name
        final var typeName = getTypeName(typeInformation);
        logger.debug("Recorded type name is {}", typeName);
        if (typeName.isPresent()) {
            if (this.typeNameIndex.containsKey(typeName.get())) {
                throw new IllegalArgumentException(String.format(
                        "A type information object with the type name %s has already been stored.", typeName.get()));
            } else {
                this.typeNameIndex.put(typeName.get(), typeInformation);
            }
        }
        logger.exit();
    }

    private Optional<TypeName> getTypeName(ITypeInformation typeInformation) {
        logger.entry(typeInformation);
        final var frameworkType = typeInformation.getFrameworkType();
        if (frameworkType.isPresent()) {
            return logger.exit(frameworkType);
        } else {
            if (typeInformation.getNestedConfiguration().isPresent()) {
                return logger.exit(this.generatedTypeNameRegistry
                        .getNestedClassName(
                                typeInformation.getParentCanonicalURI().orElseThrow(),
                                typeInformation.getElementId().orElseThrow())
                        .map(TypeName.class::cast));
            } else {
                return logger.exit(this.generatedTypeNameRegistry
                        .getClassName(typeInformation.getCanonicalURI())
                        .map(TypeName.class::cast));
            }
        }
    }

    @Override
    public Optional<ITypeInformation> getByCanonical(URI canonicalURI) {
        logger.entry(canonicalURI);
        final Optional<ITypeInformation> result = this.canonicalIndex.containsKey(canonicalURI)
                ? Optional.of(this.canonicalIndex.get(canonicalURI))
                : Optional.empty();
        return logger.exit(result);
    }

    @Override
    public Optional<ITypeInformation> getByType(TypeName typeName) {
        logger.entry(typeName);
        final Optional<ITypeInformation> result = this.typeNameIndex.containsKey(typeName)
                ? Optional.of(this.typeNameIndex.get(typeName))
                : Optional.empty();
        return logger.exit(result);
    }
}
