/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.intermediate;

import ca.uhn.fhir.model.api.annotation.Binding;
import ca.uhn.fhir.model.api.annotation.Child;
import ca.uhn.fhir.model.api.annotation.Description;
import ca.uhn.fhir.model.api.annotation.Extension;
import com.google.common.collect.ImmutableCollection;
import com.palantir.javapoet.ClassName;
import com.palantir.javapoet.ParameterizedTypeName;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.accessors.implementations.IFrameworkTypeLocator;
import de.fhlintstone.accessors.implementations.IMappedType;
import de.fhlintstone.accessors.implementations.ITypeSpecification;
import de.fhlintstone.generator.GeneratorException;
import de.fhlintstone.generator.IGeneratedTypeNameRegistry;
import de.fhlintstone.generator.structuredefinition.code.AccessorGenerationMode;
import de.fhlintstone.generator.structuredefinition.code.ClassAttribute;
import de.fhlintstone.generator.structuredefinition.code.ClassAttributeAnnotation;
import de.fhlintstone.generator.structuredefinition.code.ClassConstructor;
import de.fhlintstone.generator.structuredefinition.code.ClassData;
import de.fhlintstone.generator.structuredefinition.code.FixedValueRule;
import de.fhlintstone.generator.structuredefinition.code.IClassAttributeAnnotation;
import de.fhlintstone.generator.structuredefinition.code.IClassConstructor.ConstructorParameter;
import de.fhlintstone.generator.structuredefinition.code.IFixedValueRule;
import de.fhlintstone.generator.structuredefinition.code.ITopLevelClass;
import de.fhlintstone.generator.structuredefinition.code.NestedClass;
import de.fhlintstone.generator.structuredefinition.code.TopLevelClass;
import de.fhlintstone.process.config.ProcessConfiguration;
import de.fhlintstone.process.config.StructureClassInstantiation;
import de.fhlintstone.utilities.StringUtilities;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Named;
import javax.lang.model.SourceVersion;
import lombok.extern.slf4j.XSlf4j;

/**
 * Default implementation of {@link ITypeInformationConverter}.
 */
@Named
@XSlf4j
public class TypeInformationConverter implements ITypeInformationConverter {

    private final IFrameworkTypeLocator frameworkTypeLocator;
    private final IGeneratedTypeNameRegistry generatedTypeNameRegistry;
    private final ITypeMapper typeMapper;

    /**
     * Constructor for dependency injection.
     *
     * @param frameworkTypeLocator      the {@link IFrameworkTypeLocator} to use
     * @param generatedTypeNameRegistry the {@link IGeneratedTypeNameRegistry} to
     *                                  use
     * @param typeMapper the {@link ITypeMapper} to use
     */
    @Inject
    public TypeInformationConverter(
            IFrameworkTypeLocator frameworkTypeLocator,
            IGeneratedTypeNameRegistry generatedTypeNameRegistry,
            ITypeMapper typeMapper) {
        super();
        this.frameworkTypeLocator = frameworkTypeLocator;
        this.generatedTypeNameRegistry = generatedTypeNameRegistry;
        this.typeMapper = typeMapper;
    }

    @Override
    public ITopLevelClass buildGeneratorModel(ProcessConfiguration configuration, ITypeInformation typeInformation)
            throws GeneratorException {
        logger.entry(typeInformation);
        final var canonicalURI = typeInformation.getCanonicalURI();
        final var typeName =
                this.generatedTypeNameRegistry.getClassName(canonicalURI).orElseThrow();
        if (typeName instanceof final ClassName className) {
            logger.debug(
                    "Converting type information for StructureDefinition {} into top-level class {}",
                    canonicalURI,
                    typeName);
            final var topLevelClass = createTopLevelClass(configuration, typeInformation, className);
            createConstructors(topLevelClass, typeInformation, Optional.empty());
            createAttributes(topLevelClass, typeInformation);
            createFixedValues(topLevelClass, typeInformation);
            createNestedTypes(topLevelClass, typeInformation);
            return logger.exit(topLevelClass);
        } else {
            throw logger.throwing(new GeneratorException(String.format(
                    "Target type name %s for StructureDefinition %s is not a ClassName", typeName, canonicalURI)));
        }
    }

    /**
     * Creates the class header data.
     * @throws GeneratorException
     */
    private TopLevelClass createTopLevelClass(
            ProcessConfiguration configuration, ITypeInformation typeInformation, ClassName className)
            throws GeneratorException {
        logger.entry(className);
        final var classConfig = typeInformation.getTopLevelConfiguration().orElseThrow();
        final var superClassName = this.typeMapper.determineSuperclass(typeInformation);
        final var classData = TopLevelClass.builder()
                .withOutputPath(configuration.getOutputPath())
                .withClassName(className)
                .withSuperclassName(superClassName.getType())
                .withAbstractClass(classConfig.getInstantiation() == StructureClassInstantiation.ABSTRACT)
                .withDerivedFromPrimitiveType(this.typeMapper.isDerivedFromPrimitiveType(typeInformation))
                .withStructureDefinitionURL(determineStructureDefintionUrl(typeInformation))
                .withStructureDefinitionName(typeInformation.getDefinitionName())
                .withDescription(typeInformation.getDescription())
                .withClassAnnotation(typeInformation.getClassAnnotation())
                .build();
        return logger.exit(classData);
    }

    /**
     * Adds the nested types to the class data.
     * @throws GeneratorException
     */
    private void createNestedTypes(TopLevelClass topLevelClass, ITypeInformation topLevelTypeInformation)
            throws GeneratorException {
        logger.entry(topLevelClass);
        for (final var nestedType : topLevelTypeInformation.getNestedTypes()) {
            final var nestedTypeName = this.generatedTypeNameRegistry
                    .getNestedClassName(
                            topLevelTypeInformation.getCanonicalURI(),
                            nestedType.getElementId().orElseThrow())
                    .orElseThrow();
            logger.debug("Transferring nested type {}", nestedTypeName);
            final var nestedClass = createNestedClass(topLevelTypeInformation, nestedType, nestedTypeName);
            createConstructors(nestedClass, topLevelTypeInformation, Optional.of(nestedType));
            createAttributes(nestedClass, nestedType);
            createFixedValues(nestedClass, nestedType);
            topLevelClass.addNestedClass(nestedClass);
        }
        logger.exit();
    }

    /**
     * Adds the constructors to the class data.
     * @throws GeneratorException
     */
    private void createConstructors(
            ClassData classData,
            ITypeInformation topLevelTypeInformation,
            Optional<ITypeInformation> nestedTypeInformation)
            throws GeneratorException {
        logger.entry(classData);
        logger.debug(
                "Creating constructor descriptors for type {}",
                classData.getClassName().reflectionName());

        // determine the framework types to derive the constructor signatures from
        IMappedType frameworkSuperclass;
        if (nestedTypeInformation.isPresent()) {
            frameworkSuperclass = this.typeMapper.determineFrameworkSuperclass(nestedTypeInformation.get());
        } else {
            frameworkSuperclass = this.typeMapper.determineFrameworkSuperclass(topLevelTypeInformation);
        }
        logger.debug("Using framwork superclass {} used to determine constructors", frameworkSuperclass.getType());

        for (final var constructor : this.frameworkTypeLocator.determineConstructors(frameworkSuperclass.getType())) {
            final var builder = ClassConstructor.builder().withDeprecated(constructor.deprecated());
            for (final var parameter : constructor.parameters()) {
                builder.withParameter(new ConstructorParameter(parameter.type(), parameter.name()));
            }
            classData.addConstructor(builder.build());
        }
        logger.exit();
    }

    /**
     * Adds the attributes to the class data.
     * @throws GeneratorException
     */
    private void createAttributes(ClassData classData, ITypeInformation typeInformation) throws GeneratorException {
        logger.entry(classData);
        for (final var sourceAttribute : typeInformation.getAttributes()) {
            logger.debug("Transferring attribute {}", sourceAttribute.getFhirName());
            final var mappedTypes = this.typeMapper.determineAttributeTypes(typeInformation, sourceAttribute);
            final var primitiveType = getAttributePrimitiveType(sourceAttribute, mappedTypes);
            final var attributeType = getAttributeType(sourceAttribute.getElementId(), mappedTypes);
            final var classAttribute = ClassAttribute.builder()
                    .withName(mapFhirNameToAttributeName(sourceAttribute.getFhirName()))
                    .withFhirName(sourceAttribute.getFhirName())
                    .withElementId(sourceAttribute.getElementId())
                    .withAttributeType(attributeType)
                    .withActualType(getActualAttributeType(sourceAttribute, mappedTypes))
                    .withCreationType(getAttributeCreationType(sourceAttribute, mappedTypes))
                    .withPrimitiveType(primitiveType)
                    .withMappedTypes(mappedTypes)
                    .withPropertyTypes(getPropertyTypes(sourceAttribute, mappedTypes))
                    .withBaseCastingMethod(getBaseCastingMethod(sourceAttribute, mappedTypes))
                    .withAccessorGenerationMode(
                            getAttributeAccessorGenerationMode(sourceAttribute, mappedTypes, primitiveType))
                    .withDescription(getAttributeDescription(sourceAttribute))
                    .withDefinition(sourceAttribute.getDefinition())
                    .withMin(sourceAttribute.getMin())
                    .withMax(sourceAttribute.getMax())
                    .withDerivedFromPrimitiveType(
                            this.typeMapper.isDerivedFromPrimitiveType(typeInformation, sourceAttribute))
                    .withExtensionUrl(sourceAttribute.getExtensionUrl())
                    .withExtension(sourceAttribute.isExtension())
                    .withModifierExtension(sourceAttribute.isModifier())
                    .withAnnotations(getAttributeAnnotations(sourceAttribute, mappedTypes))
                    .build();
            classData.addAttribute(classAttribute);
        }
        logger.exit();
    }

    /**
     * Creates the class header data.
     * @param topLevelTypeInformation
     * @throws GeneratorException
     */
    private NestedClass createNestedClass(
            ITypeInformation topLevelTypeInformation, ITypeInformation nestedTypeInformation, ClassName className)
            throws GeneratorException {
        logger.entry(className);
        final var classConfig = nestedTypeInformation.getNestedConfiguration().orElseThrow();
        final var nestedElementId = nestedTypeInformation.getElementId().orElseThrow();
        final var isAbstract = classConfig.getInstantiation() == StructureClassInstantiation.ABSTRACT;
        final var superClassName = this.generatedTypeNameRegistry
                .getNestedSuperclassName(topLevelTypeInformation.getCanonicalURI(), nestedElementId)
                .orElseThrow();
        final var classData = NestedClass.builder()
                .withElementId(nestedElementId)
                .withClassName(className)
                .withSuperclassName(superClassName)
                .withAbstractClass(isAbstract)
                .withStructureDefinitionURL(determineStructureDefintionUrl(nestedTypeInformation))
                .withStructureDefinitionName(nestedTypeInformation.getDefinitionName())
                .withDerivedFromPrimitiveType(this.typeMapper.isDerivedFromPrimitiveType(nestedTypeInformation))
                .withClassAnnotation(nestedTypeInformation.getClassAnnotation())
                .build();
        return logger.exit(classData);
    }

    /**
     * Determines the description for an attribute.
     */
    private Optional<String> getAttributeDescription(final ITypeAttribute attribute) {
        logger.entry(attribute);
        final var descriptionBuilder = new StringBuilder();
        final var shortDefinition = attribute.getShortDefinition();
        final var definition = attribute.getDefinition();
        if (shortDefinition.isPresent()) {
            descriptionBuilder.append(shortDefinition.get());
        }
        if (shortDefinition.isPresent() && definition.isPresent()) {
            descriptionBuilder.append("\n\n");
        }
        if (definition.isPresent()) {
            descriptionBuilder.append(definition.get());
        }
        if (shortDefinition.isEmpty() && definition.isEmpty()) {
            descriptionBuilder.append(String.format(
                    "FHIR element %s (no further definition provided in the model).", attribute.getFhirName()));
        }
        final var description = Optional.of(descriptionBuilder.toString());
        return logger.exit(description);
    }

    /**
     * Determines the annotations to generate for an attribute.
     * @param mappedTypes
     */
    private Collection<IClassAttributeAnnotation> getAttributeAnnotations(
            ITypeAttribute attribute, ImmutableCollection<IMappedType> mappedTypes) {
        logger.entry(attribute);
        final var result = new ArrayList<IClassAttributeAnnotation>();
        // the @Child annotation is added unconditionally
        result.add(generateAttributeAnnotationChild(attribute, mappedTypes));

        // the @Extension attribute is added if the attribute is an extension
        if (attribute.isExtension()) {
            result.add(generateAttributeAnnotationExtension(attribute));
        }

        // the @Description annotation is added when at least one of the texts is
        // available
        if (attribute.getDefinition().isPresent()
                || attribute.getShortDefinition().isPresent()) {
            result.add(generateAttributeAnnotationDescription(attribute));
        }

        // the @Binding annotation is added when a binding is defined
        if (attribute.getBindingValueSet().isPresent()) {
            result.add(generateAttributeAnnotationBinding(attribute));
        }

        return logger.exit(result);
    }

    /**
     * Generates the {@link Child} annotation for an attribute.
     * @param mappedTypes
     */
    private IClassAttributeAnnotation generateAttributeAnnotationChild(
            ITypeAttribute attribute, ImmutableCollection<IMappedType> mappedTypes) {
        logger.entry(attribute);
        final var childAnnotationBuilder =
                ClassAttributeAnnotation.builder().withAnnotation(ClassName.get(Child.class));
        childAnnotationBuilder.withParameter("name", attribute.getFhirName());
        // TODO #39 support order attribute of @Child annotation
        if (attribute.getMin() != 0) {
            childAnnotationBuilder.withParameter("min", attribute.getMin());
        }
        if (attribute.getMax() != 1) {
            childAnnotationBuilder.withParameter("max", attribute.getMax());
        }
        if (mappedTypes.size() > 1) {
            // for the type, see ca.uhn.fhir.model.api.annotation.Child#type
            childAnnotationBuilder.withParameter(
                    "type", mappedTypes.stream().map(t -> t.getType()).toArray(TypeName[]::new));
        }
        if (attribute.isModifier()) {
            childAnnotationBuilder.withParameter("modifier", Boolean.TRUE);
        }
        // TODO #39 support summary attribute of @Child annotation
        return logger.exit(childAnnotationBuilder.build());
    }

    /**
     * Generates the {@link Extension} annotation for an attribute.
     */
    private IClassAttributeAnnotation generateAttributeAnnotationExtension(ITypeAttribute attribute) {
        logger.entry(attribute);
        final var extensionAnnotationBuilder =
                ClassAttributeAnnotation.builder().withAnnotation(ClassName.get(Extension.class));
        // TODO #39 support attribute definedLocally of @Extension annotation
        if (attribute.isModifier()) {
            extensionAnnotationBuilder.withParameter("isModifier", Boolean.TRUE);
        }
        extensionAnnotationBuilder.withParameter(
                "url", attribute.getExtensionUrl().orElseThrow());
        return logger.exit(extensionAnnotationBuilder.build());
    }

    /**
     * Generates the {@link Description} annotation for an attribute.
     */
    private IClassAttributeAnnotation generateAttributeAnnotationDescription(ITypeAttribute attribute) {
        logger.entry(attribute);
        final var descriptionAnnotationBuilder =
                ClassAttributeAnnotation.builder().withAnnotation(ClassName.get(Description.class));
        final var definition = attribute.getDefinition();
        if (definition.isPresent()) {
            descriptionAnnotationBuilder.withParameter("value", definition.get());
        }
        final var shortDefinition = attribute.getShortDefinition();
        if (shortDefinition.isPresent()) {
            descriptionAnnotationBuilder.withParameter("shortDefinition", shortDefinition.get());
        }
        // TODO #39 support attribute example of @Description annotation
        return logger.exit(descriptionAnnotationBuilder.build());
    }

    /**
     * Generates the {@link Binding} annotation for an attribute.
     */
    private IClassAttributeAnnotation generateAttributeAnnotationBinding(ITypeAttribute attribute) {
        logger.entry(attribute);
        return logger.exit(ClassAttributeAnnotation.builder()
                .withAnnotation(ClassName.get(Binding.class))
                .withParameter("valueSet", attribute.getBindingValueSet().orElseThrow())
                .build());
    }

    /**
     * Determines the primitive type of an attribute, if applicable.
     * @param mappedTypes
     * @throws GeneratorException
     */
    private Optional<TypeName> getAttributePrimitiveType(
            ITypeAttribute attribute, ImmutableCollection<IMappedType> mappedTypes) throws GeneratorException {
        logger.entry(attribute);
        if (attribute.getMax() != 1) {
            // list attributes never have a primitive type
            return logger.exit(Optional.empty());
        } else {
            final var attributeType = getAttributeType(attribute.getElementId(), mappedTypes);
            return logger.exit(this.frameworkTypeLocator.determinePrimitiveType(attributeType));
        }
    }

    /**
     * Determines the actual Java type to use when generating or exposing the attribute.
     * @throws GeneratorException
     */
    private TypeName getActualAttributeType(
            final ITypeAttribute attribute, ImmutableCollection<IMappedType> mappedTypes) throws GeneratorException {
        return getAttributeCollectionType(attribute, mappedTypes, ClassName.get(List.class));
    }

    /**
     * Determines the actual Java type to use when creating an instance of the attribute.
     * @throws GeneratorException
     */
    private TypeName getAttributeCreationType(
            final ITypeAttribute attribute, ImmutableCollection<IMappedType> mappedTypes) throws GeneratorException {
        return getAttributeCollectionType(attribute, mappedTypes, ClassName.get(ArrayList.class));
    }

    /**
     * Determines the a type for an attribute, wrapping it in a collection type if necessary to represent a repeatable element.
     * @throws GeneratorException
     */
    private TypeName getAttributeCollectionType(
            final ITypeAttribute attribute, ImmutableCollection<IMappedType> mappedTypes, ClassName collectionType)
            throws GeneratorException {
        logger.entry(attribute);
        final var attributeType = getAttributeType(attribute.getElementId(), mappedTypes);
        if (attribute.getMax() != 1) {
            // generate a parameterized type (List<TheActualType>)
            if (attributeType.isPrimitive()) {
                // we have to box primitive types
                return logger.exit(ParameterizedTypeName.get(collectionType, attributeType.box()));
            } else {
                return logger.exit(ParameterizedTypeName.get(collectionType, attributeType));
            }
        } else {
            return logger.exit(attributeType);
        }
    }

    /**
     * Determines which mode to use when generating the accessors.
     *
     * @param primitiveType
     */
    private AccessorGenerationMode getAttributeAccessorGenerationMode(
            ITypeAttribute attribute, ImmutableCollection<IMappedType> mappedTypes, Optional<TypeName> primitiveType) {
        logger.entry(attribute);
        if (mappedTypes.size() > 1) {
            // the ElementDefinition specifies multiple types
            if (attribute.getMax() != 1) {
                // ElementDefinition.max > 1, so the attribute is a List<Type>
                return logger.exit(AccessorGenerationMode.MULTI_TYPE_REPEATING);
            } else {
                // ElementDefinition.max = 1, so the attribute is a Type
                return logger.exit(AccessorGenerationMode.MULTI_TYPE_NON_REPEATING);
            }
        }
        // at this point, the element specifies a single type
        if (attribute.getMax() != 1) {
            // ElementDefinition.max > 1, so the attribute is a List<SomeType>
            if (primitiveType.isPresent()) {
                // the element has a primitive type, so SomeType is a StringType, IntegerType, ...
                return logger.exit(AccessorGenerationMode.SINGLE_TYPE_REPEATING_PRIMITIVE);
            } else {
                // the element does not have a primitive type, so SomeType is an Address, Patient, ...
                return logger.exit(AccessorGenerationMode.SINGLE_TYPE_REPEATING_ELEMENT);
            }
        } else {
            // ElementDefinition.max = 1, so the attribute is a SomeType
            if (primitiveType.isPresent()) {
                // the element has a primitive type, so SomeType is a StringType, IntegerType, ...
                return logger.exit(AccessorGenerationMode.SINGLE_TYPE_NON_REPEATING_PRIMITIVE);
            } else {
                // the element does not have a primitive type, so SomeType is an Address, Patient, ...
                return logger.exit(AccessorGenerationMode.SINGLE_TYPE_NON_REPEATING_ELEMENT);
            }
        }
    }

    /**
     * Determines the property types of the attribute.
     */
    private Iterable<String> getPropertyTypes(
            ITypeAttribute sourceAttribute, ImmutableCollection<IMappedType> mappedTypes) {
        logger.entry(sourceAttribute);
        final var typeCodes = mappedTypes.stream().map(IMappedType::getTypeCode).toList();
        return logger.exit(typeCodes);
    }

    /**
     * Determines the casting method provided by the Base class for the attribute type.
     * @throws GeneratorException
     */
    private Optional<String> getBaseCastingMethod(
            ITypeAttribute sourceAttribute, ImmutableCollection<IMappedType> mappedTypes) throws GeneratorException {
        logger.entry(sourceAttribute);
        final var attributeType = getAttributeType(sourceAttribute.getElementId(), mappedTypes);
        final var result = this.frameworkTypeLocator.getBaseCastingMethod(attributeType);
        return logger.exit(result);
    }

    /**
     * Adds the fixed values to the class data.
     * @throws GeneratorException
     */
    private void createFixedValues(ClassData targetClass, ITypeInformation typeInformation) throws GeneratorException {
        logger.entry(targetClass);
        for (final var fixedValue : typeInformation.getFixedValues()) {
            logger.debug("Transferring fixed value of element {}", fixedValue.getElementPath());
            targetClass.addFixedValueRule(createFixedValueRule(fixedValue, Optional.empty()));
        }
        logger.exit();
    }

    /**
     * Creates an {@link IFixedValueRule} to represent an {@link ITypeFixedValue} for the code generator.
     * @param fixedValue
     * @param prefix
     * @return
     * @throws GeneratorException
     */
    private IFixedValueRule createFixedValueRule(ITypeFixedValue fixedValue, Optional<String> prefix)
            throws GeneratorException {
        logger.entry(fixedValue);
        final var elementId = fixedValue.getLocalName();
        final var ruleBuilder = FixedValueRule.builder()
                .withPath(fixedValue.getElementPath())
                .withPropertyName(elementId)
                .withPrefix(prefix)
                .withPropertyType(mapFixedValueType(fixedValue));
        if (fixedValue.isComplex()) {
            final var componentPrefix = prefix.isPresent() ? prefix.get() + "_" + elementId : elementId;
            for (final var componentValue : fixedValue.getComponentValues()) {
                final var componentRule = createFixedValueRule(componentValue, Optional.of(componentPrefix));
                ruleBuilder.withProperty(componentRule);
            }
        } else {
            final var stringifiedValue = fixedValue.getValue();
            if (stringifiedValue.isPresent()) {
                ruleBuilder
                        .withValue(Optional.of(stringifiedValue.get().getValue()))
                        .withLiteral(stringifiedValue.get().isLiteral());
            } else {
                logger.warn("Unable to obtain stringified fixed value for element {}", fixedValue.getElementId());
            }
        }
        return logger.exit(ruleBuilder.build());
    }

    /**
     * Maps the {@link ITypeSpecification} carried by an {@link ITypeFixedValue} instance to the
     * {@link TypeName} used to create the target value.
     * @param fixedValue
     * @return
     * @throws GeneratorException
     */
    private TypeName mapFixedValueType(ITypeFixedValue fixedValue) throws GeneratorException {
        logger.entry(fixedValue);
        final var typeSpecifications = fixedValue.getTypes();
        switch (typeSpecifications.size()) {
            case 0:
                throw logger.throwing(new GeneratorException(String.format(
                        "Unable to determine type to represent element %s as a fixed value",
                        fixedValue.getElementId())));
            case 1:
                final var typeSpecification = typeSpecifications.iterator().next();
                if (typeSpecification.hasProfiles()) {
                    logger.warn(
                            "Profiles of fixed value of element {} is currently not supported and will be ignored",
                            fixedValue.getElementId());
                }
                if (typeSpecification.hasTargetProfiles()) {
                    logger.warn(
                            "Target profiles of fixed value of element {} is currently not supported and will be ignored",
                            fixedValue.getElementId());
                }
                final var mappedType = this.frameworkTypeLocator.determineType(
                        typeSpecification.getTypeCode().canonical());
                if (mappedType.isEmpty()) {
                    throw logger.throwing(new GeneratorException(String.format(
                            "Unable to determine framework type to represent element %s as a fixed value (type code: %s)",
                            fixedValue.getElementId(), typeSpecification.getTypeCode())));
                } else {
                    return logger.exit(mappedType.get());
                }
            default:
                final String conflictingTypeCodes = typeSpecifications.stream()
                        .map(t -> t.getTypeCode().code())
                        .collect(Collectors.joining(","));
                throw logger.throwing(new GeneratorException(String.format(
                        "Unable to uniquely determine type to represent element %s as a fixed value (conflicting types: %s)",
                        fixedValue.getElementId(), conflictingTypeCodes)));
        }
    }

    private String determineStructureDefintionUrl(ITypeInformation typeInformation) {
        logger.entry(typeInformation);
        final var topLevelClassConfig =
                typeInformation.getTopLevelConfiguration().orElseThrow();
        if (topLevelClassConfig.useVersion()) {
            return logger.exit(typeInformation.getVersionedCanonicalURI());
        } else {
            return logger.exit(typeInformation.getCanonicalURI().toString());
        }
    }

    /**
     * Determines the attribute name from the FHIR element name.
     *
     * @throws GeneratorException
     */
    private String mapFhirNameToAttributeName(String localName) throws GeneratorException {
        logger.entry(localName);
        var newName = localName;

        // The element name is part of the path. The rules quoted below are taken from
        // https://hl7.org/fhir/R4/elementdefinition.html#path

        // "Element names (the parts of a path delineated by the '.' character) SHALL NOT contain
        // whitespace (i.e. Unicode characters marked as whitespace)"
        final var whitespacePattern = Pattern.compile("\\s");
        final var whitespaceMatcher = whitespacePattern.matcher(newName);
        if (whitespaceMatcher.find()) {
            logger.warn(
                    "The FHIR name '{}' contains whitespace characters that have to be dropped for the attribute name.",
                    localName);
            newName = whitespaceMatcher.replaceAll("");
        }

        // "Element names SHALL NOT contain the characters ,:;'"/|?!@#$%^&*()[]{}"
        final var prohibitedCharacterPattern = Pattern.compile("[,:;'\\\"/|?!@#$%^&*()\\[\\]\\{\\}]");
        final var prohibitedCharacterMatcher = prohibitedCharacterPattern.matcher(newName);
        if (prohibitedCharacterMatcher.find()) {
            logger.warn(
                    "The FHIR name '{}' contains prohibited characters that have to be dropped for the attribute name.",
                    localName);
            newName = prohibitedCharacterMatcher.replaceAll("");
        }

        // "Element names SHOULD not contain non-ASCII characters"
        final var nonASCIIPattern = Pattern.compile("[^\\p{ASCII}]");
        final var nonASCIIMatcher = nonASCIIPattern.matcher(newName);
        if (nonASCIIMatcher.find()) {
            logger.warn(
                    "The FHIR name '{}' contains non-ASCII characters that have to be dropped for the attribute name.",
                    localName);
            newName = nonASCIIMatcher.replaceAll("");
        }

        // "Element names SHALL NOT exceed 64 characters in length"
        // This is actually not a limitation that concerns us, so in order to avoid unnecessary noise, we
        // ignore this

        // "By convention, each path starts with an uppercase letter (type) but all the element names that
        // follow this are lowercase (not type names)."
        // --> We simly convert all first characters to lower case without reporting.
        newName = StringUtilities.toFirstLower(newName);

        // The remaining rules are not defined by the FHIR standard, but rather a result of the Java restrictions.

        // turn some-thing into someThing
        newName = StringUtilities.separatorToCamelCase(newName, "-");

        // deal with attribute names like "class"...
        if (SourceVersion.isKeyword(newName)) {
            newName = "_" + newName;
        }
        if (!SourceVersion.isIdentifier(newName)) {
            throw logger.throwing(new GeneratorException(
                    String.format("Unable to determine valid identifier for FHIR attribute '%s'", localName)));
        }
        return logger.exit(newName);
    }

    /**
     * Returns the Java type used to implement the attribute. For elements that may
     * have more than one actual type, this will be a generic Type. This can be used
     * to supply {@link Child#type()}.
     *
     * @param id the element/extension ID (only used for logging purposes)
     * @oaram modelTypes the mapped model types
     */
    private TypeName getAttributeType(String id, ImmutableCollection<IMappedType> mappedTypes)
            throws GeneratorException {
        logger.entry(id);
        if (mappedTypes.isEmpty()) {
            throw logger.throwing(new GeneratorException(String.format("No type for element %s found", id)));
        } else if (mappedTypes.size() > 1) {
            return this.frameworkTypeLocator.getGenericType();
        }
        return mappedTypes.stream().findFirst().orElseThrow().getType();
    }
}
