/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.intermediate;

import ca.uhn.fhir.model.api.annotation.Child;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.accessors.IAccessorProvider;
import de.fhlintstone.accessors.UnsupportedTypeException;
import de.fhlintstone.accessors.implementations.IFrameworkTypeLocator;
import de.fhlintstone.accessors.implementations.ITypeSpecification;
import de.fhlintstone.accessors.model.IBaseAccessor;
import de.fhlintstone.accessors.model.IElementDefinitionAccessor;
import de.fhlintstone.accessors.model.IPrimitiveTypeAccessor;
import de.fhlintstone.accessors.model.IPropertyAccessor;
import de.fhlintstone.accessors.model.IStructureDefinitionAccessor;
import de.fhlintstone.fhir.ClassAnnotation;
import de.fhlintstone.fhir.ExtensionType;
import de.fhlintstone.fhir.FhirException;
import de.fhlintstone.fhir.IStructureDefinitionIntrospector;
import de.fhlintstone.fhir.dependencies.IDependencyNode;
import de.fhlintstone.fhir.elements.ElementBaseRelationship;
import de.fhlintstone.fhir.elements.IElementTree;
import de.fhlintstone.fhir.elements.IElementTreeBuilder;
import de.fhlintstone.fhir.elements.IElementTreeNode;
import de.fhlintstone.generator.GeneratorException;
import de.fhlintstone.packages.FhirResourceType;
import de.fhlintstone.packages.IPackageRegistry;
import de.fhlintstone.process.config.NestedClassConfiguration;
import de.fhlintstone.process.config.ProcessConfiguration;
import de.fhlintstone.process.config.StructureDefinitionClassConfiguration;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Named;
import lombok.extern.slf4j.XSlf4j;
import org.hl7.fhir.instance.model.api.IBaseResource;

/**
 * Default implementation of {@link ITypeInformationBuilder}.
 */
@Named
@XSlf4j
public class TypeInformationBuilder implements ITypeInformationBuilder {

    private final IPackageRegistry packageRegistry;
    private final IAccessorProvider accessorProvider;
    private final IFrameworkTypeLocator frameworkTypeLocator;
    private final IStructureDefinitionIntrospector structureDefinitionIntrospector;
    private final ITypeInformationStore typeInformationStore;
    private final IElementTreeBuilder elementTreeBuilder;

    /**
     * Constructor for dependency injection
     *
     * @param packageRegistry                 the {@link IPackageRegistry} to use
     * @param accessorProvider                the {@link IAccessorProvider} to use
     * @param frameworkTypeLocator            the {@link IFrameworkTypeLocator} to use
     * @param structureDefinitionIntrospector the {@link IStructureDefinitionIntrospector} to use
     * @param typeInformationStore            the {@link ITypeInformationStore} to use
     * @param elementTreeBuilder              the {@link IElementTreeBuilder} to use
     */
    @Inject
    public TypeInformationBuilder(
            IPackageRegistry packageRegistry,
            IAccessorProvider accessorProvider,
            IFrameworkTypeLocator frameworkTypeLocator,
            IStructureDefinitionIntrospector structureDefinitionIntrospector,
            ITypeInformationStore typeInformationStore,
            IElementTreeBuilder elementTreeBuilder) {
        super();
        this.packageRegistry = packageRegistry;
        this.accessorProvider = accessorProvider;
        this.frameworkTypeLocator = frameworkTypeLocator;
        this.structureDefinitionIntrospector = structureDefinitionIntrospector;
        this.typeInformationStore = typeInformationStore;
        this.elementTreeBuilder = elementTreeBuilder;
    }

    @Override
    public Optional<ITypeInformation> buildTypeInformation(
            ProcessConfiguration configuration, IDependencyNode dependencyNode) throws GeneratorException {
        logger.entry(dependencyNode);
        Optional<? extends ITypeInformation> typeInformation;
        final var resource = resolveResource(dependencyNode);
        if (resource.isPresent()) {
            final var resourceType =
                    dependencyNode.getResourceType().orElse(FhirResourceType.fromResource(resource.get()));
            if (resourceType != FhirResourceType.STRUCTURE_DEFINITION) {
                throw logger.throwing(new GeneratorException(String.format(
                        "TypeInformationBuilder for Structure Definitions called for unsupported resource type %s",
                        resourceType)));
            }
            typeInformation = buildTopLevelType(configuration, dependencyNode, resource.get());
        } else {
            throw logger.throwing(new GeneratorException(
                    String.format("Resource %s can not be resolved", dependencyNode.getResourceURI())));
        }
        return logger.exit(typeInformation.map(t -> t));
    }

    /**
     * Retrieves the {@link IBaseResource} that contains the StructureDefinition.
     */
    private Optional<IBaseResource> resolveResource(final IDependencyNode dependencyNode) {
        logger.entry(dependencyNode);
        final var uri = dependencyNode.getResourceURI();
        var resource = dependencyNode.getResource();
        if (!resource.isPresent()) {
            resource = this.packageRegistry.getUniqueResource(uri);
        }
        if (resource.isEmpty()) {
            logger.warn("Ignoring unresolvable resource {}", uri);
        }
        return logger.exit(resource);
    }

    /**
     * Creates the top-level type information object for the configured StructureDefinition.
     */
    @SuppressWarnings("java:S3776") // can't be split up any more without making the intent hard to discern
    private Optional<? extends ITypeInformation> buildTopLevelType(
            ProcessConfiguration configuration, IDependencyNode dependencyNode, IBaseResource resource)
            throws GeneratorException {
        logger.entry(dependencyNode, resource);
        TypeInformation newType = null;
        final var uri = dependencyNode.getResourceURI();
        final var accessor = this.accessorProvider.provideStructureDefinitionAccessor(resource);
        final var classConfiguration = configuration.getStructureDefinition(uri);

        // First check for framework types (which do not have to be generated) and wrap
        // them accordingly.
        final var frameworkType = dependencyNode.getFrameworkType();
        if (frameworkType.isPresent()) {
            newType = buildTopLevelFrameworkType(uri, accessor, frameworkType.get());
        } else {
            // Not a framework type - check for extensions (because we need to distinguish
            // between simple and complex extensions and non-extension types)
            final var extensionType = determineExtensionType(accessor);
            if (extensionType == ExtensionType.SIMPLE) {
                newType = buildTopLevelSimpleExtension(uri, accessor);
            } else {
                if (classConfiguration.isPresent()) {
                    final var elementTree = createElementTree(accessor);
                    if (extensionType == ExtensionType.COMPLEX) {
                        newType = buildTopLevelComplexExtension(classConfiguration.get(), uri, accessor, elementTree);
                    } else {
                        newType = buildTopLevelStructure(classConfiguration.get(), uri, accessor, elementTree);
                    }
                    // Add the nested types after the top-level type has been created.
                    for (final var nestedConfiguration :
                            classConfiguration.get().getNestedClasses()) {
                        final var node = elementTree.getNode(nestedConfiguration.getElementId());
                        if (node.isPresent()) {
                            newType.addNestedType(buildNestedType(
                                    classConfiguration.get(),
                                    nestedConfiguration,
                                    node.get(),
                                    uri,
                                    accessor,
                                    elementTree));
                        } else {
                            logger.warn(
                                    "Element definition for {} not found, nested type {} cannot be generated.",
                                    nestedConfiguration.getElementId(),
                                    nestedConfiguration.getClassName());
                        }
                    }
                    // Distribute the fixed values between top-level and nested classes.
                    try {
                        generateFixedValues(newType, elementTree);
                    } catch (final FhirException e) {
                        throw logger.throwing(new GeneratorException(e.getMessage(), e));
                    }
                } else {
                    logger.warn(
                            "No configuration entry found for StructureDefinition {}. No class will be generated for this StructureDefinition.",
                            uri);
                }
            }
        }
        return logger.exit(Optional.ofNullable(newType));
    }

    /**
     * Creates the nested type information object.
     */
    private ITypeInformation buildNestedType(
            StructureDefinitionClassConfiguration topLevelConfiguration,
            NestedClassConfiguration nestedConfiguration,
            IElementTreeNode elementTreeNode,
            URI topLevelCanonicalURI,
            IStructureDefinitionAccessor topLevelAccessor,
            IElementTree elementTree)
            throws GeneratorException {
        final var elementId = nestedConfiguration.getElementId();
        logger.entry(topLevelCanonicalURI, elementId);
        logger.debug(
                "Adding nested type information for element {} of StructureDefinition {}",
                elementId,
                topLevelCanonicalURI);

        // The existence of a nested type configuration means that something has to be generated.
        // Therefore we can disregard framework types here.

        // Obtain the element tree node that the nested type configuration refers to.
        final var elementNode = elementTree
                .getNode(elementId)
                .orElseThrow(() -> new GeneratorException(String.format(
                        "Unable to locate element %s in element tree of StructureDefinition %s",
                        elementId, topLevelAccessor.getName().orElse("(none)"))));

        // Determine the type specifications of the element
        final var typeSpecifications = getTypeSpecifications(elementTreeNode.getSnapshotElement());

        // Determine the type code of the element (for now we assume there's only one type code and ignore the profile).
        final var typeCode =
                determineNestedElementTypeCode(topLevelCanonicalURI, topLevelAccessor, elementId, elementNode);

        // Determine the annotation to put on the nested class.
        URI structureDefinitionReference;
        IStructureDefinitionAccessor structureDefinition;
        Optional<ClassAnnotation> classAnnotation;
        URI baseDefinition;
        try {
            structureDefinitionReference = this.frameworkTypeLocator.makeAbsoluteStructureDefinitionReference(typeCode);
            structureDefinition =
                    this.accessorProvider.provideStructureDefinitionAccessor(structureDefinitionReference);
            classAnnotation = this.structureDefinitionIntrospector.getClassAnnotation(structureDefinition);
            baseDefinition = determineBaseDefinition(structureDefinition);
        } catch (final URISyntaxException | FhirException e) {
            throw new GeneratorException(String.format(
                    "Unable to determine StructureDefinition URI from type code %s for element %s in element tree of StructureDefinition %s",
                    typeCode, elementId, topLevelAccessor.getName().orElse("(none)")));
        }

        final var newType = TypeInformation.builder()
                .withTypeSpecifications(typeSpecifications)
                .withCanonicalURI(structureDefinitionReference)
                .withVersionedCanonicalURI(
                        structureDefinition.getVersionedUrl().orElse(structureDefinitionReference.toString()))
                .withParentCanonicalURI(Optional.of(topLevelCanonicalURI))
                .withParentVersionedCanonicalURI(
                        Optional.of(topLevelAccessor.getVersionedUrl().orElse(topLevelCanonicalURI.toString())))
                .withElementId(Optional.of(elementId))
                .withGenerated(true)
                .withDefinitionName(typeCode)
                .withClassAnnotation(classAnnotation)
                .withTopLevelConfiguration(Optional.of(topLevelConfiguration))
                .withNestedConfiguration(Optional.of(nestedConfiguration))
                .withBaseDefinition(baseDefinition)
                .withBaseTypeInformation(this.typeInformationStore.getByCanonical(baseDefinition))
                .build();
        try {
            generateStructureAttributes(newType, elementNode);
        } catch (final FhirException e) {
            throw logger.throwing(new GeneratorException(e.getMessage(), e));
        }
        return logger.exit(newType);
    }

    /**
     * Determine the type code of the element used to generate a nested class.
     * @param uri
     * @param topLevelAccessor
     * @param elementId
     * @param elementNode
     * @return
     * @throws GeneratorException
     */
    private String determineNestedElementTypeCode(
            URI uri,
            IStructureDefinitionAccessor topLevelAccessor,
            final String elementId,
            final IElementTreeNode elementNode)
            throws GeneratorException {
        logger.entry(elementNode);
        final var typeCodes = elementNode.getSnapshotElement().getType().stream()
                .map(type -> type.getCode())
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(Collectors.toSet());
        if (typeCodes.isEmpty()) {
            throw new GeneratorException(String.format(
                    "Unable to determine type code of element %s in element tree of StructureDefinition %s",
                    elementId, topLevelAccessor.getName().orElse("(none)")));
        } else if (typeCodes.size() > 1) {
            throw new GeneratorException(String.format(
                    "Unable to process multiple type codes of element %s in element tree of StructureDefinition %s",
                    elementId, topLevelAccessor.getName().orElse("(none)")));
        }
        final var typeCode = typeCodes.iterator().next();
        logger.debug("Type code of element {} of StructureDefinition {} is {}", elementId, uri, typeCode);
        return logger.exit(typeCode);
    }

    /**
     * Determines the {@link ExtensionType} of the StructureDefinition, i.e. whether
     * it is a simple, a complex or no extension at all.
     */
    private ExtensionType determineExtensionType(final IStructureDefinitionAccessor accessor) {
        logger.entry(accessor);
        var extensionType = ExtensionType.NONE;
        try {
            extensionType = this.structureDefinitionIntrospector.getExtensionType(accessor);
        } catch (final FhirException e) {
            logger.warn(
                    "Unable to determine extension type of StructureDefinition {}, assuming this is not an Extension",
                    accessor.getUrl().orElseThrow(),
                    e);
        }
        return logger.exit(extensionType);
    }

    /**
     * Creates the {@link ITypeInformation} for a framework type.
     */
    private TypeInformation buildTopLevelFrameworkType(
            final URI canonicalURI, final IStructureDefinitionAccessor accessor, final TypeName frameworkType)
            throws GeneratorException {
        logger.entry(canonicalURI);
        final Optional<ClassAnnotation> classAnnotation =
                this.structureDefinitionIntrospector.getClassAnnotationForFrameworkType(frameworkType);
        if (accessor.getSnapshot().isEmpty()) {
            throw logger.throwing(new GeneratorException(
                    "Unable to work with framework types where the FHIR package does not contain a snapshot."));
        }
        final var typeSpecifications = getTypeSpecifications(accessor);
        logger.debug(
                "StructureDefinition {} is a framework type that does not require a generated class", canonicalURI);
        final var newType = TypeInformation.builder()
                .withTypeSpecifications(typeSpecifications)
                .withCanonicalURI(canonicalURI)
                .withVersionedCanonicalURI(accessor.getVersionedUrl().orElse(canonicalURI.toString()))
                .withDefinitionName(accessor.getName().orElseThrow())
                .withDescription(accessor.getDescription())
                .withGenerated(false)
                .withFrameworkType(Optional.of(frameworkType))
                .withClassAnnotation(classAnnotation)
                .build();
        return logger.exit(newType);
    }

    /**
     * Creates the {@link ITypeInformation} for a StructureDefinition that describes
     * a simple extension.
     * @throws GeneratorException
     */
    private TypeInformation buildTopLevelSimpleExtension(URI canonicalURI, IStructureDefinitionAccessor accessor)
            throws GeneratorException {
        logger.entry(canonicalURI);
        final var typeSpecifications = getTypeSpecifications(accessor);
        logger.debug(
                "StructureDefinition {} is a simple extension that does not require a generated class", canonicalURI);
        final var baseDefinition = determineBaseDefinition(accessor);
        final var newType = TypeInformation.builder()
                .withTypeSpecifications(typeSpecifications)
                .withCanonicalURI(canonicalURI)
                .withVersionedCanonicalURI(accessor.getVersionedUrl().orElse(canonicalURI.toString()))
                .withDefinitionName(accessor.getName().orElseThrow())
                .withDescription(accessor.getDescription())
                .withGenerated(false) // no framework type, see ITypeInformation#getFrameworkType()
                .withBaseDefinition(baseDefinition)
                .build();
        return logger.exit(newType);
    }

    /**
     * Creates the {@link ITypeInformation} for a StructureDefinition that describes
     * a complex extension.
     *
     * @param classConfiguration
     * @param elementTree
     * @throws GeneratorException
     */
    private TypeInformation buildTopLevelComplexExtension(
            StructureDefinitionClassConfiguration classConfiguration,
            URI uri,
            IStructureDefinitionAccessor accessor,
            IElementTree elementTree)
            throws GeneratorException {
        logger.entry(uri);
        final var newType = prepareTopLevelClassInformation(classConfiguration, uri, accessor);
        try {
            generateExtensionAttributes(newType, elementTree.getRootNode());
        } catch (final FhirException e) {
            throw logger.throwing(new GeneratorException(e.getMessage(), e));
        }
        return logger.exit(newType);
    }

    /**
     * Creates the {@link ITypeInformation} for a StructureDefinition that describes
     * a non-extension structure
     */
    private TypeInformation buildTopLevelStructure(
            StructureDefinitionClassConfiguration classConfiguration,
            URI uri,
            IStructureDefinitionAccessor accessor,
            IElementTree elementTree)
            throws GeneratorException {
        logger.entry(uri);
        final var newType = prepareTopLevelClassInformation(classConfiguration, uri, accessor);
        try {
            generateStructureAttributes(newType, elementTree.getRootNode());
        } catch (final FhirException e) {
            throw logger.throwing(new GeneratorException(e.getMessage(), e));
        }
        return logger.exit(newType);
    }

    /**
     * Prepares the type information object with properties that are common for all
     * generated classes. Precondition: The StructureDefinition describes a custom
     * structure or a complex extension (for simple extensions and framework types,
     * no class is generated).
     */
    private TypeInformation prepareTopLevelClassInformation(
            StructureDefinitionClassConfiguration classConfiguration,
            final URI canonicalURI,
            final IStructureDefinitionAccessor accessor)
            throws GeneratorException {
        logger.entry(canonicalURI);
        Optional<ClassAnnotation> classAnnotation;
        try {
            classAnnotation = this.structureDefinitionIntrospector.getClassAnnotation(accessor);
        } catch (final FhirException e) {
            throw logger.throwing(new GeneratorException(
                    String.format("Unable to determine class annotation for StructureDefinition %s", canonicalURI), e));
        }
        ensureSnapshotExists(canonicalURI, accessor);
        final var typeSpecifications = getTypeSpecifications(accessor);
        final var baseDefinition = determineBaseDefinition(accessor);
        logger.debug("Building type information for StructureDefinition {}", canonicalURI);
        final var newType = TypeInformation.builder()
                .withTypeSpecifications(typeSpecifications)
                .withCanonicalURI(canonicalURI)
                .withVersionedCanonicalURI(accessor.getVersionedUrl().orElse(canonicalURI.toString()))
                .withDefinitionName(accessor.getName().orElseThrow())
                .withDescription(accessor.getDescription())
                .withGenerated(true)
                .withTopLevelConfiguration(Optional.of(classConfiguration))
                .withBaseDefinition(baseDefinition)
                .withBaseTypeInformation(this.typeInformationStore.getByCanonical(baseDefinition))
                .withClassAnnotation(classAnnotation)
                .build();
        return logger.exit(newType);
    }

    /**
     * Determines the base definition URI.
     * @param accessor
     * @return
     * @throws GeneratorException
     */
    private URI determineBaseDefinition(IStructureDefinitionAccessor accessor) throws GeneratorException {
        logger.entry(accessor);
        final var baseDefinition = accessor.getBaseDefinition();
        if (baseDefinition.isPresent()) {
            try {
                final var baseDefinitionURI =
                        this.frameworkTypeLocator.makeAbsoluteStructureDefinitionReference(baseDefinition.get());
                return logger.exit(baseDefinitionURI);
            } catch (final URISyntaxException e) {
                throw logger.throwing(new GeneratorException(
                        String.format(
                                "Unable to resolve BaseDefinition %s of StructureDefinition %s.",
                                baseDefinition.get(), accessor.getUrl().orElse("(unknown)")),
                        e));
            }
        } else {
            throw logger.throwing(new GeneratorException(String.format(
                    "StructureDefinition %s has no baseDefinition, unable to determine parent type",
                    accessor.getUrl().orElse("(unknown)"))));
        }
    }

    /**
     * Ensures that the structure definition has a complete snapshot.
     */
    private void ensureSnapshotExists(final URI uri, final IStructureDefinitionAccessor accessor)
            throws GeneratorException {
        logger.entry(uri);
        if (accessor.getSnapshot().isEmpty()) {
            throw logger.throwing(
                    new GeneratorException(String.format("StructureDefinition %s does not have a snapshot", uri)));
        }
        // TODO #17 enable snapshot generation
        //			try {
        //				final var fhirPackage = this.packageRegistry.getPackageOfResource(uri);
        //				if (fhirPackage.isEmpty()) {
        //					throw this.logger
        //						.throwing(new GeneratorException(
        //								String.format("Unable to determine package for StructureDefinition %s", uri)));
        //				}
        //				final var canonicalURL = fhirPackage.get().getCanonicalURL();
        //				String url = "";
        //				if (canonicalURL.isEmpty()) {
        //					this.logger
        //						.warn("Package {} of StructureDefinition {} is missing a canonical URL",
        //								fhirPackage.get().getName(), uri);
        //				} else {
        //					url = canonicalURL.get().toString();
        //				}
        //				final var webUrl = url; // maybe use the "homepage" entry... probably...?
        //				final var profileName = url; // check whether this is correct - not uri.toString() ?
        //				this.profileUtilities.generateSnapshot(parentType.getAccessor(), accessor, url, webUrl, profileName);
        //			} catch (FHIRException | AmbiguousResourceURIException e) {
        //				throw this.logger
        //					.throwing(new GeneratorException(
        //							String.format("Unable to generate snapshot for StructureDefinition %s", uri), e));
        //			}
        logger.exit();
    }

    /**
     * Creates an {@link IElementTree} to extract the attribute information.
     */
    private IElementTree createElementTree(final IStructureDefinitionAccessor accessor) throws GeneratorException {
        logger.entry(accessor);
        Optional<IBaseResource> baseResource;
        baseResource = this.packageRegistry.getUniqueResource(URI.create(accessor.getBaseDefinition()
                .orElseThrow(() -> new GeneratorException("StructureDefinition must have a baseDefinition set"))));
        final var baseAccessor = this.accessorProvider.provideStructureDefinitionAccessor(
                baseResource.orElseThrow(() -> new GeneratorException("Unable to load resource")));
        final var elementTree = this.elementTreeBuilder.buildElementTree(accessor, baseAccessor);
        return logger.exit(elementTree);
    }

    /**
     * Generates the attributes used to represent the elements of a complex
     * extension.
     * @throws FhirException
     */
    private void generateExtensionAttributes(TypeInformation information, IElementTreeNode elementTreeNode)
            throws GeneratorException, FhirException {
        logger.entry(elementTreeNode);
        // For a complex extension, we have to examine the slices of the element
        // Extension.extension.
        final var extensionNode = elementTreeNode
                .getChild("extension")
                .orElseThrow(
                        () -> new GeneratorException("A complex extension must have an Extension.extension element"));
        for (final var slice : extensionNode.getSlices()) {
            final var fhirName = slice.getSliceName().orElseThrow();
            // Each slice has to have the the type Extension - check that first.
            final var sliceTypes = slice.getSnapshotElement().getType();
            if (sliceTypes.size() != 1) {
                throw logger.throwing(new GeneratorException(String.format(
                        "Slice %s has no or multiple type entries, expected exactly one", slice.getId())));
            }
            final var sliceTypeCode = sliceTypes.get(0).getCode().orElse("(not set)");
            if (!sliceTypeCode.equals("Extension")) {
                throw logger.throwing(new GeneratorException(
                        String.format("Slice %s has type code %s, expected Extension", slice.getId(), sliceTypeCode)));
            }
            final var sliceTypeProfiles = sliceTypes.get(0).getProfile();
            if (sliceTypeProfiles.isEmpty()) {
                // This slice is an inline extension without a distinct type. We have to get the
                // Extension.extension.value[x] element to determine the type.
                final var valueNode = slice.getChild("value[x]")
                        .orElseThrow(() -> new GeneratorException(
                                String.format("Element Extension.value[x] is missing in element %s", slice.getId())));
                final var types =
                        this.structureDefinitionIntrospector.getTypeSpecifications(valueNode.getSnapshotElement());
                final var valueSet = getBindingValueSet(valueNode.getSnapshotElement());
                final var element = slice.getSnapshotElement();
                logger.debug("Generating attribute for inline extension element {}", slice.getId());
                information.addAttribute(TypeAttribute.builder()
                        .withFhirName(fhirName)
                        .withElementId(slice.getId())
                        .withMin(mapMinCardinality(element))
                        .withMax(mapMaxCardinality(element))
                        .withTypes(types)
                        .withExtensionUrl(Optional.of(fhirName))
                        .withExtension(true)
                        .withModifier(false)
                        .withDefinition(element.getDefinition())
                        .withShortDefinition(element.getShort())
                        .withBindingValueSet(valueSet)
                        .build());
            } else {
                // This slice is a nested extension with a distinct type.
                final var types = getTypesForExtension(slice.getSnapshotElement());
                final var element = slice.getSnapshotElement();
                final var typeProfile = element.getType().stream()
                        .flatMap(t -> t.getProfile().stream())
                        .map(canonical -> canonical.getValue().orElse(""))
                        .filter(s -> !Strings.isNullOrEmpty(s))
                        .findFirst();
                logger.debug("Generating attribute for nested extension element {}", slice.getId());
                information.addAttribute(TypeAttribute.builder()
                        .withFhirName(fhirName)
                        .withElementId(slice.getId())
                        .withMin(mapMinCardinality(element))
                        .withMax(mapMaxCardinality(element))
                        .withTypes(types)
                        .withExtensionUrl(typeProfile)
                        .withExtension(true)
                        .withModifier(false)
                        .withDefinition(element.getDefinition())
                        .withShortDefinition(element.getShort())
                        .build());
            }
        }
        logger.exit();
    }

    /**
     * Generates the attributes used to represent the elements of a custom
     * structure.
     * @throws FhirException
     */
    private void generateStructureAttributes(TypeInformation information, IElementTreeNode elementTreeNode)
            throws GeneratorException, FhirException {
        logger.entry(elementTreeNode);
        // For a custom structure, check the entire element tree. Note: We
        // only create attributes for the added nodes and extensions of the root node,
        // not for any extension or addition lower down in the tree.
        for (final var node : elementTreeNode.getChildren()) {
            final var fhirName = node.getLocalName();
            if (node.getBaseRelationship() == ElementBaseRelationship.ADDED) {
                final var types = this.structureDefinitionIntrospector.getTypeSpecifications(node.getSnapshotElement());
                final var valueSet = getBindingValueSet(node.getSnapshotElement());
                final var element = node.getSnapshotElement();
                logger.debug("Generating attribute for added element {}", node.getId());

                information.addAttribute(TypeAttribute.builder()
                        .withFhirName(fhirName)
                        .withElementId(node.getId())
                        .withMin(mapMinCardinality(element))
                        .withMax(mapMaxCardinality(element))
                        .withTypes(types)
                        .withExtension(false)
                        .withModifier(false)
                        .withDefinition(element.getDefinition())
                        .withShortDefinition(element.getShort())
                        .withBindingValueSet(valueSet)
                        .build());
            }
            final var isExtension = fhirName.equals("extension");
            final var isModifierExtension = fhirName.equals("modifierExtension");
            if (isExtension || isModifierExtension) {
                for (final var extensionNode : node.getSlices()) {
                    // do not generate new attributes for extensions that are already present in the parent class
                    if (extensionNode.getBaseRelationship() != ElementBaseRelationship.UNCHANGED) {
                        generateAttributeForExtension(information, isExtension, isModifierExtension, extensionNode);
                    }
                }
            }
        }
        logger.exit();
    }

    /**
     * Generates an attribute for an extension that was added to a structure.
     * @throws FhirException
     */
    private void generateAttributeForExtension(
            TypeInformation information,
            final boolean isExtension,
            final boolean isModifierExtension,
            final IElementTreeNode extensionNode)
            throws GeneratorException, FhirException {
        logger.entry(extensionNode);
        logger.debug(
                "Generating attribute for{}extension {}",
                (isModifierExtension ? " modifier " : " "),
                extensionNode.getId());
        final var fhirName = extensionNode.getSliceName().orElseThrow();
        final var element = extensionNode.getSnapshotElement();

        // the extension type is the profile of the type reference with type code
        // Extension
        final var extensionProfiles = element.getType().stream()
                .filter(type -> type.getCode().orElse("").equals("Extension"))
                .flatMap(type -> type.getProfile().stream())
                .map(canonical -> canonical.getValue().orElse(""))
                .filter(s -> !Strings.isNullOrEmpty(s))
                .toList();
        if (extensionProfiles.size() != 1) {
            throw logger.throwing(new GeneratorException(
                    String.format("Type of extension %s is not present or not unique", element.getId())));
        }

        final var types = getTypesForExtension(element);
        final var valueSet = getBindingValueSet(element);
        information.addAttribute(TypeAttribute.builder()
                .withFhirName(fhirName)
                .withElementId(extensionNode.getId())
                .withMin(mapMinCardinality(element))
                .withMax(mapMaxCardinality(element))
                .withTypes(types)
                .withExtension(isExtension || isModifierExtension)
                .withModifier(isModifierExtension)
                .withExtensionUrl(Optional.of(extensionProfiles.getFirst()))
                .withDefinition(element.getDefinition())
                .withShortDefinition(element.getShort())
                .withBindingValueSet(valueSet)
                .build());
        logger.exit();
    }

    /**
     * Maps the minimum cardinality from the FHIR value to an integer usable by the
     * generator.
     */
    private int mapMinCardinality(IElementDefinitionAccessor element) throws GeneratorException {
        logger.entry(element);
        final var min = element.getMin();
        if (min.isEmpty()) {
            throw logger.throwing(
                    new GeneratorException(String.format("min cardinality of element %s is missing", element.getId())));
        }
        return logger.exit(min.get());
    }

    /**
     * Maps the maximum cardinality from the FHIR value to an integer usable by the
     * generator.
     */
    private int mapMaxCardinality(IElementDefinitionAccessor element) throws GeneratorException {
        final var max = element.getMax();
        if (max.isEmpty()) {
            throw logger.throwing(
                    new GeneratorException(String.format("max cardinality of element %s is missing", element.getId())));
        }
        final var maxValue = max.get();
        if (maxValue.equals("*")) {
            return logger.exit(Child.MAX_UNLIMITED);
        }
        try {
            return logger.exit(Integer.parseInt(maxValue));
        } catch (final NumberFormatException e) {
            throw logger.throwing(new GeneratorException(
                    String.format("max cardinality of element %s is not * or a valid number", element.getId()), e));
        }
    }

    /**
     * Determines the value set reference to add to the generated attribute.
     */
    private Optional<String> getBindingValueSet(IElementDefinitionAccessor element) {
        logger.entry(element);
        Optional<String> result = Optional.empty();
        final var binding = element.getBinding();
        if (binding.isPresent()) {
            result = binding.get().getValueSet();
        }
        return logger.exit(result);
    }

    /**
     * Converts the FHIR type specification of an element definition for an extension element.
     * @throws FhirException
     */
    private ImmutableCollection<ITypeSpecification> getTypesForExtension(IElementDefinitionAccessor element)
            throws GeneratorException, FhirException {
        logger.entry(element);
        final var extensionURIs = getExtensionURIs(element);
        // For a simple extension with a single type, we have to incorporate the
        // extension into the top-level class.
        if (extensionURIs.size() == 1) {
            final var extensionAccessor = getExtensionAccessor(extensionURIs.getFirst());
            ExtensionType extensionType;
            try {
                extensionType = this.structureDefinitionIntrospector.getExtensionType(extensionAccessor);
            } catch (final FhirException e) {
                throw logger.throwing(
                        new GeneratorException(String.format(element.getId().orElse("(no ID)")), e));
            }
            if (extensionType == ExtensionType.SIMPLE) {
                // For a simple extension, the type of the value[x] element determines the
                // attribute type (because the generated attributes will be incorporated into
                // the top-level class).
                final var extensionElementTree = this.elementTreeBuilder.buildElementTree(extensionAccessor);
                final var valueNode = extensionElementTree
                        .getNode("Extension.value[x]")
                        .orElseThrow(() -> new GeneratorException(String.format(
                                "Element Extension.value[x] is missing in Extension %s",
                                extensionAccessor.getUrl().orElse("?"))));
                return logger.exit(
                        this.structureDefinitionIntrospector.getTypeSpecifications(valueNode.getSnapshotElement()));
            }
        }
        // For a complex extension or a simple extension with multiple profiles, we
        // return the original reference because the extension
        // has to be generated as a separate class.
        return logger.exit(this.structureDefinitionIntrospector.getTypeSpecifications(element));
    }

    /**
     * Retrieves an accessor to examine an extension element.
     */
    private IStructureDefinitionAccessor getExtensionAccessor(URI extensionURI) throws GeneratorException {
        logger.entry(extensionURI);
        final IBaseResource extensionResource;
        extensionResource = this.packageRegistry
                .getUniqueResource(FhirResourceType.STRUCTURE_DEFINITION, extensionURI)
                .orElseThrow(() -> logger.throwing(new GeneratorException(
                        String.format("Type of extension %s could not be located", extensionURI))));
        final var extensionAccessor = this.accessorProvider.provideStructureDefinitionAccessor(extensionResource);
        return logger.exit(extensionAccessor);
    }

    /**
     * Determines the profile URIs of an extension element.
     * @param element
     * @return
     * @throws GeneratorException
     */
    private List<URI> getExtensionURIs(IElementDefinitionAccessor element) throws GeneratorException {
        logger.entry(element);
        final var extensionProfiles = element.getType().stream()
                .filter(type -> type.getCode().orElse("").equals("Extension"))
                .flatMap(type -> type.getProfile().stream())
                .map(canonical -> canonical.getValue().orElse(""))
                .filter(s -> !Strings.isNullOrEmpty(s))
                .map(URI::create)
                .toList();
        if (extensionProfiles.isEmpty()) {
            throw logger.throwing(new GeneratorException(String.format(
                    "Extension %s does not have a profile set", element.getId().orElse("(no ID)"))));
        }
        return logger.exit(extensionProfiles);
    }

    /**
     * Generates the fixed/pattern value information.
     * @throws GeneratorException
     * @throws FhirException
     */
    private void generateFixedValues(TypeInformation newType, IElementTree elementTree)
            throws GeneratorException, FhirException {
        logger.entry(newType, elementTree);
        // We need a map of Element.id to the nested classes to route the fixed values
        // that belong to nested classes correctly.
        final var targetTypes = newType.getNestedTypes().stream()
                .map(nt -> (TypeInformation) nt)
                .collect(Collectors.toMap(nt -> nt.getElementId().orElseThrow(), nt -> nt));
        generateFixedValuesForType(newType, elementTree.getRootNode(), targetTypes);
        logger.exit();
    }

    /**
     * Generates the fixed/pattern values for either the top-level type or a nested
     * class
     *
     * @param newType     the target type to add the fixed values to
     * @param rootNode    the root node of the tree for the top-level structure or
     *                    the node that corresponds to the nested class
     * @param targetTypes a map of new target types to adress when encountering the
     *                    sub-tree
     * @throws GeneratorException
     * @throws FhirException
     */
    private void generateFixedValuesForType(
            TypeInformation newType, IElementTreeNode rootNode, Map<String, TypeInformation> targetTypes)
            throws GeneratorException, FhirException {
        logger.entry(newType, rootNode);
        logger.debug("Generating fixed values for element {}", rootNode.getId());
        for (final var childNode : rootNode.getChildren()) {
            generateFixedValuesForNode(childNode, targetTypes).forEach(newType::addFixedValue);
        }
        if (rootNode.hasSlices()) {
            // TODO #147 - include slices of regular elements in fixed/pattern value generation
            // TODO #148 - include extensions in fixed/pattern value generation
            logger.warn(
                    "Root element {} has slices - don't know how to handle these, ignoring them for now",
                    rootNode.getId());
        }
        logger.exit();
    }

    /**
     * Generates the fixed/pattern values for a node by either deferring to
     * {@link #generateFixedValuesForType(TypeInformation, IElementTreeNode, Map)}
     * for a new type or by creating a combined rule for the entire sub-tree of the
     * node.
     *
     * @param node the node to examine
     * @param targetTypes a map of new target types to address when encountering the
     *                    sub-tree
     * @return the fixed value description of the node, if needed
     * @throws GeneratorException
     * @throws FhirException
     */
    private ImmutableList<ITypeFixedValue> generateFixedValuesForNode(
            IElementTreeNode node, Map<String, TypeInformation> targetTypes) throws GeneratorException, FhirException {
        logger.entry(node);
        if (targetTypes.containsKey(node.getId())) {
            // start a new sub-tree for the nested class
            generateFixedValuesForType(targetTypes.get(node.getId()), node, targetTypes);
            return logger.exit(ImmutableList.of());
        } else {
            final var result = new ArrayList<ITypeFixedValue>();
            // collect the fixed values of the child nodes and if necessary create a
            // combined value for the entire node
            if (node.hasChildren()) {
                generateFixedValuesForChildNodes(node, targetTypes).ifPresent(result::add);
            }
            // check whether the node has a fixed value or
            // pattern itself, unless it is excluded from fixed value generation
            if (!isElementExcludedFromFixedValues(node)) {
                generateFixedValueForNodeValue(node).ifPresent(result::add);
            }
            if (node.hasSlices()) {
                // TODO #147 - include slices of regular elements in fixed/pattern value generation
                // TODO #148 - include extensions in fixed/pattern value generation
                logger.warn(
                        "Slices of element {} are currently not included in fixed / pattern value determination",
                        node.getId());
            }
            return logger.exit(ImmutableList.copyOf(result));
        }
    }

    /**
     * For a node that has children, generate the fixed values for the child nodes and if necessary combine them into one value.
     * @param node the node to examine
     * @param targetTypes a map of new target types to address when encountering the
     *                    sub-tree
     * @return the fixed value description of the node, if needed
     * @throws GeneratorException
     * @throws FhirException
     */
    private Optional<ITypeFixedValue> generateFixedValuesForChildNodes(
            IElementTreeNode node, Map<String, TypeInformation> targetTypes) throws GeneratorException, FhirException {
        logger.entry(node);
        final var snapshotElement = node.getSnapshotElement();
        final var childValues = new ArrayList<ITypeFixedValue>();
        for (final var child : node.getChildren()) {
            childValues.addAll(generateFixedValuesForNode(child, targetTypes));
        }
        if (!childValues.isEmpty()) {
            final var nodeValue = TypeFixedValue.builder()
                    .withElementId(snapshotElement.getId().orElseThrow())
                    .withElementPath(snapshotElement.getPath().orElseThrow())
                    .withLocalName(node.getLocalName())
                    .withTypes(this.structureDefinitionIntrospector.getTypeSpecifications(snapshotElement))
                    .build();
            childValues.forEach(nodeValue::addComponentValue);
            return logger.exit(Optional.of(nodeValue));
        }
        return logger.exit(Optional.empty());
    }

    /**
     * Determines whether a node is excluded from the fixed/pattern value generation.
     */
    private boolean isElementExcludedFromFixedValues(IElementTreeNode node) {
        logger.entry(node);
        final var elementId = node.getId();
        if (elementId.endsWith(".meta.profile") || elementId.equals("Extension.url")) {
            return logger.exit(true);
        }
        return logger.exit(false);
    }

    /**
     * Generates the value entry for a single node that has no children, if required.
     * @throws GeneratorException
     * @throws UnsupportedTypeException
     * @throws FhirException
     */
    private Optional<ITypeFixedValue> generateFixedValueForNodeValue(IElementTreeNode node)
            throws UnsupportedTypeException, GeneratorException, FhirException {
        logger.entry(node);
        final var snapshotElement = node.getSnapshotElement();
        // an ElementDefinition can have either a fixed value or a pattern, but not both
        final var nodeFixedOrPattern = snapshotElement.getFixed().or(snapshotElement::getPattern);
        if (nodeFixedOrPattern.isPresent()) {
            if (nodeFixedOrPattern.get() instanceof final IPrimitiveTypeAccessor primitiveAccessor) {
                // we can use the primitive type to generate the value directly
                return logger.exit(Optional.of(TypeFixedValue.builder()
                        .withElementId(snapshotElement.getId().orElseThrow())
                        .withElementPath(snapshotElement.getPath().orElseThrow())
                        .withLocalName(node.getLocalName())
                        .withTypes(this.structureDefinitionIntrospector.getTypeSpecifications(snapshotElement))
                        .withValue(primitiveAccessor.getStringifiedValue())
                        .build()));
            } else {
                // we have to examine the properties of the value and transfer them individually
                return logger.exit(generateFixedValueForProperties(
                        nodeFixedOrPattern.get(),
                        snapshotElement.getId().orElseThrow(),
                        snapshotElement.getPath().orElseThrow(),
                        node.getLocalName()));
            }
        }
        return logger.exit(Optional.empty());
    }

    /**
     * Generates the value entry for a (simple or complex) property container.
     * @throws GeneratorException
     * @throws FhirException
     */
    private Optional<ITypeFixedValue> generateFixedValueForProperties(
            final IBaseAccessor propertyContainer, String id, String path, String localName)
            throws GeneratorException, FhirException {
        logger.entry(propertyContainer);
        final var properties = propertyContainer.children().stream()
                .filter(IPropertyAccessor::hasValues)
                .toList();
        if (!properties.isEmpty()) {
            final var nodeValue = TypeFixedValue.builder()
                    .withElementId(id)
                    .withElementPath(path)
                    .withLocalName(localName)
                    .withTypes(this.structureDefinitionIntrospector.getTypeSpecifications(propertyContainer))
                    .build();
            for (final var property : properties) {
                if (property.getValues().size() > 1) {
                    logger.warn(
                            "Multiple values of property {} found - only first value will be used for code generation",
                            property.getName());
                }
                final var propertyId = id + "." + property.getName();
                final var propertyPath = path + "." + property.getName();
                final var propertyValue = property.getValues().get(0);
                if (propertyValue instanceof final IPrimitiveTypeAccessor primitiveValue) {
                    // we can use the primitive type to generate the value directly
                    final var stringifiedValue = primitiveValue.getStringifiedValue();
                    nodeValue.addComponentValue(TypeFixedValue.builder()
                            .withElementId(propertyId)
                            .withElementPath(propertyPath)
                            .withLocalName(property.getName())
                            .withTypes(this.structureDefinitionIntrospector.getTypeSpecifications(property))
                            .withValue(stringifiedValue)
                            .build());
                } else {
                    // we have to examine the sub-properties
                    final var subPropertyValue = generateFixedValueForProperties(
                            propertyValue, propertyId, propertyPath, property.getName());
                    if (subPropertyValue.isPresent()) {
                        nodeValue.addComponentValue(subPropertyValue.get());
                    } else {
                        logger.warn(
                                "Property {} has non-primitive value of type {} and did not provide any sub-properties and will be ignored",
                                property.getName(),
                                propertyValue.getClass().getCanonicalName());
                    }
                }
            }
            return logger.exit(Optional.of(nodeValue));
        }
        return logger.exit(Optional.empty());
    }

    /**
     * Attempts to extract the {@link ITypeSpecification} instances describing a StructureDefinition.
     * @param accessor the StructureDefinition to examine
     * @return the {@link ITypeSpecification} instances describing a StructureDefinition
     * @throws GeneratorException
     */
    private ImmutableCollection<ITypeSpecification> getTypeSpecifications(final IStructureDefinitionAccessor accessor)
            throws GeneratorException {
        final var canonicalURI = accessor.getUrl().orElse("(missing canonical URI");
        logger.entry(canonicalURI);
        ImmutableCollection<ITypeSpecification> typeSpecifications;
        try {
            typeSpecifications = this.structureDefinitionIntrospector.getTypeSpecifications(accessor);
        } catch (final FhirException e) {
            throw logger.throwing(new GeneratorException(
                    String.format("Unable to determine type specifications of type %s", canonicalURI), e));
        }
        return logger.exit(typeSpecifications);
    }

    /**
     * Attempts to extract the {@link ITypeSpecification} instances describing the target type of an ElementDefinition.
     * @param snapshotElement the snapshot of the ElementDefinition to examine
     * @return the {@link ITypeSpecification} instances describing the target type of an ElementDefinition
     * @throws GeneratorException
     */
    private ImmutableCollection<ITypeSpecification> getTypeSpecifications(
            final IElementDefinitionAccessor snapshotElement) throws GeneratorException {
        logger.entry(snapshotElement);
        final String elementId = snapshotElement.getId().orElse("(no ID)");
        ImmutableCollection<ITypeSpecification> typeSpecifications;
        try {
            typeSpecifications = this.structureDefinitionIntrospector.getTypeSpecifications(snapshotElement);
        } catch (final FhirException e) {
            throw logger.throwing(new GeneratorException(
                    String.format("Unable to determine type specifications of element1 %s", elementId), e));
        }
        return logger.exit(typeSpecifications);
    }
}
