/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.intermediate;

import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.accessors.implementations.ITypeSpecification;
import de.fhlintstone.fhir.ClassAnnotation;
import de.fhlintstone.process.config.NestedClassConfiguration;
import de.fhlintstone.process.config.StructureDefinitionClassConfiguration;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import lombok.Builder;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NonNull;
import lombok.extern.slf4j.XSlf4j;

/**
 * Default implementation of {@link ITypeInformation}.
 */
@EqualsAndHashCode
@Builder(setterPrefix = "with")
@XSlf4j
public final class TypeInformation implements ITypeInformation {

    @Getter
    @NonNull
    private final ImmutableCollection<ITypeSpecification> typeSpecifications;

    @Getter
    @NonNull
    private final URI canonicalURI;

    @Getter
    @NonNull
    private final String versionedCanonicalURI;

    @Getter
    @Builder.Default
    private final Optional<URI> parentCanonicalURI = Optional.empty();

    @Getter
    @Builder.Default
    private final Optional<String> parentVersionedCanonicalURI = Optional.empty();

    @Getter
    private final URI baseDefinition;

    @Getter
    @Builder.Default
    private final Optional<ITypeInformation> baseTypeInformation = Optional.empty();

    @Getter
    @Builder.Default
    private final Optional<String> elementId = Optional.empty();

    @Getter
    private final boolean generated;

    @Getter
    @Builder.Default
    private final Optional<TypeName> frameworkType = Optional.empty();

    @Getter
    private final String definitionName;

    @Getter
    @Builder.Default
    private final Optional<String> description = Optional.empty();

    @Getter
    @Builder.Default
    private final Optional<ClassAnnotation> classAnnotation = Optional.empty();

    @Getter
    @Builder.Default
    private final Optional<StructureDefinitionClassConfiguration> topLevelConfiguration = Optional.empty();

    @Getter
    @Builder.Default
    private final Optional<NestedClassConfiguration> nestedConfiguration = Optional.empty();

    private final List<ITypeAttribute> attributeList = new ArrayList<>();

    private final List<ITypeFixedValue> fixedValueList = new ArrayList<>();

    private final List<ITypeInformation> nestedTypeList = new ArrayList<>();

    @Override
    public ImmutableList<ITypeAttribute> getAttributes() {
        return ImmutableList.copyOf(this.attributeList);
    }

    @Override
    public ImmutableList<ITypeFixedValue> getFixedValues() {
        return ImmutableList.copyOf(this.fixedValueList);
    }

    @Override
    public ImmutableList<ITypeInformation> getNestedTypes() {
        return ImmutableList.copyOf(this.nestedTypeList);
    }

    /**
     * Adds an attribute to the type information.
     *
     * @param attribute the attribute to add
     */
    public void addAttribute(ITypeAttribute attribute) {
        this.attributeList.add(attribute);
    }

    /**
     * Adds a fixed value to the type information.
     *
     * @param value the value to add
     */
    public void addFixedValue(ITypeFixedValue value) {
        this.fixedValueList.add(value);
    }

    /**
     * Adds a nested type to the type information.
     *
     * @param nestedType the nested type to add
     */
    public void addNestedType(ITypeInformation nestedType) {
        this.nestedTypeList.add(nestedType);
    }

    @Override
    public String toString() {
        return "TypeInformation [definitionName=" + this.definitionName + ", versionedCanonicalURI="
                + this.versionedCanonicalURI + "]";
    }
}
