/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.intermediate;

import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.implementations.ITypeSpecification;
import de.fhlintstone.accessors.model.StringifiedValue;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import lombok.Builder;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.Singular;

/**
 * Default implementation of {@link ITypeFixedValue}.
 */
@EqualsAndHashCode
@Builder(setterPrefix = "with")
public final class TypeFixedValue implements ITypeFixedValue {

    @Getter
    private final String elementId;

    @Getter
    private final String elementPath;

    @Getter
    private final String localName;

    @Getter
    @Singular
    private final ImmutableCollection<ITypeSpecification> types;

    @Getter
    @Builder.Default
    private final Optional<StringifiedValue> value = Optional.empty();

    private final List<ITypeFixedValue> componentValueList = new ArrayList<>();

    @Override
    public boolean isComplex() {
        return !this.componentValueList.isEmpty();
    }

    @Override
    public ImmutableList<ITypeFixedValue> getComponentValues() {
        return ImmutableList.copyOf(this.componentValueList);
    }

    /**
     * Adds a component of a complex fixed value to the parent value.
     *
     * @param value the value to add
     */
    public void addComponentValue(ITypeFixedValue value) {
        this.componentValueList.add(value);
    }

    @Override
    public String toString() {
        if (this.value.isPresent()) {
            return "TypeFixedValue [elementID=" + this.elementId + ", value=" + this.value.get() + "]";
        } else {
            return "TypeFixedValue [elementID=" + this.elementId + ", values={ "
                    + this.componentValueList.stream().map(Object::toString).collect(Collectors.joining(", ")) + " }]";
        }
    }
}
