/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.intermediate;

import com.google.common.collect.ImmutableCollection;
import de.fhlintstone.accessors.implementations.IMappedType;
import de.fhlintstone.generator.GeneratorException;

/**
 * This component of the StructureDefinition generator is responsible for mapping FHIR types to the
 * Java type names that represent them, whether they be types supplied by the HAPI framework or types
 * generated using the controller. It also comprises a number of other type-related functions.
 */
public interface ITypeMapper {

    /**
     * Returns the Java type that represents the superclass of a given type. This can be a framework or a generated
     * type.
     *
     * @param typeInformation information about the type to be generated
     * @return the Java type that is used as a superclass of the generated type
     * @throws GeneratorException if the superclass cannot be determined
     */
    IMappedType determineSuperclass(ITypeInformation typeInformation) throws GeneratorException;

    /**
     * Returns the Java types that can be used to represent the attribute.
     *
     * Since the attribute might be a choice attribute with multiple types or the mapper might have to revert to the
     * type specification that can name multiple profiles, this method returns a list that may contain multiple types.
     *
     * @param typeInformation information about the surrounding type
     * @param sourceAttribute the attribute to map
     * @return the Java types that can be used to represent the attribute
     */
    ImmutableCollection<IMappedType> determineAttributeTypes(
            ITypeInformation typeInformation, ITypeAttribute sourceAttribute);

    /**
     * Determines the first (as in, most specialized) superclass of the type specified that is not a generated type but a
     * HAPI framework type. If the type specified as a parameter is a framework type, it is returned and no superclass
     * determination takes place. The type may refer to either a top-level or a nested class.
     *
     * @param typeInformation the type to examine
     * @return the first superclass that is not a generated type
     * @throws GeneratorException
     */
    IMappedType determineFrameworkSuperclass(ITypeInformation typeInformation) throws GeneratorException;

    /**
     * Determines the first (as in, most specialized) superclass of the attribute specified that is not a generated type
     * but a HAPI framework type. If the type of the attribute specified as a parameter is a framework type, it is
     * returned and no superclass determination takes place.
     *
     * Since the attribute might be a choice attribute with multiple types or the mapper might have to revert to the
     * type specification that can name multiple profiles, this method returns a list that may contain multiple types.
     *
     * @param typeInformation the type that contains the attribute
     * @param sourceAttribute the attribute to examine
     * @return the first superclass that is not a generated type
     * @throws GeneratorException
     */
    ImmutableCollection<IMappedType> determineFrameworkSuperclasses(
            ITypeInformation typeInformation, ITypeAttribute sourceAttribute) throws GeneratorException;

    /**
     * Determines whether the type of the element referred to will be derived from the HAPI PrimititveType. This version
     * of the method is used for the entire (top-level or nested) class that is generated.
     *
     * The results of this method are used to implement the equality checks because certain methods are only available
     * for PrimitiveTypes and Lists thereof. Note that for this reason this method has to return false for multi-type
     * attributes.
     *
     * @param typeInformation the type to examine
     * @return <code>true</code> if the type of the element will be derived from the HAPI PrimititveType
     * @throws GeneratorException
     */
    boolean isDerivedFromPrimitiveType(ITypeInformation typeInformation) throws GeneratorException;

    /**
     * Determines whether the type of the element referred to will be derived from the HAPI PrimititveType. This version
     * of the method is used for a single attribute of a generated class.
     *
     * The results of this method are used to implement the equality checks because certain methods are only available
     * for PrimitiveTypes and Lists thereof. Note that for this reason this method has to return false for multi-type
     * attributes.
     *
     * @param typeInformation the type that contains the attribute
     * @param attributeInformation the attribute to examine
     * @return <code>true</code> if the type of the element will be derived from the HAPI PrimititveType
     * @throws GeneratorException
     */
    boolean isDerivedFromPrimitiveType(ITypeInformation typeInformation, ITypeAttribute attributeInformation)
            throws GeneratorException;
}
