/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.intermediate;

import com.palantir.javapoet.TypeName;
import java.net.URI;
import java.util.Optional;

/**
 * This object stores the {@link ITypeInformation} instances that are generated by the {@link ITypeInformationBuilder}
 * and holds index information to allow access to the instances by various identifying attributes.
 */
public interface ITypeInformationStore {

    /**
     * Clears the store and removes all stored {@link ITypeInformation} instances and the corresponding index
     * information.
     */
    void clear();

    /**
     * Adds an {@link ITypeInformation} instance to the store.
     *
     * @param typeInformation the object to store
     * @throws IllegalArgumentException if the object violates one of the index unique constraints
     */
    void add(ITypeInformation typeInformation) throws IllegalArgumentException;

    /**
     * Retrieves an {@link ITypeInformation} instance as identified by its canonical URI.
     *
     * <b>Caution:</b> This method will only retrieve framework and top-level type information instances because the
     * canonical URIs for nested classes are not guaranteed to be unique.
     *
     * @param canonicalURI the canonical URI to search for
     * @return the {@link ITypeInformation} instance if it is contained in the store
     */
    Optional<ITypeInformation> getByCanonical(URI canonicalURI);

    /**
     * Retrieves an {@link ITypeInformation} instance as identified by the type name. The name might either be a
     * framework type name or the name of a generated type.
     *
     * @param typeName the type name
     * @return the {@link ITypeInformation} instance if it is contained in the store
     */
    Optional<ITypeInformation> getByType(TypeName typeName);
}
