/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.intermediate;

import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.accessors.implementations.ITypeSpecification;
import de.fhlintstone.fhir.ClassAnnotation;
import de.fhlintstone.fhir.dependencies.IDependencyGraph;
import de.fhlintstone.process.config.NestedClassConfiguration;
import de.fhlintstone.process.config.StructureDefinitionClassConfiguration;
import java.net.URI;
import java.util.Optional;

/**
 * A representation of a FHIR type that is used to generated Java code. This is generated from the
 * {@link IDependencyGraph} and incorporates information about the generated object that is required to refer to the
 * type in other dependent types.
 */
public interface ITypeInformation {

    /**
     * Returns the type specification that identifies the type (a type code and any number of profiles the type may
     * conform to.
     *
     * @return the type specification that identifies the type
     */
    ImmutableCollection<ITypeSpecification> getTypeSpecifications();

    /**
     * Returns the canonical URI of the type (whether nested or top-level).
     *
     * Note that a type might conform to multiple profiles. In that case, only the base type code canonical is returned
     * here. The full list of conforming profiles can be obtained via {@link #getTypeSpecifications()}.
     *
     * @return the canonical URI of the type
     */
    URI getCanonicalURI();

    /**
     * Returns the canonical URI of the type (whether nested or top-level), including the resource version if present.
     *
     * Note that a type might conform to multiple profiles. In that case, only the base type code canonical is returned
     * here. The full list of conforming profiles can be obtained via {@link #getTypeSpecifications()}.
     *
     * @return the canonical URI of the type including the resource version if present
     */
    String getVersionedCanonicalURI();

    /**
     * For nested types, returns the canonical URI of the parent top-level type. For top-level types, this method
     * returns an empty object.
     *
     * Note that a type might conform to multiple profiles. In that case, only the base type code canonical is returned
     * here. The full list of conforming profiles can be obtained via {@link #getTypeSpecifications()}.
     *
     * @return the canonical URI of the parent top-level type
     */
    Optional<URI> getParentCanonicalURI();

    /**
     * For nested types, returns the canonical URI of the parent top-level type, including the resource version if
     * present. For top-level types, this method returns an empty object.
     *
     * Note that a type might conform to multiple profiles. In that case, only the base type code canonical is returned
     * here. The full list of conforming profiles can be obtained via {@link #getTypeSpecifications()}.
     *
     * @return the canonical URI of the type including the resource version if present
     */
    Optional<String> getParentVersionedCanonicalURI();

    /**
     * Returns the URI of the base the StructureDefinition is derived from.
     *
     * @return the URI of the base the StructureDefinition is derived from
     */
    URI getBaseDefinition();

    /**
     * Returns the {@link ITypeInformation} instance that describes the base definition of the StructureDefinition.
     * May be empty for non-generated framework types.
     *
     * @see #getBaseDefinition()
     * @return the type information of the base definition
     */
    Optional<ITypeInformation> getBaseTypeInformation();

    /**
     * For nested types, this method returns the ID of the element in the parent type the nested type is used for.
     *
     * @return the ID of the element the type is used for
     */
    Optional<String> getElementId();

    /**
     * Returns <code>true</code> if the generator is supposed to produce code for this type, <code>false</code> for an
     * existing framework type.
     *
     * @return <code>true</code> if the generator is supposed to produce code for this type, <code>false</code> for an
     *         existing framework type
     */
    boolean isGenerated();

    /**
     * For framework (i.e. non-generated) types, this method returns the name of the HAPI FHIR type that implements the
     * StructureDefinition. Caution: For simple extensions, no class is generated (so {@link #isGenerated()} returns
     * <code>false</code>), but they do not have a superclass assigned either.
     *
     * @return the name of the HAPI FHIR type that implements the StructureDefinition, if this is a framework type
     */
    Optional<TypeName> getFrameworkType();

    /**
     * Returns the name of the underlying FHIR definition, e.g. Patient. For (anonymous) nested types, this returns the
     * type specified for the target element (e.g. HumanName if the nested type is built for Patient.name).
     *
     * @return the name of the underlying FHIR definition
     */
    String getDefinitionName();

    /**
     * Returns the description of the underlying FHIR definition. For (anonymous) nested types, this method currently
     * does not return anything.
     *
     * @return the description of the underlying FHIR definition
     */
    Optional<String> getDescription();

    /**
     * Determines the {@link ClassAnnotation} which either has been applied to the framework type or is to be added to
     * the generated type.
     *
     * @return the {@link ClassAnnotation}
     */
    // TODO #218 move getClassAnnotation() to the TypeInformationConverter
    Optional<ClassAnnotation> getClassAnnotation();

    /**
     * Returns the configuration entry for the top-level type, if present. For nested types, this method returns the
     * configuration entry of the containing top-level type.
     *
     * @return the configuration entry for the top-level type, if present
     */
    Optional<StructureDefinitionClassConfiguration> getTopLevelConfiguration();

    /**
     * Returns the configuration entry for the nested type, if present.
     *
     * @return the configuration entry for the nested type, if present
     */
    Optional<NestedClassConfiguration> getNestedConfiguration();

    /**
     * Returns the list of attributes that have to be generated for this type. Note that this list does <b>not</b>
     * include the inherited fields!
     *
     * @return the list of attributes that have to be generated for this type
     */
    ImmutableList<ITypeAttribute> getAttributes();

    /**
     * Returns the list of fixed and pattern values that have been defined in the profile of this type.
     *
     * @return the list of fixed and pattern values that have been defined in the profile of this type
     */
    ImmutableList<ITypeFixedValue> getFixedValues();

    /**
     * Returns the list of nested types that have to be generated for this type.
     *
     * @return the list of nested types that have to be generated for this type
     */
    ImmutableList<ITypeInformation> getNestedTypes();
}
