/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.intermediate;

import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.implementations.ITypeSpecification;
import de.fhlintstone.accessors.model.StringifiedValue;
import java.util.Optional;

/**
 * Describes a fixed or pattern value that is defined in the profile for an
 * attribute or rather an element.
 */
public interface ITypeFixedValue {

    /**
     * Determines whether this is a simple or a complex value. A simple value has an
     * ElementDefinition and a value. A complex value has an ElementDefinition
     * (describing the complex target field), a StructureDefinition describing the
     * target structure and a list (or rather tree) of component values.
     *
     * @return <code>true</code> if this is a complex value.
     */
    boolean isComplex();

    /**
     * Returns the value of ElementDefinition.id for the target element.
     *
     * @return the ID of the target element
     */
    String getElementId();

    /**
     * Returns the value of ElementDefinition.path for the target element.
     *
     * @return the path of the target element
     */
    String getElementPath();

    /**
     * Returns the "local" name of the wrapped element, i.e. the last part of the
     * element ID. Example: "profile" for the node wrapping "Patient.meta.profile".
     *
     * @return the "local" name of the wrapped element
     */
    String getLocalName();

    /**
     * Returns the type specification required to interpret or represent the
     * fixed value.
     *
     * @return the type specification
     * @see ITypeAttribute#getTypes()
     */
    ImmutableCollection<ITypeSpecification> getTypes();

    /**
     * Returns the value to set according to the profile, if this is a simple value.
     *
     * @return value to set according to the profile
     */
    Optional<StringifiedValue> getValue();

    /**
     * Returns the list of individual component values if this is a complex value.
     *
     * @return the list of individual component values.
     */
    ImmutableList<ITypeFixedValue> getComponentValues();
}
