/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.intermediate;

import ca.uhn.fhir.model.api.annotation.Binding;
import ca.uhn.fhir.model.api.annotation.Child;
import ca.uhn.fhir.model.api.annotation.Description;
import ca.uhn.fhir.model.api.annotation.Extension;
import com.google.common.collect.ImmutableCollection;
import de.fhlintstone.accessors.implementations.ITypeSpecification;
import java.util.Optional;

/**
 * Describes an attribute of a {@link ITypeInformation}. This may be an
 * extension, a modifier extension or an entirely new attribute.
 *
 * Note that the following annotation attributes are not (yet) covered by this
 * object:
 * <ul>
 * <li>{@link Child#order()}</li>
 * <li>{@link Child#summary()}</li>
 * <li>{@link Extension#definedLocally()}</li>
 * <li>{@link Description#example()}</li>
 * </ul>
 *
 */
public interface ITypeAttribute {

    /**
     * Returns the local name (the path element) of the FHIR element, e.g. lastUpdated for Patient.meta.lastUpdated. This can be
     * used to supply {@link Child#name()}.
     *
     * @return the local name of the FHIR element
     */
    String getFhirName();

    /**
     * Returns the full ID of the FHIR element definition, e.g. Patient.meta.lastUpdated.
     *
     * @return the ID of the FHIR element definition
     */
    String getElementId();

    /**
     * Returns the minimum cardinality. This can be used to supply
     * {@link Child#min()}.
     *
     * @return the minimum cardinality.
     */
    int getMin();

    /**
     * Returns the maximum cardinality. This can be used to supply
     * {@link Child#max()}. Use {@link Child#MAX_UNLIMITED} to denote an unlimited
     * number of occurrences.
     *
     * @return the maximum cardinality.
     */
    int getMax();

    /**
     * Returns the type specification as contained in the <a href=
     * "https://hl7.org/fhir/R4/elementdefinition-definitions.html#ElementDefinition.type">ElementDefinition.type</a>
     * substructure. Type specifications must be unique by type code. The type of
     * the element can be left blank in a differential constraint, in which case the
     * type is inherited from the resource. Abstract types are not permitted to
     * appear as a type when multiple types are listed. (I.e. Abstract types cannot
     * be part of a choice).
     *
     * @return the type specification
     */
    ImmutableCollection<ITypeSpecification> getTypes();

    /**
     * Returns <code>true</code> if the attribute describes an extension. Can be
     * used to trigger the generation of an {@link Extension} annotation. If the
     * attribute specifies an extension, {@link #getExtensionUrl()} must also return
     * a value.
     *
     * @return <code>true</code> if the attribute describes an extension
     */
    boolean isExtension();

    /**
     * Returns <code>true</code> if the attribute is a modifier or modifier
     * extension. Can be used to supply {@link Child#modifier()} and
     * {@link Extension#isModifier()}.
     *
     * @return <code>true</code> if the attribute is a modifier or modifier
     *         extension.
     */
    boolean isModifier();

    /**
     * Returns the URL identifying the extension, if the attribute is an extension.
     * Can be used to supply {@link Extension#url()}.
     *
     * @return the URL identifying the extension, if the attribute is an extension
     */
    Optional<String> getExtensionUrl();

    /**
     * Returns a definition of the attribute that can be used to supply
     * {@link Description#value()}. This corresponds to the FHIR element
     * ElementDefinition.definition.
     *
     * @return a description of the attribute
     */
    Optional<String> getDefinition();

    /**
     * Returns a short definition of the attribute that can be used to supply
     * {@link Description#shortDefinition()}. This corresponds to the FHIR element
     * ElementDefinition.short.
     *
     * @return a short definition of the attribute
     */
    Optional<String> getShortDefinition();

    /**
     * Returns the identifier of the ValueSet the element is bound to, if any. Can
     * be used to supply {@link Binding#valueSet()}.
     *
     * @return the identifier of the ValueSet the element is bound to, if any
     */
    Optional<String> getBindingValueSet();
}
