/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.code;

import com.google.common.collect.ImmutableList;
import com.palantir.javapoet.TypeName;
import java.util.Optional;
import org.hl7.fhir.r4.model.Base;

/**
 * Describes a fixed value (or the contents of a pattern) that is applied when
 * generating a new instance of a generated class with default values. This can
 * either be a simple value (transporting a single string or literal value) or a
 * complex value that contains a list of name-value assignments.
 */
public interface IFixedValueRule {

    /**
     * Returns the FHIR path of the attribute. This is used to generate the Javadoc
     * comment.
     *
     * @return the FHIR path of the attribute
     */
    String getPath();

    /**
     * Returns the FHIR name of the property. This has to be the name that can be
     * used to change the value via {@link Base#setProperty(String, Base)}.
     *
     * @return the FHIR name of the property
     */
    String getPropertyName();

    /**
     * Returns an optional prefix to use when generating variables. This is used to
     * prevent name collisions.
     *
     * @return the prefix used to generate variables, if set
     */
    Optional<String> getPrefix();

    /**
     * Returns the Java type of the property object to create.
     *
     * @return the Java type of the property
     */
    TypeName getPropertyType();

    /**
     * Returns the value to set, if the object is a simple value.
     *
     * @return the value to set
     */
    Optional<String> getValue();

    /**
     * Determines whether the value has to be generated as a literal value or a
     * string value.
     *
     * @return <code>true</code> to generate a literal value, <code>false</code> to
     *         generate a string value.
     */
    boolean isLiteral();

    /**
     * Returns the name-value assignments of a complex type, or an empty list for a
     * simple value.
     *
     * @return the name-value assignments of a complex type
     */
    ImmutableList<IFixedValueRule> getProperties();
}
