/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.code;

import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.palantir.javapoet.ClassName;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.fhir.ClassAnnotation;
import de.fhlintstone.generator.structuredefinition.IStructureDefinitionGenerator;
import java.util.Optional;

/**
 * The common class information transferred from the {@link IStructureDefinitionGenerator} to
 * the {@link ICodeEmitter}. This interface describes either a {@link ITopLevelClass} or a {@link INestedClass}.
 */
public interface IClassData {

    /**
     * Determines the name of the generated class.
     *
     * @return the name of the generated class
     */
    ClassName getClassName();

    /**
     * Determines the type of the super class this class is derived from.
     *
     * @return the type the class is derived from
     */
    TypeName getSuperclassName();

    /**
     * Determines whether an abstract class should be generated.
     *
     * @return <code>true</code> if an abstract class should be generated
     */
    boolean isAbstractClass();

    /**
     * Returns the name of the FHIR StructureDefinition.
     *
     * @return the name of the FHIR StructureDefinition
     */
    String getStructureDefinitionName();

    /**
     * Returns the canonical URI of the FHIR StructureDefinition.
     *
     * @return the canonical URI of the FHIR StructureDefinition
     */
    String getStructureDefinitionURL();

    /**
     * Determines whether the class to be generated is derived from PrimitiveType. If that is the case, additional
     * methods need to be overridden in order to fix a potential bug in the HAPI renderers that cause extensions to be
     * ignored.
     *
     * @return <code>true</code> if the class is derived from PrimitiveType
     */
    boolean isDerivedFromPrimitiveType();

    /**
     * Determines the class annotation mode.
     *
     * @return the class annotation mode
     * @see ClassAnnotation
     */
    Optional<ClassAnnotation> getClassAnnotation();

    /**
     * Determines the list of constructors to generate. Note that these objects currently only describe inherited
     * constructors, i.e. constructors that only call the constructor of the superclass with the same signature.
     *
     * @return the list of constructors to generate
     */
    ImmutableCollection<IClassConstructor> getConstructors();

    /**
     * Determines the list of class attributes.
     *
     * @return the list of attributes
     */
    ImmutableList<IClassAttribute> getAttributes();

    /**
     * Determines the list of rules to generate the fixed or pattern value supply method.
     *
     * @return the list of rules to generate the fixed or pattern value supply method
     */
    ImmutableList<IFixedValueRule> getFixedValueRules();
}
