/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.code;

import ca.uhn.fhir.model.api.annotation.Child;
import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.accessors.implementations.IMappedType;
import java.util.Optional;

/**
 * An attribute of a generated class that is described by {@link IClassData}.
 */
public interface IClassAttribute {

    /**
     * The name of the attribute as it should be generated in the class
     *
     * @return the name of the attribute
     */
    String getName();

    /**
     * The local name (the path element) of the FHIR element, e.g. lastUpdated for Patient.meta.lastUpdated.
     *
     * @return the local name of the FHIR element
     */
    String getFhirName();

    /**
     * The full ID of the FHIR element definition, e.g. Patient.meta.lastUpdated.
     *
     * @return the ID of the FHIR element definition
     */
    String getElementId();

    /**
     * Returns the URL of the extension if the attribute represents an extension.
     *
     * @return the URL of the extension if the attribute represents an extension
     * @see IClassAttribute#isExtension()
     * @see #isModifierExtension()
     */
    Optional<String> getExtensionUrl();

    /**
     * Determines whether the attribute represents an extension.
     *
     * @return <code>true</code> if the attribute represents an extension.
     * @see #isModifierExtension()
     */
    boolean isExtension();

    /**
     * Determines whether the attribute represents a modifier extension.
     *
     * @return <code>true</code> if the attribute represents a modifier extension.
     * @see #isExtension()
     */
    boolean isModifierExtension();

    /**
     * The description of the attribute which is used as a JavaDoc in the generated class
     *
     * @return the description of the attribute
     */
    Optional<String> getDescription();

    /**
     * The base type of the attribute. For single elements, this is the actual type of the attribute. For elements that
     * can occur more than once, this type returned by this method has to be be wrapped into a list by the code
     * generator.
     *
     * @see #isRepeating()
     * @return the attribute type
     */
    TypeName getAttributeType();

    /**
     * The actual Java type to use when generating or exposing the attribute. This is the type used to generate the
     * attribute as well as for some method return types. For elements that can occur more than once, this is the base
     * type wrapped into a generic list type.
     *
     * @see #isRepeating()
     * @return the actual attribute type
     */
    TypeName getActualType();

    /**
     * The actual Java type to use when creating an instance of the attribute. This is similar to the actual type, but
     * for elements that can occur more than once, it uses an instantiable list type.
     *
     * @see #isRepeating()
     * @see #getActualType()
     * @return the attribute creation type
     */
    TypeName getCreationType();

    /**
     * If the attribute represents a simple Element of a primitive type (i.e. a StringType, a BooleanType, an
     * IntegerType, ...), then the primitive type returned by this method is the Java type for which a the interface
     * methods are generated (like String, boolean, int, ...).
     *
     * @return the Java type that is contained in the primitive Element
     */
    Optional<TypeName> getPrimitiveType();

    /**
     * For multi-typed Elements, this collection contains the types that are specified by the StructureDefinition. For
     * other elements, it only contains the value returned by {@link #getAttributeType()}.
     *
     * @return the types that are specified by the StructureDefinition
     */
    ImmutableCollection<IMappedType> getMappedTypes();

    /**
     * The list of property types that are returned by the methods <code>getProperty</code> and
     * <code>getTypesForProperty</code>.
     *
     * @return the list of property types
     */
    ImmutableCollection<String> getPropertyTypes();

    /**
     * Determines the name of the method in the Base implementation that is used to perform a type-safe case of a Base
     * reference to the attribute type.
     *
     * @return the name of the casting method, or an empty value if no method exists
     */
    Optional<String> getBaseCastingMethod();

    /**
     * Determines whether this is a multi-type attribute (value[x]).
     *
     * @see #getMappedTypes()
     * @return <code>true</code> if this is a multi-type attribute
     */
    boolean isMultiType();

    /**
     * Determines whether this is a single attribute (ElementDefinition.max == 1) or if a List needs to be generated.
     *
     * @return <code>true</code> if a List has to be generated
     */
    boolean isRepeating();

    /**
     * Determines the kind of accessors that are generated for this attribute.
     *
     * @see AccessorGenerationMode
     * @return the kind of accessors that are generated for this attribute
     */
    AccessorGenerationMode getAccessorGenerationMode();

    /**
     * The minimum cardinality.
     *
     * @return the minimum cardinality.
     */
    int getMin();

    /**
     * The maximum cardinality. Use {@link Child#MAX_UNLIMITED} to denote an unlimited
     * number of occurrences.
     *
     * @return the maximum cardinality.
     */
    int getMax();

    /**
     * Determines whether the type of the attribute provided is derived from the HAPI PrimititveType. This is used to
     * implement the equality checks because certain methods are only available for PrimitiveTypes and Lists thereof.
     * Note that for this reason this method has to return false for multi-type attributes.
     *
     * @return <code>true</code> if the type of the attribute is derived from the HAPI PrimititveType
     * @see #getAttributeType()
     * @see de.fhlintstone.accessors.implementations.IFrameworkTypeLocator#isDerivedFromPrimitiveType(TypeName)
     */
    boolean isDerivedFromPrimitiveType();

    /**
     * A definition of the attribute. This corresponds to the FHIR element
     * ElementDefinition.definition.
     *
     * @return a description of the attribute
     */
    Optional<String> getDefinition();

    /**
     * The list of annotations which should be put on the generated attribute
     *
     * @return the annotations of the attribute
     */
    ImmutableList<IClassAttributeAnnotation> getAnnotations();
}
