/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.code;

import ca.uhn.fhir.model.api.annotation.Block;
import ca.uhn.fhir.model.api.annotation.DatatypeDef;
import ca.uhn.fhir.model.api.annotation.ResourceDef;
import ca.uhn.fhir.util.ElementUtil;
import com.google.common.collect.Lists;
import com.palantir.javapoet.AnnotationSpec;
import com.palantir.javapoet.ArrayTypeName;
import com.palantir.javapoet.ClassName;
import com.palantir.javapoet.FieldSpec;
import com.palantir.javapoet.JavaFile;
import com.palantir.javapoet.MethodSpec;
import com.palantir.javapoet.ParameterizedTypeName;
import com.palantir.javapoet.TypeName;
import com.palantir.javapoet.TypeSpec;
import com.palantir.javapoet.TypeSpec.Builder;
import de.fhlintstone.accessors.IAccessorProvider;
import de.fhlintstone.accessors.implementations.IFrameworkTypeLocator;
import de.fhlintstone.accessors.implementations.IMappedType;
import de.fhlintstone.generator.GeneratorException;
import de.fhlintstone.utilities.StringUtilities;
import jakarta.annotation.Generated;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Array;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Named;
import javax.lang.model.element.Modifier;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.extern.slf4j.XSlf4j;
import org.hl7.fhir.exceptions.FHIRException;
import org.hl7.fhir.instance.model.api.IBaseHasExtensions;
import org.hl7.fhir.instance.model.api.IBaseHasModifierExtensions;
import org.hl7.fhir.r4.model.Base;
import org.hl7.fhir.r4.model.Element;

/**
 * Default implementation of {@link ICodeEmitter}.
 */
@Named
@XSlf4j
@SuppressWarnings({"java:S3776", "java:S1192"}) // increased method complexity cannot really be avoided in this class
public class CodeEmitter implements ICodeEmitter {

    @Getter(AccessLevel.PRIVATE)
    private final IAccessorProvider accessorProvider;

    private final IFrameworkTypeLocator frameworkTypeLocator;

    /**
     * Constructor for dependency injection.
     *
     * @param accessorProvider     the {@link IAccessorProvider} to use
     * @param frameworkTypeLocator the {@link IFrameworkTypeLocator} to use
     */
    @Inject
    public CodeEmitter(IAccessorProvider accessorProvider, IFrameworkTypeLocator frameworkTypeLocator) {
        super();
        this.accessorProvider = accessorProvider;
        this.frameworkTypeLocator = frameworkTypeLocator;
    }

    @Override
    public void generate(ITopLevelClass topLevelClass) throws GeneratorException {
        logger.debug(
                "Generating code of class {} for StructureDefinition {}",
                topLevelClass.getClassName().canonicalName(),
                topLevelClass.getStructureDefinitionName());
        final var topLevelClassBuilder = initializeTopLevelBuilder(topLevelClass);
        addAttributes(topLevelClassBuilder, topLevelClass);
        generateConstructors(topLevelClassBuilder, topLevelClass);
        generateCommonMethods(topLevelClassBuilder, topLevelClass);
        generateMethodFhirType(topLevelClassBuilder, topLevelClass);
        for (final var nestedClass : topLevelClass.getNestedClasses()) {
            logger.debug(
                    "Adding nested class {} for element {}",
                    nestedClass.getClassName().canonicalName(),
                    nestedClass.getElementId());
            final var nestedClassBuilder = initializeNestedBuilder(topLevelClass, nestedClass);
            addAttributes(nestedClassBuilder, nestedClass);
            generateConstructors(nestedClassBuilder, nestedClass);
            generateCommonMethods(nestedClassBuilder, nestedClass);
            topLevelClassBuilder.addType(nestedClassBuilder.build());
        }

        writeClassToFile(topLevelClassBuilder, topLevelClass);
        logger.exit();
    }

    /**
     * Initializes the class builder for the top-level class.
     */
    private TypeSpec.Builder initializeTopLevelBuilder(ITopLevelClass classData) {
        logger.entry(classData);
        final var classBuilder = TypeSpec.classBuilder(classData.getClassName())
                .superclass(classData.getSuperclassName())
                .addModifiers(Modifier.PUBLIC)
                .addAnnotation(AnnotationSpec.builder(Generated.class)
                        .addMember("value", "$S", "fhlintstone")
                        .addMember("date", "$S", ZonedDateTime.now().format(DateTimeFormatter.ISO_INSTANT))
                        .build())
                .addJavadoc(
                        "Representation of the HL7 FHIR StructureDefinition $L.\n($L)\n",
                        classData.getStructureDefinitionName(),
                        classData.getStructureDefinitionURL());
        final Optional<String> structureDefinitionDescription = classData.getDescription();
        if (structureDefinitionDescription.isPresent()) {
            classBuilder.addJavadoc(structureDefinitionDescription.get());
        }
        if (classData.isAbstractClass()) {
            classBuilder.addModifiers(Modifier.ABSTRACT);
        }
        final var classAnnotation = classData.getClassAnnotation();
        if (classAnnotation.isPresent()) {
            switch (classAnnotation.get()) {
                case RESOURCE:
                    classBuilder.addAnnotation(AnnotationSpec.builder(ResourceDef.class)
                            .addMember("profile", "$S", classData.getStructureDefinitionURL())
                            .build());
                    break;
                case DATATYPE:
                    classBuilder.addAnnotation(AnnotationSpec.builder(DatatypeDef.class)
                            .addMember("name", "$S", classData.getStructureDefinitionName())
                            .build());
                    break;
                case BLOCK:
                    classBuilder.addAnnotation(
                            AnnotationSpec.builder(Block.class).build());
                    break;
                default:
                    // do nothing
                    break;
            }
        }
        return logger.exit(classBuilder);
    }

    /**
     * Initializes the class builder for a nested class.
     * @throws GeneratorException
     */
    private TypeSpec.Builder initializeNestedBuilder(ITopLevelClass topLevelClass, INestedClass nestedClass)
            throws GeneratorException {
        logger.entry(topLevelClass, nestedClass);
        final var classBuilder = TypeSpec.classBuilder(nestedClass.getClassName())
                .superclass(nestedClass.getSuperclassName())
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                .addJavadoc("Representation of the element $L.\n", nestedClass.getElementId());
        if (nestedClass.isAbstractClass()) {
            classBuilder.addModifiers(Modifier.ABSTRACT);
        }
        final var classAnnotation = nestedClass.getClassAnnotation();
        if (classAnnotation.isPresent()) {
            switch (classAnnotation.get()) {
                case RESOURCE ->
                    throw logger.throwing(new GeneratorException(
                            "Adding @ResourceDef annotations to nested classes is not supported at the moment."));
                case DATATYPE ->
                    classBuilder.addAnnotation(AnnotationSpec.builder(DatatypeDef.class)
                            .addMember("name", "$S", nestedClass.getStructureDefinitionName())
                            .build());
                case BLOCK ->
                    classBuilder.addAnnotation(
                            AnnotationSpec.builder(Block.class).build());
            }
        }
        return logger.exit(classBuilder);
    }

    /**
     * Generates the attributes added to the implementation class.
     */
    private void addAttributes(TypeSpec.Builder classBuilder, IClassData classData) {
        logger.entry();
        for (final var attribute : classData.getAttributes()) {
            logger.debug("Adding attribute {}", attribute.getName());
            final var attributeBuilder =
                    FieldSpec.builder(attribute.getActualType(), attribute.getName(), Modifier.PRIVATE);

            final var description = attribute.getDescription();
            if (description.isPresent()) {
                attributeBuilder.addJavadoc(description.get());
            }

            for (final var annotation : attribute.getAnnotations()) {
                final var annotationBuilder = AnnotationSpec.builder(annotation.getAnnotation());
                for (final var parameter : annotation.getParameters().entrySet()) {
                    final var value = parameter.getValue();
                    if (value.getClass().isArray()) {
                        final var values = (Object[]) value;
                        final String formatString = getFormatString(values);
                        annotationBuilder.addMember(parameter.getKey(), formatString, values);
                    } else {
                        final String formatString = getFormatString(value);
                        annotationBuilder.addMember(parameter.getKey(), formatString, value);
                    }
                }
                attributeBuilder.addAnnotation(annotationBuilder.build());
            }

            classBuilder.addField(attributeBuilder.build());
        }
        logger.exit();
    }

    /**
     * Generates the constructors, if required.
     */
    private void generateConstructors(final Builder classBuilder, IClassData classData) throws GeneratorException {
        logger.entry(classData);
        for (final var constructor : classData.getConstructors()) {
            final String superclassName = determineShortName(classData.getSuperclassName());

            // assemble the signature with the superclass name (e.g. "Enumeration#Enumeration(EnumFactory, Enum,
            // Element)")
            final var superclassSignatureBuilder = new StringBuilder();
            superclassSignatureBuilder.append(superclassName);
            superclassSignatureBuilder.append("#");
            superclassSignatureBuilder.append(superclassName);
            superclassSignatureBuilder.append("(");
            superclassSignatureBuilder.append(constructor.getParameters().stream()
                    .map(param -> determineShortName(param.type()))
                    .collect(Collectors.joining(", ")));
            superclassSignatureBuilder.append(")");
            final var signature = superclassSignatureBuilder.toString();
            logger.debug("Generating constructor {}", signature);

            // assemble the Javadoc text block
            final var javadocBuilder = new StringBuilder();
            javadocBuilder.append("Inherited constructor.\n");
            javadocBuilder.append(String.format("@see %s%n", signature));
            constructor.getParameters().stream()
                    .forEach(p -> javadocBuilder.append(
                            String.format("@param %s see constructor of superclass%n", p.name())));
            if (constructor.isDeprecated()) {
                javadocBuilder.append("@deprecated see constructor of superclass\n");
            }
            final var javadoc = javadocBuilder.toString();

            // assemble the super-constructor call
            final var superCallBuilder = new StringBuilder();
            superCallBuilder.append("super(");
            superCallBuilder.append(constructor.getParameters().stream()
                    .map(param -> param.name())
                    .collect(Collectors.joining(", ")));
            superCallBuilder.append(")");
            final var superCall = superCallBuilder.toString();

            // add the constructor
            final var constructorSpec = MethodSpec.constructorBuilder()
                    .addModifiers(Modifier.PUBLIC)
                    .addJavadoc(javadoc)
                    .addStatement(superCall);
            if (constructor.isDeprecated()) {
                constructorSpec.addAnnotation(Deprecated.class);
            }
            constructor.getParameters().forEach(p -> constructorSpec.addParameter(p.type(), p.name()));
            classBuilder.addMethod(constructorSpec.build());
        }
        logger.exit();
    }

    /**
     * Determines the short name of the type (attempting to abbreviate e.g. "org.hl7.fhir.r4.model.Address" to
     * "Address").
     *
     * @param typeName
     * @return the short name of the type
     */
    private String determineShortName(final TypeName typeName) {
        logger.entry(typeName);
        String result = typeName.toString();
        if (typeName instanceof final ClassName cn) {
            result = cn.simpleName();
        }
        return logger.exit(result);
    }

    /**
     * Generates the methods that are common to top-level and nested classes.
     */
    private void generateCommonMethods(final Builder classBuilder, IClassData classData) throws GeneratorException {
        logger.entry(classData);
        addAccessors(classBuilder, classData);
        generateMethodAddChild(classBuilder, classData);
        generateMethodCopy(classBuilder, classData);
        generateMethodEqualsDeep(classBuilder, classData);
        generateMethodEqualsShallow(classBuilder, classData);
        generateMethodIsEmpty(classBuilder, classData);
        generateMethodGetNamedProperty(classBuilder, classData);
        generateMethodGetProperty(classBuilder, classData);
        generateMethodGetTypesForProperty(classBuilder, classData);
        generateMethodListChildren(classBuilder, classData);
        generateMethodMakeProperty(classBuilder, classData);
        generateMethodRemoveChild(classBuilder, classData);
        generateMethodSetPropertyByHash(classBuilder, classData);
        generateMethodSetPropertyByName(classBuilder, classData);
        if (classData.isDerivedFromPrimitiveType()) {
            generateMethodGetExtension(classBuilder, classData);
            generateMethodHasExtension(classBuilder, classData);
            generateMethodGetModifierExtension(classBuilder, classData);
            generateMethodHasModifierExtension(classBuilder, classData);
        }
        generateMethodCreate(classBuilder, classData);
        logger.exit();
    }

    /**
     * Generates the accessor methods to expose the attributes.
     *
     * @throws GeneratorException
     */
    private void addAccessors(Builder classBuilder, IClassData classData) throws GeneratorException {
        logger.entry();
        for (final var attribute : classData.getAttributes()) {
            logger.debug(
                    "Using accessor generation mode {} for attribute {}",
                    attribute.getAccessorGenerationMode(),
                    attribute.getName());
            switch (attribute.getAccessorGenerationMode()) {
                case SINGLE_TYPE_NON_REPEATING_PRIMITIVE:
                    // addFooBar... - only for repeating elements
                    // getFooBar...
                    addGetterForSinglePrimitiveType(classBuilder, attribute);
                    addGetterForSingleElementType(classBuilder, attribute);
                    // hasFooBar...
                    addCheckerForSingle(classBuilder, attribute, "");
                    addCheckerForSingle(classBuilder, attribute, "Element"); // duplicate for consistency
                    // setFooBar...
                    addSetterForElement(classBuilder, classData, attribute, "Element");
                    addSetterForSinglePrimitiveType(classBuilder, classData, attribute);
                    break;
                case SINGLE_TYPE_NON_REPEATING_ELEMENT:
                    // addFooBar... - only for repeating elements
                    // getFooBar...
                    addGetterForSingleElementType(classBuilder, attribute);
                    // hasFooBar...
                    addCheckerForSingle(classBuilder, attribute, "");
                    // setFooBar...
                    addSetterForElement(classBuilder, classData, attribute, "");
                    break;
                case SINGLE_TYPE_REPEATING_PRIMITIVE:
                    // addFooBar...
                    addAdderForNewSingleType(classBuilder, attribute);
                    addAdderForExistingSinglePrimitiveType(classBuilder, classData, attribute);
                    // getFooBar...
                    addGetterForRepeatingElement(classBuilder, attribute);
                    addGetterForRepeatingElementFirst(classBuilder, attribute);
                    // hasFooBar...
                    addCheckerForRepeating(classBuilder, attribute);
                    // setFooBar...
                    addSetterForElement(classBuilder, classData, attribute, "");
                    break;
                case SINGLE_TYPE_REPEATING_ELEMENT:
                    // addFooBar...
                    addAdderForNewSingleType(classBuilder, attribute);
                    addAdderForExistingSingleElementType(classBuilder, classData, attribute);
                    // getFooBar...
                    addGetterForRepeatingElement(classBuilder, attribute);
                    addGetterForRepeatingElementFirst(classBuilder, attribute);
                    // hasFooBar...
                    addCheckerForRepeating(classBuilder, attribute);
                    // setFooBar...
                    addSetterForElement(classBuilder, classData, attribute, "");
                    break;
                case MULTI_TYPE_NON_REPEATING:
                    // addFooBar... - only for repeating elements
                    // getFooBar...
                    addGettersForMultiType(classBuilder, attribute);
                    // hasFooBar...
                    addCheckerForSingle(classBuilder, attribute, "");
                    addCheckersForMultiTypes(classBuilder, attribute);
                    // setFooBar...
                    addSettersForGenericTypeWithCheck(classBuilder, classData, attribute);
                    addSettersForMultiTypes(classBuilder, classData, attribute);
                    break;
                case MULTI_TYPE_REPEATING:
                    // addFooBar...
                    addAddersForNewMultiType(classBuilder, attribute);
                    addAddersForRepeatingMultiType(classBuilder, classData, attribute);
                    // getFooBar...
                    addGetterForRepeatingElement(classBuilder, attribute);
                    // hasFooBar...
                    addCheckerForRepeating(classBuilder, attribute);
                    // setFooBar...
                    addSetterForElement(classBuilder, classData, attribute, "");
                    break;
            }
        }
        logger.exit();
    }

    /**
     * Adds a no-argument adder method that creates a new element for a single-type
     * element.
     */
    private void addAdderForNewSingleType(Builder classBuilder, IClassAttribute attribute) {
        logger.entry(attribute);
        final var methodName = getAdderForNewSingleTypeName(attribute);
        logger.debug("Generating method {}", methodName);
        classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                .addModifiers(Modifier.PUBLIC)
                .returns(attribute.getAttributeType())
                .addJavadoc(
                        """
						Creates a new $1L entry and adds it to the list of existing entries.
						This method will create a new list if necessary.
						@return the newly created $1L entry
						""",
                        attribute.getElementId())
                .addStatement("$1T t = new $1T()", attribute.getAttributeType())
                .beginControlFlow("if (this.$N == null)", attribute.getName())
                .addStatement("this.$N = new $T()", attribute.getName(), attribute.getCreationType())
                .endControlFlow()
                .addStatement("this.$N.add(t)", attribute.getName())
                .addStatement("return t")
                .build());
        logger.exit();
    }

    /**
     * Determines the name of the no-argument adder method that creates a new
     * element for a single-type element.
     *
     * @see #addAdderForNewSingleType(Builder, IClassAttribute)
     */
    private String getAdderForNewSingleTypeName(IClassAttribute attribute) {
        // For attributes with a primitive type, the method is called addFooBarElement.
        // For element types, the method is called addFooBar.
        return attribute.getPrimitiveType().isPresent()
                ? getMethodName("add", attribute, "Element")
                : getMethodName("add", attribute);
    }

    /**
     * Adds a no-argument adder methods that create new elements for a
     * multi-type-type element.
     *
     * @throws GeneratorException
     */
    private void addAddersForNewMultiType(Builder classBuilder, IClassAttribute attribute) throws GeneratorException {
        logger.entry(attribute);
        for (final var modelType : attribute.getMappedTypes()) {
            final var mappedModelType = modelType.getType();
            final var methodName = getAdderForNewMultiTypeName(attribute, modelType);
            logger.debug("Generating method {}", methodName);
            classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                    .addModifiers(Modifier.PUBLIC)
                    .returns(attribute.getAttributeType())
                    .addJavadoc(
                            """
							Creates a new $1L entry of type $2T and adds it to the list of existing entries.
							This method will create a new list if necessary.
							@return the newly created $1L entry of type $2T
							""",
                            attribute.getElementId(),
                            mappedModelType)
                    .addStatement("$1T t = new $1T()", mappedModelType)
                    .beginControlFlow("if (this.$N == null)", attribute.getName())
                    .addStatement("this.$N = new $T()", attribute.getName(), attribute.getCreationType())
                    .endControlFlow()
                    .addStatement("this.$N.add(t)", attribute.getName())
                    .addStatement("return t")
                    .build());
        }
        logger.exit();
    }

    /**
     * Determines the name of the no-argument adder method that creates a new
     * element for a single-type element.
     *
     * @throws GeneratorException
     * @see #addAdderForNewSingleType(Builder, IClassAttribute)
     */
    private String getAdderForNewMultiTypeName(IClassAttribute attribute, IMappedType modelType) {
        final var typeCode = modelType.getTypeCode();
        return getMethodName("add", attribute, typeCode);
    }

    /**
     * Adds an adder method that adds an existing element for a single-type element
     * with a primitive type.
     */
    private void addAdderForExistingSinglePrimitiveType(
            Builder classBuilder, IClassData classData, IClassAttribute attribute) {
        logger.entry(attribute);
        final var methodName = getMethodName("add", attribute);
        logger.debug("Generating method {}", methodName);
        classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                .addModifiers(Modifier.PUBLIC)
                .returns(classData.getClassName())
                .addJavadoc(
                        """
						Adds an existing $1L entry to the list of existing entries.
						This method will create a new list if necessary.
						@param value the existing value to add to the element $1L
						@return the parent $2T object to allow for method chaining
						""",
                        attribute.getElementId(),
                        classData.getClassName())
                .addParameter(attribute.getPrimitiveType().orElseThrow(), "value")
                .addStatement("$1T t = new $1T()", attribute.getAttributeType())
                .addStatement("t.setValue(value)")
                .beginControlFlow("if (this.$N == null)", attribute.getName())
                .addStatement("this.$N = new $T()", attribute.getName(), attribute.getCreationType())
                .endControlFlow()
                .addStatement("this.$N.add(t)", attribute.getName())
                .addStatement("return this")
                .build());
        logger.exit();
    }

    /**
     * Adds an adder method that adds an existing element for a single-type element.
     */
    private void addAdderForExistingSingleElementType(
            Builder classBuilder, IClassData classData, IClassAttribute attribute) {
        logger.entry(attribute);
        final var methodName = getMethodName("add", attribute);
        logger.debug("Generating method {}", methodName);
        classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                .addModifiers(Modifier.PUBLIC)
                .returns(classData.getClassName())
                .addParameter(attribute.getAttributeType(), "value")
                .addJavadoc(
                        """
						Adds an existing $1L entry to the list of existing entries.
						This method will create a new list if necessary.
						@param value the existing value to add to the element $1L
						@return the parent $2T object to allow for method chaining
						""",
                        attribute.getElementId(),
                        classData.getClassName())
                .beginControlFlow("if (value == null)")
                .addStatement("return this")
                .endControlFlow()
                .beginControlFlow("if (this.$N == null)", attribute.getName())
                .addStatement("this.$N = new $T()", attribute.getName(), attribute.getCreationType())
                .endControlFlow()
                .addStatement("this.$N.add(value)", attribute.getName())
                .addStatement("return this")
                .build());
        logger.exit();
    }

    /**
     * Adds an adder method for each type of a multi-typed repeating element.
     *
     * @throws GeneratorException
     */
    private void addAddersForRepeatingMultiType(Builder classBuilder, IClassData classData, IClassAttribute attribute)
            throws GeneratorException {
        logger.entry(attribute);
        for (final var modelType : attribute.getMappedTypes()) {
            final var mappedModelType = modelType.getType();
            if (!(mappedModelType instanceof final ClassName modelClass)) {
                throw new GeneratorException("Model type must refer to a class");
            }
            final var methodName = getMethodName("add", attribute);
            logger.debug("Generating method {}", methodName);
            classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                    .addModifiers(Modifier.PUBLIC)
                    .returns(classData.getClassName())
                    .addParameter(modelClass, "value")
                    .addJavadoc(
                            """
							Adds an existing $1L entry of type $2L to the list of existing entries.
							This method will create a new list if necessary.
							@param value the existing $2L value to add to the element $1L
							@return the parent $3T object to allow for method chaining
							""",
                            attribute.getElementId(),
                            modelClass.simpleName(),
                            classData.getClassName())
                    .beginControlFlow("if (value == null)")
                    .addStatement("return this")
                    .endControlFlow()
                    .beginControlFlow("if (this.$N == null)", attribute.getName())
                    .addStatement("this.$N = new $T()", attribute.getName(), attribute.getCreationType())
                    .endControlFlow()
                    .addStatement("this.$N.add(value)", attribute.getName())
                    .addStatement("return this")
                    .build());
        }
        logger.exit();
    }

    /**
     * Adds a getter for a single type non-repeating attribute that has a primitive
     * value assigned.
     */
    private void addGetterForSinglePrimitiveType(Builder classBuilder, IClassAttribute attribute) {
        logger.entry(attribute);
        final var methodName = getGetterForGenericMultiTypeName(attribute);
        logger.debug("Generating method {}", methodName);
        classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                .addModifiers(Modifier.PUBLIC)
                .returns(attribute.getPrimitiveType().orElseThrow())
                .addJavadoc(
                        """
						Retrieves the value of the element $1L.
						This method will return <code>null</code> if the element is empty.
						@return the value of the element $1L
						""",
                        attribute.getElementId())
                .addStatement(
                        "return this.$1N == null || this.$1N.isEmpty() ? null : this.$1N.getValue()",
                        attribute.getName())
                .build());
        logger.exit();
    }

    /**
     * Adds a getter for a single type non-repeating attribute that has a primitive
     * value assigned.
     */
    private void addGetterForSingleElementType(Builder classBuilder, IClassAttribute attribute) {
        logger.entry(attribute);
        final var configurationClass = this.frameworkTypeLocator.getConfigurationType();
        final var methodName = getGetterForSingleElementTypeName(attribute);
        logger.debug("Generating method {}", methodName);
        classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                .addModifiers(Modifier.PUBLIC)
                .returns(attribute.getAttributeType())
                .addException(FHIRException.class)
                .addJavadoc(
                        """
						Retrieves the value of the element $1L.
						This method will automatically create an entry or raise an error according to the
						settings provided by $2T.
						@return the value of the element $1L
						@throws $3T if the element $1L is not yet initialized and auto-creation is disabled.
						""",
                        attribute.getElementId(),
                        configurationClass,
                        ClassName.get(FHIRException.class))
                .beginControlFlow("if (this.$N == null)", attribute.getName())
                .beginControlFlow("if ($T.errorOnAutoCreate())", configurationClass)
                .addStatement(
                        "throw new $T($S)",
                        ClassName.get(FHIRException.class),
                        String.format("Unable to auto-create element %s", attribute.getElementId()))
                .nextControlFlow("else if ($T.doAutoCreate())", configurationClass)
                .addStatement("this.$N = new $T()", attribute.getName(), attribute.getAttributeType())
                .endControlFlow()
                .endControlFlow()
                .addStatement("return this.$N", attribute.getName())
                .build());
        logger.exit();
    }

    /**
     * Determines the name of a getter for a single type non-repeating attribute
     * that has a primitive value assigned.
     */
    private String getGetterForSingleElementTypeName(IClassAttribute attribute) {
        logger.entry(attribute);
        // For attributes with a primitive type, the method is called getFooBarElement.
        // For element types, the method is called getFooBar.
        return logger.exit(
                attribute.getPrimitiveType().isPresent()
                        ? getMethodName("get", attribute, "Element")
                        : getGetterForGenericMultiTypeName(attribute));
    }

    /**
     * Adds a getter for a repeating-attribute (both single and multi-type variant).
     */
    private void addGetterForRepeatingElement(Builder classBuilder, IClassAttribute attribute) {
        logger.entry(attribute);
        final var methodName = getGetterForGenericMultiTypeName(attribute);
        logger.debug("Generating method {}", methodName);
        classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                .addModifiers(Modifier.PUBLIC)
                .returns(attribute.getActualType())
                .addJavadoc(
                        """
						Retrieves the list of entries of the element $1L.
						This method will create a new empty list if necessary.
						@return the list of entries of the element $1L
						""",
                        attribute.getElementId())
                .beginControlFlow("if (this.$N == null)", attribute.getName())
                .addStatement("this.$N = new $T()", attribute.getName(), attribute.getCreationType())
                .endControlFlow()
                .addStatement("return this.$N", attribute.getName())
                .build());
        logger.exit();
    }

    /**
     * Adds a getter for the first instance of a repeating-attribute (single-type
     * variant only!).
     */
    private void addGetterForRepeatingElementFirst(Builder classBuilder, IClassAttribute attribute) {
        logger.entry(attribute);
        final var listGetterName = getGetterForGenericMultiTypeName(attribute);
        // For attributes with a primitive type, the method is called addFooBarElement.
        // For element types, the method is called adFooBar.
        final var listAdderName = attribute.getPrimitiveType().isPresent()
                ? getMethodName("add", attribute, "Element")
                : getMethodName("add", attribute);
        final var methodName = getMethodName("get", attribute, "FirstRep");
        logger.debug("Generating method {}", methodName);
        classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                .addModifiers(Modifier.PUBLIC)
                .returns(attribute.getAttributeType())
                .addJavadoc(
                        """
						Retrieves the first entry in the list of entries of the element $1L.
						This method will create a new list if necessary and will also create
						a new $1L element if the list is empty.
						@return the first entry in the list of $1L entries
						""",
                        attribute.getElementId())
                .beginControlFlow("if (this.$N().isEmpty())", listGetterName)
                .addStatement("$N()", listAdderName)
                .endControlFlow()
                .addStatement("return $N().get(0)", listGetterName)
                .build());
        logger.exit();
    }

    /**
     * Adds the generic getter and the type-specific getters for a non-repeating
     * multi-type element.
     *
     * @throws GeneratorException
     */
    private void addGettersForMultiType(Builder classBuilder, IClassAttribute attribute) throws GeneratorException {
        logger.entry(attribute);

        // generate a generic getter, returning only a Type, without auto-creation
        final var methodName = getGetterForGenericMultiTypeName(attribute);
        logger.debug("Generating method {}", methodName);
        classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                .addModifiers(Modifier.PUBLIC)
                .returns(this.frameworkTypeLocator.getGenericType())
                .addJavadoc("@return the value of the field $L", attribute.getElementId())
                .addStatement("return this.$N", attribute.getName())
                .build());

        // generate a type-specific getter for each type
        for (final var modelType : attribute.getMappedTypes()) {
            final var mappedModelType = modelType.getType();
            if (!(mappedModelType instanceof final ClassName modelClass)) {
                throw new GeneratorException("Model type must refer to a class");
            }
            final var methodName2 = getMethodName("get", attribute, modelClass.simpleName());
            logger.debug("Generating method {}", methodName2);
            classBuilder.addMethod(MethodSpec.methodBuilder(methodName2)
                    .addModifiers(Modifier.PUBLIC)
                    .returns(modelClass)
                    .addException(FHIRException.class)
                    .addJavadoc(
                            """
							Retrieves the value of the element $1L as type $2T.
							This method will create a instance of $2T if necessary.
							@return the value of the field $1L as a $2T
							@throws $3T if the element $1L already contains an instance of a different type
							""",
                            attribute.getElementId(),
                            modelClass,
                            ClassName.get(FHIRException.class))
                    .beginControlFlow("if (this.$N == null)", attribute.getName())
                    .addStatement("this.$N = new $T()", attribute.getName(), mappedModelType)
                    .endControlFlow()
                    .beginControlFlow("if (!(this.$N instanceof $T))", attribute.getName(), mappedModelType)
                    .addStatement(
                            "throw new $T(String.format($S, this.$N.getClass().getName()))",
                            ClassName.get(FHIRException.class),
                            String.format(
                                    "Type mismatch: The type %s was expected, but %%s was encountered",
                                    modelClass.simpleName()),
                            attribute.getName())
                    .endControlFlow()
                    .addStatement("return ($T) this.$N", mappedModelType, attribute.getName())
                    .build());
        }

        logger.exit();
    }

    /**
     * Determines the name for the generic getter for a non-repeating multi-type
     * element.
     */
    private String getGetterForGenericMultiTypeName(IClassAttribute attribute) {
        return getMethodName("get", attribute);
    }

    /**
     * Adds an existence check for a single-occurence value (which may be a single
     * or multi type element, we don't care at this point).
     */
    private void addCheckerForSingle(Builder classBuilder, IClassAttribute attribute, String suffix) {
        logger.entry(attribute);
        final var methodName = getMethodName("has", attribute, suffix);
        logger.debug("Generating method {}", methodName);
        classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                .addModifiers(Modifier.PUBLIC)
                .returns(TypeName.BOOLEAN)
                .addJavadoc(
                        """
						Determines whether the element $1L is present and contains a value.
						@return <code>true</code> if the element $1L is present and contains a value
						""",
                        attribute.getElementId())
                .addStatement("return this.$1N != null && !this.$1N.isEmpty()", attribute.getName())
                .build());
        logger.exit();
    }

    /**
     * Adds an existence check for a single-occurence value (which may be a single
     * or multi type element, we don't care at this point).
     */
    private void addCheckerForRepeating(Builder classBuilder, IClassAttribute attribute) {
        logger.entry(attribute);
        final var methodName = getMethodName("has", attribute);
        logger.debug("Generating method {}", methodName);
        classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                .addModifiers(Modifier.PUBLIC)
                .returns(TypeName.BOOLEAN)
                .addJavadoc(
                        """
						Determines whether the element $1L is present and contains at least one value.
						This method will return <code>false</code> if the list only contains empty items.
						@return <code>true</code> if the element $1L is present and contains a value
						""",
                        attribute.getElementId())
                .beginControlFlow("if (this.$N == null)", attribute.getName())
                .addStatement("return false")
                .endControlFlow()
                .beginControlFlow("for (var item : this.$N)", attribute.getName())
                .beginControlFlow("if (!item.isEmpty())")
                .addStatement("return true")
                .endControlFlow()
                .endControlFlow()
                .addStatement("return false")
                .build());
        logger.exit();
    }

    /**
     * Adds type checks for a single-occurrence multi-type field. This method does
     * NOT check the actual attribute contents to keep consistent with the HAPI
     * implementations.
     *
     * @throws GeneratorException
     */
    private void addCheckersForMultiTypes(Builder classBuilder, IClassAttribute attribute) throws GeneratorException {
        logger.entry(attribute);
        for (final var modelType : attribute.getMappedTypes()) {
            final var mappedModelType = modelType.getType();
            if (!(mappedModelType instanceof final ClassName modelClass)) {
                throw new GeneratorException("Model type must refer to a class");
            }
            // see also https://github.com/hapifhir/org.hl7.fhir.core/issues/1953
            final var methodName = getMethodName("has", attribute, modelClass.simpleName());
            logger.debug("Generating method {}", methodName);
            classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                    .addModifiers(Modifier.PUBLIC)
                    .returns(TypeName.BOOLEAN)
                    .addJavadoc(
                            """
							Determines whether the element $1L is present and of type $2T.
							Note that this method will return <code>true</code> even if the actual
							value is empty; this is in accordance with the behaviour of the HAPI
							implementations.
							@return <code>true</code> if the element $1L is present and of type $2T
							""",
                            attribute.getElementId(),
                            modelClass)
                    // null check omitted to keep SonarQube silent (or rather because it's
                    // superfluous)
                    .addStatement("return this.$1N instanceof $2T", attribute.getName(), modelClass)
                    .build());
        }
        logger.exit();
    }

    /**
     * Adds a setter to replace the entire attribute.
     */
    private void addSetterForElement(
            Builder classBuilder, IClassData classData, IClassAttribute attribute, String suffix) {
        logger.entry(attribute);
        final var methodName = getMethodName("set", attribute, suffix);
        logger.debug("Generating method {}", methodName);
        classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                .addModifiers(Modifier.PUBLIC)
                .returns(classData.getClassName())
                .addParameter(attribute.getActualType(), "value")
                .addJavadoc(
                        """
						Changes the value of the element $1L.
						@param value the new value of the element $1L
						@return the parent $2T object to allow for method chaining
						""",
                        attribute.getElementId(),
                        classData.getClassName())
                .addStatement("this.$N = value", attribute.getName())
                .addStatement("return this")
                .build());
        logger.exit();
    }

    /**
     * Adds a setter taking a primitive type, replacing the entire attribute.
     */
    private void addSetterForSinglePrimitiveType(
            Builder classBuilder, IClassData classData, IClassAttribute attribute) {
        logger.entry(attribute);
        final var methodName = getMethodName("set", attribute);
        logger.debug("Generating method {}", methodName);
        classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                .addModifiers(Modifier.PUBLIC)
                .returns(classData.getClassName())
                .addParameter(attribute.getPrimitiveType().orElseThrow(), "value")
                .addJavadoc(
                        """
						Changes the value of the element $1L.
						This method will create a new $2T instance if required and transfer the value.
						@param value the new value of the element $1L
						@return the parent $3T object to allow for method chaining
						""",
                        attribute.getElementId(),
                        attribute.getAttributeType(),
                        classData.getClassName())
                .beginControlFlow("if (this.$N == null)", attribute.getName())
                .addStatement("this.$N = new $T()", attribute.getName(), attribute.getAttributeType())
                .endControlFlow()
                .addStatement("this.$N.setValue(value)", attribute.getName())
                .addStatement("return this")
                .build());
        logger.exit();
    }

    /**
     * Adds a setter for each supported type of a multi-type element.
     *
     * @throws GeneratorException
     */
    private void addSettersForMultiTypes(Builder classBuilder, IClassData classData, IClassAttribute attribute)
            throws GeneratorException {
        logger.entry(attribute);
        for (final var modelType : attribute.getMappedTypes()) {
            final var mappedModelType = modelType.getType();
            final var methodName = getMethodName("set", attribute);
            logger.debug("Generating method {}", methodName);
            classBuilder.addMethod(MethodSpec.methodBuilder(methodName)
                    .addModifiers(Modifier.PUBLIC)
                    .returns(classData.getClassName())
                    .addParameter(mappedModelType, "value")
                    .addJavadoc(
                            """
							Changes the value of the element $1L.
							@param value the new value of the element $1L of type $2T
							@return the parent $3T object to allow for method chaining
							""",
                            attribute.getElementId(),
                            mappedModelType,
                            classData.getClassName())
                    .addStatement("this.$N = value", attribute.getName())
                    .addStatement("return this")
                    .build());
        }
        logger.exit();
    }

    /**
     * Adds a generic setter for a multi-type element that checks the type of the
     * parameter.
     *
     * @throws GeneratorException
     */
    private void addSettersForGenericTypeWithCheck(
            Builder classBuilder, IClassData classData, IClassAttribute attribute) throws GeneratorException {
        logger.entry(attribute);
        final var methodName = getMethodName("set", attribute);
        logger.debug("Generating method {}", methodName);
        final var method = MethodSpec.methodBuilder(methodName)
                .addModifiers(Modifier.PUBLIC)
                .returns(classData.getClassName())
                .addParameter(attribute.getAttributeType(), "value")
                .addException(FHIRException.class)
                .addJavadoc(
                        """
					Changes the value of the element $1L. This method accepts a generic parameter,
					but performs a check to only allow compatible types.
					@param value the new value of the element $1L
					@return the parent $2T object to allow for method chaining
					@throws $3T if the value is of an unsupported type
					""",
                        attribute.getElementId(),
                        classData.getClassName(),
                        ClassName.get(FHIRException.class));
        for (final var modelType : attribute.getMappedTypes()) {
            final var mappedModelType = modelType.getType();
            method.beginControlFlow("if (value instanceof $T)", mappedModelType)
                    .addStatement("this.$N = value", attribute.getName())
                    .addStatement("return this")
                    .endControlFlow();
        }
        classBuilder.addMethod(method.addStatement(
                        "throw new $1T(String.format($2S, value.getClass().getName()))",
                        ClassName.get(FHIRException.class),
                        String.format("Unsupported type %%s for element %s", attribute.getElementId()))
                .build());
        logger.exit();
    }

    /**
     * Generates the override of {@link Base#addChild(String)} if necessary.
     */
    private void generateMethodAddChild(Builder classBuilder, IClassData classData) throws GeneratorException {
        logger.entry(classBuilder, classData);
        // only override addChild(String) if there are additional attributes to cover
        final var attributes = classData.getAttributes();
        if (!attributes.isEmpty()) {
            final var baseType = this.frameworkTypeLocator.getBaseType();
            logger.debug("Generating method addChild");
            final var addChildMethod = MethodSpec.methodBuilder("addChild")
                    .addModifiers(Modifier.PUBLIC)
                    .returns(baseType)
                    .addParameter(String.class, "name")
                    .addException(FHIRException.class)
                    .addJavadoc("@see $1T#addChild($2T)", baseType, String.class);
            for (final var attribute : attributes) {
                if (attribute.isMultiType()) {
                    // multi-type attribute: generate a branch for each type
                    for (final var modelType : attribute.getMappedTypes()) {
                        final var typedFhirName =
                                attribute.getFhirName() + StringUtilities.toFirstUpper(modelType.getTypeCode());
                        addChildMethod.beginControlFlow("if (name.equals($S))", typedFhirName);
                        if (attribute.isRepeating()) {
                            // repeating multi-type attribute
                            addChildMethod.addStatement(
                                    "return $N()", getAdderForNewMultiTypeName(attribute, modelType));
                        } else {
                            // non-repeating multi-type attribute
                            addChildMethod.addStatement(
                                    "throw new $T($S)",
                                    ClassName.get(FHIRException.class),
                                    String.format(
                                            "Cannot call addChild on singleton property %s", attribute.getElementId()));
                        }
                        addChildMethod.endControlFlow();
                    }
                } else {
                    addChildMethod.beginControlFlow("if (name.equals($S))", attribute.getFhirName());
                    if (attribute.isRepeating()) {
                        // repeating single-type attribute
                        addChildMethod.addStatement("return $N()", getAdderForNewSingleTypeName(attribute));
                    } else {
                        // non-repeating single-type attribute
                        addChildMethod.addStatement(
                                "throw new $T($S)",
                                ClassName.get(FHIRException.class),
                                String.format(
                                        "Cannot call addChild on singleton property %s", attribute.getElementId()));
                    }
                    addChildMethod.endControlFlow();
                }
            }
            addChildMethod.addStatement("return super.addChild(name)");
            classBuilder.addMethod(addChildMethod.build());
        }
        logger.exit();
    }

    /**
     * Generates the overrides of {@link Element#copy()} and
     * {@link Element#copyValues(Element)} if necessary.
     */
    private void generateMethodCopy(Builder classBuilder, IClassData classData) {
        logger.entry(classBuilder, classData);

        // only override copy() for non-abstract classes (contains instantiation)
        if (!classData.isAbstractClass()) {
            logger.debug("Generating method copy");
            classBuilder.addMethod(MethodSpec.methodBuilder("copy")
                    .addModifiers(Modifier.PUBLIC)
                    .returns(classData.getClassName())
                    .addJavadoc("@see $T#copy()", classData.getSuperclassName())
                    .addStatement("var dst = new $T()", classData.getClassName())
                    .addStatement("copyValues(dst)")
                    .addStatement("return dst")
                    .build());
        }

        // only override copyValues() if there are additional attributes to cover
        final var attributes = classData.getAttributes();
        if (!attributes.isEmpty()) {
            logger.debug("Generating method copyValues");
            final var copyValuesMethod = MethodSpec.methodBuilder("copyValues")
                    .addModifiers(Modifier.PUBLIC)
                    .addParameter(classData.getClassName(), "dst")
                    .addJavadoc(
                            """
                            @param dst the destination object to which values should be copied
                            @see $1T#copyValues($1T)
                            """,
                            classData.getSuperclassName())
                    .addStatement("super.copyValues(dst)");
            for (final var attribute : attributes) {
                if (attribute.isRepeating()) {
                    copyValuesMethod
                            .beginControlFlow("if ($N != null)", attribute.getName())
                            .addStatement("dst.$N = new $T()", attribute.getName(), attribute.getCreationType())
                            .beginControlFlow("for (var entry : $N)", attribute.getName())
                            .addStatement("dst.$N.add(entry.copy())", attribute.getName())
                            .endControlFlow()
                            .endControlFlow();
                } else {
                    copyValuesMethod.addStatement("dst.$1N = $1N == null ? null : $1N.copy()", attribute.getName());
                }
            }
            classBuilder.addMethod(copyValuesMethod.build());
        }
        logger.exit();
    }

    /**
     * Generates the override of {@link Element#equalsDeep(Base)} if required.
     */
    private void generateMethodEqualsDeep(Builder classBuilder, IClassData classData) {
        logger.entry(classBuilder, classData);
        final var attributes = classData.getAttributes();
        if (!attributes.isEmpty()) {
            logger.debug("Generating method equalsDeep");
            final var equalsDeepMethod = MethodSpec.methodBuilder("equalsDeep")
                    .addModifiers(Modifier.PUBLIC)
                    .returns(TypeName.BOOLEAN)
                    .addParameter(this.frameworkTypeLocator.getBaseType(), "other")
                    .addAnnotation(Override.class)
                    .beginControlFlow("if (!super.equalsDeep(other))")
                    .addStatement("return false")
                    .endControlFlow()
                    .beginControlFlow("if (!(other instanceof $T castOther))", classData.getClassName())
                    .addStatement("return false")
                    .endControlFlow();
            for (final var attribute : attributes) {
                equalsDeepMethod
                        .beginControlFlow("if (!compareDeep(this.$1N, castOther.$1N, true))", attribute.getName())
                        .addStatement("return false")
                        .endControlFlow();
            }
            equalsDeepMethod.addStatement("return true");
            classBuilder.addMethod(equalsDeepMethod.build());
        }
        logger.exit();
    }

    /**
     * Generates the override of {@link Element#equalsShallow(Base)} if required.
     */
    private void generateMethodEqualsShallow(Builder classBuilder, IClassData classData) {
        logger.entry(classBuilder, classData);
        final var attributes = classData.getAttributes();
        if (!attributes.isEmpty()) {
            logger.debug("Generating method equalsShallow");
            final var equalsShallowMethod = MethodSpec.methodBuilder("equalsShallow")
                    .addModifiers(Modifier.PUBLIC)
                    .returns(TypeName.BOOLEAN)
                    .addParameter(this.frameworkTypeLocator.getBaseType(), "other")
                    .addAnnotation(Override.class)
                    .beginControlFlow("if (!super.equalsShallow(other))")
                    .addStatement("return false")
                    .endControlFlow()
                    .beginControlFlow("if (!(other instanceof $T))", classData.getClassName())
                    .addStatement("return false")
                    .endControlFlow();

            if (attributes.stream().anyMatch(IClassAttribute::isDerivedFromPrimitiveType)) {
                equalsShallowMethod.addStatement("var castOther = ($T) other", classData.getClassName());
            }
            attributes.stream()
                    .filter(IClassAttribute::isDerivedFromPrimitiveType)
                    .forEach(attribute -> equalsShallowMethod
                            .beginControlFlow("if (!compareValues(this.$1N, castOther.$1N, true))", attribute.getName())
                            .addStatement("return false")
                            .endControlFlow());
            equalsShallowMethod.addStatement("return true");
            classBuilder.addMethod(equalsShallowMethod.build());
        }
        logger.exit();
    }

    /**
     * Generates the override of {@link Element#fhirType()}.
     */
    private void generateMethodFhirType(Builder classBuilder, ITopLevelClass classData) {
        logger.entry(classBuilder, classData);
        logger.debug("Generating method fhirType");
        classBuilder.addMethod(MethodSpec.methodBuilder("fhirType")
                .addModifiers(Modifier.PUBLIC)
                .returns(ClassName.get(String.class))
                .addAnnotation(Override.class)
                .addStatement("return $S", classData.getStructureDefinitionName())
                .build());
        logger.exit();
    }

    /**
     * Generates the override of {@link Element#isEmpty()} if required.
     */
    private void generateMethodIsEmpty(Builder classBuilder, IClassData classData) {
        logger.entry(classBuilder, classData);
        final var attributes = classData.getAttributes();
        if (!attributes.isEmpty()) {
            // We need a format string for the varargs call to ElementUtil.isEmpty: As many
            // $N (name references) as we
            // have custom attributes, separated by comma.
            final var attributeFormatEntries = new String[attributes.size()];
            Arrays.fill(attributeFormatEntries, "$N");

            final var statement = String.format(
                    "return super.isEmpty() && $T.isEmpty(%s)", String.join(", ", attributeFormatEntries));

            // The first parameter is the ElementUtil type reference ($T above), the
            // remaining ones are the names of the
            // attributes.
            final var parameters = attributes.stream()
                    .map(a -> a.getName())
                    .collect(Collectors.toCollection(() -> new ArrayList<Object>()));
            parameters.add(0, ClassName.get(ElementUtil.class));

            logger.debug("Generating method isEmpty");
            classBuilder.addMethod(MethodSpec.methodBuilder("isEmpty")
                    .addModifiers(Modifier.PUBLIC)
                    .returns(TypeName.BOOLEAN)
                    .addAnnotation(Override.class)
                    .addStatement(statement, parameters.toArray())
                    .build());
        }
        logger.exit();
    }

    /**
     * Generates the override of
     * {@link Element#getNamedProperty(int, String, boolean)} if required.
     *
     * @throws GeneratorException
     */
    private void generateMethodGetNamedProperty(Builder classBuilder, IClassData classData) throws GeneratorException {
        logger.entry(classBuilder, classData);
        final var attributes = classData.getAttributes();
        if (!attributes.isEmpty()) {
            // TODO #72 support typed references
            // For a reference, the typeCode should be something like
            // "Reference(Organization|Practitioner|PractitionerRole)".
            // We don't supply this information at the moment.
            logger.debug("Generating method getNamedProperty");
            final var getNamedPropertMethod = MethodSpec.methodBuilder("getNamedProperty")
                    .addModifiers(Modifier.PUBLIC)
                    .returns(this.frameworkTypeLocator.getPropertyType())
                    .addParameter(TypeName.INT, "hash")
                    .addParameter(ClassName.get(String.class), "name")
                    .addParameter(TypeName.BOOLEAN, "checkValid")
                    .addAnnotation(Override.class)
                    .beginControlFlow("switch(hash)");
            for (final var attribute : attributes) {
                final var modelTypes = attribute.getMappedTypes();
                if (attribute.isMultiType()) {
                    // multi-type attribute
                    // The HAPI framework classes supply the same information for all variants of
                    // the
                    // property name, so we'll just do the same.
                    final var propertyNames =
                            Lists.newArrayList(attribute.getFhirName(), attribute.getFhirName() + "[x]");
                    modelTypes.stream()
                            .map(mt -> mt.getTypeCode())
                            .map(tc -> attribute.getFhirName() + StringUtilities.toFirstUpper(tc))
                            .forEach(propertyNames::add);
                    // All of the properties are generated with the same type information like
                    // "boolean|integer".
                    final var typeCodes = String.join(
                            "|", attribute.getPropertyTypes().stream().toList());
                    for (final var propertyName : propertyNames) {
                        getNamedPropertMethod
                                .addCode("case $L:\n", propertyName.hashCode())
                                .addCode("$>// $N\n", propertyName) // addComment doesn't get the indentation right
                                .addStatement(
                                        "return new $T($S, $S, $S, $L, $L, this.$N)",
                                        this.frameworkTypeLocator.getPropertyType(),
                                        propertyName,
                                        typeCodes,
                                        attribute.getDefinition().orElse(""),
                                        attribute.getMin(),
                                        attribute.getMax(),
                                        attribute.getName())
                                .addCode("$<");
                    }
                } else {
                    // single-type attribute
                    final var modelType = modelTypes.iterator().next();
                    getNamedPropertMethod
                            .addCode("case $L:\n", attribute.getFhirName().hashCode())
                            .addCode(
                                    "$>// $N\n",
                                    attribute.getFhirName()) // addComment doesn't get the indentation right
                            .addStatement(
                                    "return new $T($S, $S, $S, $L, $L, this.$N)",
                                    this.frameworkTypeLocator.getPropertyType(),
                                    attribute.getFhirName(),
                                    modelType.getTypeCode(),
                                    attribute.getDefinition().orElse(""),
                                    attribute.getMin(),
                                    attribute.getMax(),
                                    attribute.getName())
                            .addCode("$<");
                }
            }
            getNamedPropertMethod
                    .addCode("default:\n")
                    .addStatement("$>return super.getNamedProperty(hash, name, checkValid)")
                    .addCode("$<")
                    .endControlFlow();
            classBuilder.addMethod(getNamedPropertMethod.build());
        }
        logger.exit();
    }

    /**
     * Generates the override of {@link Element#getProperty(int, String, boolean)}
     * if required.
     *
     * @throws GeneratorException
     */
    private void generateMethodGetProperty(Builder classBuilder, IClassData classData) throws GeneratorException {
        logger.entry(classBuilder, classData);
        final var attributes = classData.getAttributes();
        if (!attributes.isEmpty()) {
            logger.debug("Generating method getProperty");
            final var getPropertMethod = MethodSpec.methodBuilder("getProperty")
                    .addModifiers(Modifier.PUBLIC)
                    .returns(ArrayTypeName.of(this.frameworkTypeLocator.getBaseType()))
                    .addParameter(TypeName.INT, "hash")
                    .addParameter(ClassName.get(String.class), "name")
                    .addParameter(TypeName.BOOLEAN, "checkValid")
                    .addException(ClassName.get(FHIRException.class))
                    .addAnnotation(Override.class)
                    .beginControlFlow("switch(hash)");
            for (final var attribute : attributes) {
                getPropertMethod
                        .addCode("case $L:\n", attribute.getFhirName().hashCode())
                        .addCode("$>// $N\n", attribute.getFhirName()); // addComment doesn't get the indentation right
                if (attribute.isRepeating()) {
                    // repeating attribute
                    getPropertMethod.addStatement(
                            "return this.$1N == null ? new $2T[0] : this.$1N.toArray(new $2T[this.$1N.size()])",
                            attribute.getName(),
                            this.frameworkTypeLocator.getBaseType());
                } else {
                    // non-repeating attribute
                    getPropertMethod.addStatement(
                            "return this.$1N == null ? new $2T[0] : new $2T[] { this.$1N } ",
                            attribute.getName(),
                            this.frameworkTypeLocator.getBaseType());
                }
                getPropertMethod.addCode("$<");
            }
            getPropertMethod
                    .addCode("default:\n")
                    .addStatement("$>return super.getProperty(hash, name, checkValid)")
                    .addCode("$<")
                    .endControlFlow();
            classBuilder.addMethod(getPropertMethod.build());
        }
        logger.exit();
    }

    /**
     * Generates the override of {@link Element#getTypesForProperty(int, String)} if
     * required.
     *
     * @throws GeneratorException
     */
    private void generateMethodGetTypesForProperty(Builder classBuilder, IClassData classData)
            throws GeneratorException {
        logger.entry(classBuilder, classData);
        final var attributes = classData.getAttributes();
        if (!attributes.isEmpty()) {
            logger.debug("Generating method getTypesForProperty");
            final var getPropertMethod = MethodSpec.methodBuilder("getTypesForProperty")
                    .addModifiers(Modifier.PUBLIC)
                    .returns(ArrayTypeName.of(ClassName.get(String.class)))
                    .addParameter(TypeName.INT, "hash")
                    .addParameter(ClassName.get(String.class), "name")
                    .addException(ClassName.get(FHIRException.class))
                    .addAnnotation(Override.class)
                    .beginControlFlow("switch(hash)");
            for (final var attribute : attributes) {
                final var typeCodes = String.join(
                        ", ",
                        attribute.getPropertyTypes().stream()
                                .map(s -> String.format("\"%s\"", s))
                                .toList());
                getPropertMethod
                        .addCode("case $L:\n", attribute.getFhirName().hashCode())
                        .addCode("$>// $N\n", attribute.getFhirName()) // addComment doesn't get the indentation right
                        .addStatement("return new $1T[] { $2L }", ClassName.get(String.class), typeCodes)
                        .addCode("$<");
            }
            getPropertMethod
                    .addCode("default:\n")
                    .addStatement("$>return super.getTypesForProperty(hash, name)")
                    .addCode("$<")
                    .endControlFlow();
            classBuilder.addMethod(getPropertMethod.build());
        }
        logger.exit();
    }

    /**
     * Generates the override of {@link Resource#listChildren(String)} if required.
     *
     * @throws GeneratorException
     */
    private void generateMethodListChildren(Builder classBuilder, IClassData classData) throws GeneratorException {
        logger.entry(classBuilder, classData);
        final var attributes = classData.getAttributes();
        if (!attributes.isEmpty()) {
            // TODO #72 support typed references
            // For a reference, the typeCode should be something like
            // "Reference(Organization|Practitioner|PractitionerRole)".
            // We don't supply this information at the moment.
            logger.debug("Generating method listChildren");
            final var listChildrenMethod = MethodSpec.methodBuilder("listChildren")
                    .addModifiers(Modifier.PROTECTED)
                    .addParameter(
                            ParameterizedTypeName.get(
                                    ClassName.get(List.class), this.frameworkTypeLocator.getPropertyType()),
                            "children")
                    .addAnnotation(Override.class)
                    .addStatement("super.listChildren(children)");
            for (final var attribute : attributes) {
                final var typeCodes =
                        String.join("|", attribute.getPropertyTypes().stream().toList());
                final var propertyName =
                        (attribute.isMultiType()) ? attribute.getFhirName() + "[x]" : attribute.getFhirName();
                listChildrenMethod.addStatement(
                        "children.add(new $T($S, $S, $S, $L, $L, this.$N))",
                        this.frameworkTypeLocator.getPropertyType(),
                        propertyName,
                        typeCodes,
                        attribute.getDefinition().orElse(""),
                        attribute.getMin(),
                        attribute.getMax(),
                        attribute.getName());
            }
            classBuilder.addMethod(listChildrenMethod.build());
        }
        logger.exit();
    }

    /**
     * Generates the override of {@link Element#makeProperty(int, String)} if
     * required.
     *
     * @throws GeneratorException
     */
    private void generateMethodMakeProperty(Builder classBuilder, IClassData classData) throws GeneratorException {
        logger.entry(classBuilder, classData);
        final var attributes = classData.getAttributes();
        if (!attributes.isEmpty()) {
            logger.debug("Generating method makeProperty");
            final var makePropertyMethod = MethodSpec.methodBuilder("makeProperty")
                    .addModifiers(Modifier.PUBLIC)
                    .returns(this.frameworkTypeLocator.getBaseType())
                    .addParameter(TypeName.INT, "hash")
                    .addParameter(ClassName.get(String.class), "name")
                    .addException(ClassName.get(FHIRException.class))
                    .addAnnotation(Override.class)
                    .beginControlFlow("switch(hash)");
            for (final var attribute : attributes) {
                final var propertyNames = Lists.newArrayList(attribute.getFhirName());
                if (attribute.isMultiType()) {
                    // multi-type attribute: add both fooBar and fooBar[X] as property name
                    // alternatives
                    propertyNames.add(attribute.getFhirName() + "[x]");
                }
                for (final var propertyName : propertyNames) {
                    makePropertyMethod
                            .addCode("case $L:\n", propertyName.hashCode())
                            .addCode("$>// $N\n", propertyName); // addComment doesn't get the indentation right
                    if (attribute.isRepeating()) {
                        if (attribute.isMultiType()) {
                            // This does not appear to be supported by the framework? More research needed!
                            logger.warn(
                                    "Not generating a makeProperty branch for multi-type repeating element {}",
                                    attribute.getElementId());
                        } else {
                            makePropertyMethod.addStatement("return $N()", getAdderForNewSingleTypeName(attribute));
                        }
                    } else {
                        makePropertyMethod.addStatement("return $N()", getGetterForSingleElementTypeName(attribute));
                    }
                    makePropertyMethod.addCode("$<");
                }
            }
            makePropertyMethod
                    .addCode("default:\n")
                    .addStatement("$>return super.makeProperty(hash, name)")
                    .addCode("$<")
                    .endControlFlow();
            classBuilder.addMethod(makePropertyMethod.build());
        }
        logger.exit();
    }

    /**
     * Generates the override of {@link Base#removeChild(String, Base)} if
     * necessary.
     */
    private void generateMethodRemoveChild(Builder classBuilder, IClassData classData) throws GeneratorException {
        logger.entry(classBuilder, classData);
        // only override removeChild(String) if there are additional attributes to cover
        final var attributes = classData.getAttributes();
        if (!attributes.isEmpty()) {
            final var baseType = this.frameworkTypeLocator.getBaseType();
            logger.debug("Generating method removeChild");
            final var removeChildMethod = MethodSpec.methodBuilder("removeChild")
                    .addModifiers(Modifier.PUBLIC)
                    .addParameter(String.class, "name")
                    .addParameter(baseType, "value")
                    .addException(FHIRException.class)
                    .addAnnotation(Override.class)
                    .beginControlFlow("switch(name)");
            for (final var attribute : attributes) {
                final var propertyNames = Lists.newArrayList(attribute.getFhirName());
                if (attribute.isMultiType()) {
                    // multi-type attribute: add both fooBar and fooBar[X] as property name
                    // alternatives
                    propertyNames.add(attribute.getFhirName() + "[x]");
                }
                for (final var propertyName : propertyNames) {
                    removeChildMethod.addCode("case $S:\n", propertyName).addCode("$>");
                    if (attribute.isRepeating()) {
                        final var castingMethod = attribute.getBaseCastingMethod();
                        if (castingMethod.isPresent()) {
                            removeChildMethod.addStatement(
                                    "this.$N().remove($N(value))",
                                    getGetterForGenericMultiTypeName(attribute),
                                    castingMethod.get());
                        } else {
                            removeChildMethod.addStatement(
                                    "this.$N().remove(value)", getGetterForGenericMultiTypeName(attribute));
                        }
                    } else {
                        removeChildMethod.addStatement("this.$N = null", attribute.getName());
                    }
                    removeChildMethod.addStatement("break").addCode("$<");
                }
            }
            removeChildMethod
                    .addCode("default:\n")
                    .addStatement("$>super.removeChild(name, value)")
                    .addCode("$<")
                    .endControlFlow();
            classBuilder.addMethod(removeChildMethod.build());
        }
        logger.exit();
    }

    /**
     * Generates the override of {@link Base#setProperty(int, String, Base)} if
     * necessary.
     */
    @SuppressWarnings("java:S125") // see comment below
    private void generateMethodSetPropertyByHash(Builder classBuilder, IClassData classData) throws GeneratorException {
        // TODO #73 support enum conversion in generic setters
        // This relates to the following code found in the framework methods:
        // value = new AdministrativeGenderEnumFactory().fromType(castToCode(value));
        // this.gender = (Enumeration) value; // Enumeration<AdministrativeGender>

        logger.entry(classBuilder, classData);
        // only override removeChild(String) if there are additional attributes to cover
        final var attributes = classData.getAttributes();
        if (!attributes.isEmpty()) {
            final var baseType = this.frameworkTypeLocator.getBaseType();
            logger.debug("Generating method setProperty");
            final var removeChildMethod = MethodSpec.methodBuilder("setProperty")
                    .addModifiers(Modifier.PUBLIC)
                    .returns(this.frameworkTypeLocator.getBaseType())
                    .addParameter(TypeName.INT, "hash")
                    .addParameter(String.class, "name")
                    .addParameter(baseType, "value")
                    .addException(FHIRException.class)
                    .addJavadoc("@see $1T#setProperty(int, String, $1T)", baseType)
                    .beginControlFlow("switch(hash)");
            for (final var attribute : attributes) {
                final var castingMethod = attribute.getBaseCastingMethod();
                final var propertyNames = Lists.newArrayList(attribute.getFhirName());
                if (attribute.isMultiType()) {
                    // multi-type attribute: add both fooBar and fooBar[X] as property name
                    // alternatives
                    propertyNames.add(attribute.getFhirName() + "[x]");
                }
                for (final var propertyName : propertyNames) {
                    removeChildMethod
                            .addCode("case $L:\n", propertyName.hashCode())
                            .addCode("$>// $N\n", propertyName); // addComment
                    // doesn't
                    // get
                    // the
                    // indentation
                    // right
                    if (attribute.isRepeating()) {
                        if (castingMethod.isPresent()) {
                            removeChildMethod.addStatement(
                                    "this.$N().add($N(value))",
                                    getGetterForGenericMultiTypeName(attribute),
                                    castingMethod.get());
                        } else {
                            removeChildMethod.addStatement(
                                    "this.$N().add(($T)value)",
                                    getGetterForGenericMultiTypeName(attribute),
                                    attribute.getAttributeType());
                        }
                    } else {
                        if (castingMethod.isPresent()) {
                            removeChildMethod.addStatement(
                                    "this.$N = $N(value)", attribute.getName(), castingMethod.get());
                        } else {
                            removeChildMethod.addStatement(
                                    "this.$N = ($T)value", attribute.getName(), attribute.getAttributeType());
                        }
                    }
                    removeChildMethod.addStatement("return value").addCode("$<");
                }
            }
            removeChildMethod
                    .addCode("default:\n")
                    .addStatement("$>return super.setProperty(name, value)")
                    .addCode("$<")
                    .endControlFlow();
            classBuilder.addMethod(removeChildMethod.build());
        }
        logger.exit();
    }

    /**
     * Generates the override of {@link Base#setProperty(String, Base)} if
     * necessary.
     */
    @SuppressWarnings("java:S125") // see comment below
    private void generateMethodSetPropertyByName(Builder classBuilder, IClassData classData) throws GeneratorException {
        // TODO #73 support enum conversion in generic setters
        // This relates to the following code found in the framework methods:
        // value = new AdministrativeGenderEnumFactory().fromType(castToCode(value));
        // this.gender = (Enumeration) value; // Enumeration<AdministrativeGender>

        logger.entry(classBuilder, classData);
        // only override removeChild(String) if there are additional attributes to cover
        final var attributes = classData.getAttributes();
        if (!attributes.isEmpty()) {
            final var baseType = this.frameworkTypeLocator.getBaseType();
            logger.debug("Generating method setProperty");
            final var removeChildMethod = MethodSpec.methodBuilder("setProperty")
                    .addModifiers(Modifier.PUBLIC)
                    .returns(this.frameworkTypeLocator.getBaseType())
                    .addParameter(String.class, "name")
                    .addParameter(baseType, "value")
                    .addException(FHIRException.class)
                    .addJavadoc("@see $1T#setProperty(int, String, $1T)", baseType)
                    .beginControlFlow("switch(name)");
            for (final var attribute : attributes) {
                final var castingMethod = attribute.getBaseCastingMethod();
                final var propertyNames = Lists.newArrayList(attribute.getFhirName());
                if (attribute.isMultiType()) {
                    // multi-type attribute: add both fooBar and fooBar[X] as property name
                    // alternatives
                    propertyNames.add(attribute.getFhirName() + "[x]");
                }
                for (final var propertyName : propertyNames) {
                    removeChildMethod.addCode("case $S:\n", propertyName).addCode("$>");
                    if (attribute.isRepeating()) {
                        if (castingMethod.isPresent()) {
                            removeChildMethod.addStatement(
                                    "this.$N().add($N(value))",
                                    getGetterForGenericMultiTypeName(attribute),
                                    castingMethod.get());
                        } else {
                            removeChildMethod.addStatement(
                                    "this.$N().add(($T)value)",
                                    getGetterForGenericMultiTypeName(attribute),
                                    attribute.getAttributeType());
                        }
                    } else {
                        if (castingMethod.isPresent()) {
                            removeChildMethod.addStatement(
                                    "this.$N = $N(value)", attribute.getName(), castingMethod.get());
                        } else {
                            removeChildMethod.addStatement(
                                    "this.$N = ($T)value", attribute.getName(), attribute.getAttributeType());
                        }
                    }
                    removeChildMethod.addStatement("return value").addCode("$<");
                }
            }
            removeChildMethod
                    .addCode("default:\n")
                    .addStatement("$>return super.setProperty(name, value)")
                    .addCode("$<")
                    .endControlFlow();
            classBuilder.addMethod(removeChildMethod.build());
        }
        logger.exit();
    }

    /**
     * Generates the override of {@link IBaseHasExtensions#getExtension()} if
     * necessary.
     */
    private void generateMethodGetExtension(Builder classBuilder, IClassData classData) throws GeneratorException {
        logger.entry(classBuilder, classData);
        final var extensionAttributes = classData.getAttributes().stream()
                .filter(IClassAttribute::isExtension)
                .toList();
        if (!extensionAttributes.isEmpty()) {
            logger.debug("Generating method getExtension");
            final var extensionType = this.frameworkTypeLocator.getExtensionType();
            final var getExtensionMethod = MethodSpec.methodBuilder("getExtension")
                    .addModifiers(Modifier.PUBLIC)
                    .returns(ParameterizedTypeName.get(ClassName.get(List.class), extensionType))
                    .addAnnotation(Override.class)
                    .addStatement("final var extensions = super.getExtension()");

            generateGetExtensionControlFlow(classBuilder, getExtensionMethod, extensionAttributes);
        }
        logger.exit();
    }

    /**
     * Generates the override of {@link IBaseHasExtensions#hasExtension()} if
     * necessary.
     */
    private void generateMethodHasExtension(Builder classBuilder, IClassData classData) {
        logger.entry(classBuilder, classData);
        final var extensionAttributes = classData.getAttributes().stream()
                .filter(IClassAttribute::isExtension)
                .toList();
        if (!extensionAttributes.isEmpty()) {
            logger.debug("Generating method hasExtension");
            final var hasExtensionMethod = MethodSpec.methodBuilder("hasExtension")
                    .addModifiers(Modifier.PUBLIC)
                    .returns(TypeName.BOOLEAN)
                    .addAnnotation(Override.class)
                    .addStatement("var hasExtension = super.hasExtension()");
            for (final var attribute : extensionAttributes) {
                hasExtensionMethod.addStatement("hasExtension |= this.$N()", getMethodName("has", attribute));
            }
            hasExtensionMethod.addStatement("return hasExtension");
            classBuilder.addMethod(hasExtensionMethod.build());
        }
        logger.exit();
    }

    /**
     * Generates the override of {@link IBaseHasModifierExtensions#getModifierExtension()} if
     * necessary.
     */
    private void generateMethodGetModifierExtension(Builder classBuilder, IClassData classData)
            throws GeneratorException {
        logger.entry(classBuilder, classData);
        final var extensionAttributes = classData.getAttributes().stream()
                .filter(IClassAttribute::isModifierExtension)
                .toList();
        if (!extensionAttributes.isEmpty()) {
            logger.debug("Generating method getModifierExtension");
            final var extensionType = this.frameworkTypeLocator.getExtensionType();
            final var getExtensionMethod = MethodSpec.methodBuilder("getModifierExtension")
                    .addModifiers(Modifier.PUBLIC)
                    .returns(ParameterizedTypeName.get(ClassName.get(List.class), extensionType))
                    .addAnnotation(Override.class)
                    .addStatement("final var extensions = super.getModifierExtension()");

            generateGetExtensionControlFlow(classBuilder, getExtensionMethod, extensionAttributes);
        }
        logger.exit();
    }

    private void generateGetExtensionControlFlow(
            Builder classBuilder, MethodSpec.Builder getExtensionMethod, List<IClassAttribute> extensionAttributes)
            throws GeneratorException {
        logger.entry(classBuilder, getExtensionMethod, extensionAttributes);
        final var iterator = extensionAttributes.listIterator();
        while (iterator.hasNext()) {
            final var attributeIndex = String.valueOf(iterator.nextIndex());
            final var attribute = iterator.next();

            if (attribute.isRepeating()) {
                getExtensionMethod.beginControlFlow("for(var value: $N)", attribute.getName());
                generateGetExtensionAttributeCheck(getExtensionMethod, attribute, attributeIndex);
                getExtensionMethod.endControlFlow();
            } else {
                generateGetExtensionAttributeCheck(getExtensionMethod, attribute, attributeIndex);
            }
        }
        getExtensionMethod.addStatement("return extensions");
        classBuilder.addMethod(getExtensionMethod.build());
        logger.exit();
    }

    private void generateGetExtensionAttributeCheck(
            MethodSpec.Builder getExtensionMethod, IClassAttribute attribute, String attributeIndex)
            throws GeneratorException {
        logger.entry(getExtensionMethod, attribute, attributeIndex);
        final var extensionType = this.frameworkTypeLocator.getExtensionType();
        final var url = attribute
                .getExtensionUrl()
                .orElseThrow(() -> new GeneratorException(String.format(
                        "Attribute %s is marked as an extension but does not provide an extension URL",
                        attribute.getName())));

        if (attribute.isRepeating()) {
            getExtensionMethod.addStatement("$T e$N = new $T($S).setValue(value)", extensionType, attributeIndex, url);
        } else {
            getExtensionMethod.addStatement(
                    "$1T e$2N = new $1T($3S).setValue(this.$4N)",
                    extensionType,
                    attributeIndex,
                    url,
                    attribute.getName());
        }
        getExtensionMethod.beginControlFlow(
                "if ((this.$N != null) && extensions.stream().noneMatch(e -> e.equalsDeep(e$N)))",
                attribute.getName(),
                attributeIndex);

        getExtensionMethod.addStatement("extensions.add(e$N)", attributeIndex);
        getExtensionMethod.endControlFlow();
        logger.exit();
    }

    /**
     * Generates the override of {@link IBaseHasModifierExtensions#hasModifierExtension()} if
     * necessary.
     */
    private void generateMethodHasModifierExtension(Builder classBuilder, IClassData classData) {
        logger.entry(classBuilder, classData);
        final var extensionAttributes = classData.getAttributes().stream()
                .filter(IClassAttribute::isModifierExtension)
                .toList();
        if (!extensionAttributes.isEmpty()) {
            logger.debug("Generating method hasModifierExtension");
            final var hasExtensionMethod = MethodSpec.methodBuilder("hasModifierExtension")
                    .addModifiers(Modifier.PUBLIC)
                    .returns(TypeName.BOOLEAN)
                    .addAnnotation(Override.class)
                    .addStatement("var hasExtension = super.hasModifierExtension()");
            for (final var attribute : extensionAttributes) {
                hasExtensionMethod.addStatement("hasExtension |= this.$N()", getMethodName("has", attribute));
            }
            hasExtensionMethod.addStatement("return hasExtension");
            classBuilder.addMethod(hasExtensionMethod.build());
        }
        logger.exit();
    }

    /*
     * Generates the factory method create() if necessary.
     */
    private void generateMethodCreate(Builder classBuilder, IClassData classData) {
        logger.entry(classBuilder, classData);

        final var rules = classData.getFixedValueRules();
        if (!rules.isEmpty()) {
            logger.debug("Generating factory method create");
            final var generatedClassName = classData.getClassName();
            final var createMethod = MethodSpec.methodBuilder("create")
                    .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                    .returns(generatedClassName);

            // generate Javadoc comment
            createMethod.addJavadoc(
                    "Creates a new instance of $T with the following fixed or pattern values already preset:\n<ul>\n",
                    generatedClassName);
            rules.forEach(rule -> addValueJavadocForCreateMethod(createMethod, rule));
            createMethod.addJavadoc(
                    "</ul>\n@return a new instance of $T with the fixed/pattern values preset", generatedClassName);

            // generate method body
            createMethod.addStatement("final var _result = new $T()", generatedClassName);
            rules.forEach(rule -> addValueTransferForCreateMethod(createMethod, rule, "_result"));
            createMethod.addStatement("return _result");

            classBuilder.addMethod(createMethod.build());
        }
        logger.exit();
    }

    private void addValueJavadocForCreateMethod(final MethodSpec.Builder createMethod, final IFixedValueRule rule) {
        logger.entry(rule);
        final var value = rule.getValue();
        if (value.isPresent()) {
            if (rule.isLiteral()) {
                createMethod.addJavadoc("<li>$L = $L</li>\n", rule.getPath(), value.get());
            } else {
                createMethod.addJavadoc("<li>$L = $S</li>\n", rule.getPath(), value.get());
            }
        } else {
            for (final var property : rule.getProperties()) {
                addValueJavadocForCreateMethod(createMethod, property);
            }
        }
        logger.exit();
    }

    private void addValueTransferForCreateMethod(MethodSpec.Builder createMethod, IFixedValueRule rule, String target) {
        logger.entry(rule);
        final var prefix = rule.getPrefix();
        final var variableName = prefix.isPresent()
                ? prefix.get() + "_" + rule.getPropertyName().replace("[x]", "")
                : rule.getPropertyName().replace("[x]", "");
        final var value = rule.getValue();
        createMethod.addComment(rule.getPath());
        if (value.isPresent()) {
            if (rule.isLiteral()) {
                createMethod.addStatement(
                        "final var $N = new $T($L)", variableName, rule.getPropertyType(), value.get());
            } else {
                createMethod.addStatement(
                        "final var $N = new $T($S)", variableName, rule.getPropertyType(), value.get());
            }
        } else {
            createMethod.addStatement("final var $N = new $T()", variableName, rule.getPropertyType());
            for (final var property : rule.getProperties()) {
                addValueTransferForCreateMethod(createMethod, property, variableName);
            }
        }
        createMethod.addStatement("$1N.setProperty($2S, $3N)", target, rule.getPropertyName(), variableName);
        logger.exit();
    }

    /**
     * Determines the format string used to transfer a value as a generated
     * parameter.
     *
     * @param value
     * @return
     */
    private String getFormatString(final Object value) {
        logger.entry(value);
        final String formatString;
        switch (value) {
            case final String s:
                formatString = "$S";
                break;
            case final TypeName t:
                formatString = "$T.class";
                break;
            default:
                formatString = "$L";
        }
        return logger.exit(formatString);
    }

    /**
     * Determines the format string used to transfer an array of values as a
     * generated parameter.
     *
     * @param array
     * @return
     */
    private String getFormatString(final Object[] value) {
        logger.entry(value);
        final String formatString;
        final var arrayType = value.getClass().getComponentType();
        String elementFormat;
        if (String.class.isAssignableFrom(arrayType)) {
            elementFormat = "$S";
        } else if (TypeName.class.isAssignableFrom(arrayType)) {
            elementFormat = "$T.class";
        } else {
            elementFormat = "$L";
        }
        final var elementFormats = new String[Array.getLength(value)];
        Arrays.fill(elementFormats, elementFormat);
        formatString = String.format("{ %s }", String.join(", ", elementFormats));
        return logger.exit(formatString);
    }

    /**
     * Writes the generated code to the output file.
     *
     * @param classBuilder
     * @param className
     * @param outputPath
     * @throws GeneratorException
     */
    private void writeClassToFile(TypeSpec.Builder enumBuilder, ITopLevelClass classData) throws GeneratorException {
        logger.entry();
        final var enumSpec = enumBuilder.build();
        final JavaFile javaFile = JavaFile.builder(classData.getClassName().packageName(), enumSpec)
                .skipJavaLangImports(true)
                .indent("\t")
                .build();
        try {
            javaFile.writeTo(new File(classData.getOutputPath()));
        } catch (final IOException e) {
            throw logger.throwing(new GeneratorException(
                    String.format(
                            "Unable to write generated implementation %s to output path %s",
                            classData.getClassName().canonicalName(), classData.getOutputPath()),
                    e));
        }
        logger.exit();
    }

    private String getMethodName(String prefix, IClassAttribute attribute) {
        return StringUtilities.toFirstLower(prefix) + StringUtilities.toFirstUpper(attribute.getName());
    }

    private String getMethodName(String prefix, IClassAttribute attribute, String suffix) {
        return StringUtilities.toFirstLower(prefix)
                + StringUtilities.toFirstUpper(attribute.getName())
                + StringUtilities.toFirstUpper(suffix);
    }
}
