/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition.code;

import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.palantir.javapoet.ClassName;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.fhir.ClassAnnotation;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.experimental.SuperBuilder;

/**
 * Base class of {@link TopLevelClass} and {@link NestedClass}.
 */
@EqualsAndHashCode
@SuperBuilder(setterPrefix = "with")
public abstract class ClassData implements IClassData {

    @Getter
    private final ClassName className;

    @Getter
    private final TypeName superclassName;

    @Getter
    private final boolean abstractClass;

    @Getter
    private final String structureDefinitionName;

    @Getter
    private final String structureDefinitionURL;

    @Getter
    private final boolean derivedFromPrimitiveType;

    @Getter
    private final Optional<ClassAnnotation> classAnnotation;

    private final List<IClassConstructor> constructorList = new ArrayList<>();

    private final List<IClassAttribute> attributeList = new ArrayList<>();

    private final List<IFixedValueRule> fixedValueRuleList = new ArrayList<>();

    @Override
    public ImmutableCollection<IClassConstructor> getConstructors() {
        return ImmutableList.copyOf(this.constructorList);
    }

    /**
     * Adds an constructor to the class.
     *
     * @param constructor the constructor to add
     */
    public void addConstructor(IClassConstructor constructor) {
        this.constructorList.add(constructor);
    }

    @Override
    public ImmutableList<IClassAttribute> getAttributes() {
        return ImmutableList.copyOf(this.attributeList);
    }

    /**
     * Adds an attribute to the class.
     *
     * @param attribute the attribute to add
     */
    public void addAttribute(IClassAttribute attribute) {
        this.attributeList.add(attribute);
    }

    @Override
    public ImmutableList<IFixedValueRule> getFixedValueRules() {
        return ImmutableList.copyOf(this.fixedValueRuleList);
    }

    /**
     * Adds a fixed value rule to the class.
     *
     * @param rule the rule to add
     */
    public void addFixedValueRule(IFixedValueRule rule) {
        this.fixedValueRuleList.add(rule);
    }
}
