/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator.structuredefinition;

import de.fhlintstone.fhir.dependencies.IDependencyGraph;
import de.fhlintstone.fhir.dependencies.IDependencyNode;
import de.fhlintstone.generator.GeneratorException;
import de.fhlintstone.generator.structuredefinition.code.ICodeEmitter;
import de.fhlintstone.generator.structuredefinition.intermediate.ITypeInformation;
import de.fhlintstone.generator.structuredefinition.intermediate.ITypeInformationBuilder;
import de.fhlintstone.generator.structuredefinition.intermediate.ITypeInformationConverter;
import de.fhlintstone.generator.structuredefinition.intermediate.ITypeInformationStore;
import de.fhlintstone.packages.FhirResourceType;
import de.fhlintstone.process.config.ProcessConfiguration;
import java.util.List;
import java.util.Optional;
import javax.inject.Inject;
import javax.inject.Named;
import lombok.extern.slf4j.XSlf4j;

/**
 * Default implementation of {@link IStructureDefinitionGenerator}.
 */
@Named
@XSlf4j
public class StructureDefinitionGenerator implements IStructureDefinitionGenerator {

    private final ITypeInformationBuilder typeInformationBuilder;
    private final ITypeInformationStore typeInformationStore;
    private final ITypeInformationConverter typeInformationConverter;
    private final ICodeEmitter codeEmitter;

    /**
     * Constructor for dependency injection.
     *
     * @param typeInformationBuilder   the {@link ITypeInformationBuilder} to use
     * @param typeInformationStore the {@link ITypeInformationStore} to use
     * @param typeInformationConverter the {@link ITypeInformationConverter} to use
     * @param codeEmitter              the {@link ICodeEmitter} to use
     */
    @Inject
    public StructureDefinitionGenerator(
            ITypeInformationBuilder typeInformationBuilder,
            ITypeInformationStore typeInformationStore,
            ITypeInformationConverter typeInformationConverter,
            ICodeEmitter codeEmitter) {
        super();
        this.typeInformationBuilder = typeInformationBuilder;
        this.typeInformationStore = typeInformationStore;
        this.typeInformationConverter = typeInformationConverter;
        this.codeEmitter = codeEmitter;
    }

    @Override
    public boolean generate(ProcessConfiguration globalConfiguration, IDependencyGraph dependencyGraph) {
        logger.entry();
        var result = true;
        this.typeInformationStore.clear();
        final var structureDefinitionNodes = getOrderedStructureDefinitionNodes(dependencyGraph);
        for (final var node : structureDefinitionNodes) {
            result &= processNode(globalConfiguration, node);
        }
        return logger.exit(result);
    }

    private List<IDependencyNode> getOrderedStructureDefinitionNodes(IDependencyGraph dependencyGraph) {
        logger.entry(dependencyGraph);
        final var structureDefinitionNodes = dependencyGraph.getOrderedNodeList().stream()
                .filter(n -> n.getResourceType().isPresent()
                        && n.getResourceType().get().equals(FhirResourceType.STRUCTURE_DEFINITION))
                .toList();
        logger.debug("Found {} StructureDefinition nodes to process", structureDefinitionNodes.size());
        return logger.exit(structureDefinitionNodes);
    }

    private boolean processNode(ProcessConfiguration globalConfiguration, final IDependencyNode node) {
        logger.entry(node);
        logger.debug("Processing StructureDefinition node {}", node.getResourceURI());

        Optional<ITypeInformation> typeInformation = Optional.empty();
        try {
            typeInformation = this.typeInformationBuilder.buildTypeInformation(globalConfiguration, node);
        } catch (final GeneratorException e) {
            logger.error("Unable to generate StructureDefinition type information", e);
            return logger.exit(false);
        }
        if (typeInformation.isPresent()) {
            // store the type information (whether it is a generated or framework type)
            final var actualTypeInformation = typeInformation.get();
            this.typeInformationStore.add(actualTypeInformation);
            if (actualTypeInformation.isGenerated()) {
                try {
                    logger.info("Generating class for StructureDefinition {}", actualTypeInformation.getCanonicalURI());
                    final var topLevelClass = this.typeInformationConverter.buildGeneratorModel(
                            globalConfiguration, actualTypeInformation);
                    this.codeEmitter.generate(topLevelClass);
                } catch (final GeneratorException e) {
                    logger.error(
                            "Unable to generate class for StructureDefinition {}",
                            actualTypeInformation.getCanonicalURI(),
                            e);
                    return logger.exit(false);
                }
            }
        }
        return logger.exit(true);
    }
}
