/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator;

import com.palantir.javapoet.ClassName;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.process.config.ProcessConfiguration;
import java.net.URI;
import java.util.Optional;

/**
 * Provides central access to the fully qualified type names that are generated
 * according to the configuration.
 */
public interface IGeneratedTypeNameRegistry {

    /**
     * Resets the registry to use the configuration provided.
     *
     * @param configuration
     * @throws GeneratorException
     */
    void setConfiguration(ProcessConfiguration configuration) throws GeneratorException;

    /**
     * Returns the name of the enum generated for a ValueSet with the given URI.
     *
     * @param valueSetURI the URI of the ValueSet
     * @return the name of the enum generated for a ValueSet with the given URI
     */
    Optional<ClassName> getEnumName(URI valueSetURI);

    /**
     * Returns the name of the class generated for a StructureDefinition with
     * the given URI. This method will only resolve top-level classes and ignore
     * nested classes.
     *
     * @param structureDefinitionURI the URI of the StructureDefinition
     * @return the name of the class generated for a StructureDefinition with the
     *         given URI
     */
    Optional<ClassName> getClassName(URI structureDefinitionURI);

    /**
     * Returns the name of the nested class generated for an anonymous type within
     * a StructureDefinition with the given URI.
     *
     * @param parentURI the URI of the StructureDefinition that contains the anonymous type
     * @param elementId the ID of the element for which the nested class is generated
     * @return the name of the nested class
     */
    Optional<ClassName> getNestedClassName(URI parentURI, String elementId);

    /**
     * Returns the name of the nested superclass configured for an anonymous type within
     * a StructureDefinition with the given URI.
     *
     * @param parentURI the URI of the StructureDefinition that contains the anonymous type
     * @param elementId the ID of the element for which the nested class is generated
     * @return the name of superclass of the nested class
     */
    Optional<TypeName> getNestedSuperclassName(URI parentURI, String elementId);
}
