/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.generator;

import com.google.common.base.Strings;
import com.palantir.javapoet.ClassName;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.process.config.ProcessConfiguration;
import de.fhlintstone.process.config.StructureDefinitionClassConfiguration;
import de.fhlintstone.process.config.ValueSetEnumConfiguration;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import javax.inject.Named;
import javax.inject.Singleton;
import javax.lang.model.SourceVersion;
import lombok.extern.slf4j.XSlf4j;

/**
 * Default implementation of {@link IGeneratedTypeNameRegistry}.
 */
@Named
@Singleton
@XSlf4j
public class GeneratedTypeNameRegistry implements IGeneratedTypeNameRegistry {

    private final Map<URI, ClassName> enumNames = new HashMap<>();
    private final Map<URI, ClassName> classNames = new HashMap<>();
    private final Map<URI, TypeName> superclassNames = new HashMap<>(); // only filled for nested classes!

    @Override
    public void setConfiguration(ProcessConfiguration configuration) throws GeneratorException {
        logger.entry(configuration);
        this.enumNames.clear();
        this.classNames.clear();
        for (final var valueSet : configuration.getValueSetEnums()) {
            processValueSet(configuration, valueSet);
        }
        for (final var structureDefinition : configuration.getStructureDefinitionClasses()) {
            processStructureDefinition(configuration, structureDefinition);
        }
        logger.exit();
    }

    @Override
    public Optional<ClassName> getEnumName(URI valueSetURI) {
        logger.entry(valueSetURI);
        if (this.enumNames.containsKey(valueSetURI)) {
            return logger.exit(Optional.of(this.enumNames.get(valueSetURI)));
        }
        return logger.exit(Optional.empty());
    }

    @Override
    public Optional<ClassName> getClassName(URI structureDefinitionURI) {
        logger.entry(structureDefinitionURI);
        if (this.classNames.containsKey(structureDefinitionURI)) {
            return logger.exit(Optional.of(this.classNames.get(structureDefinitionURI)));
        }
        return logger.exit(Optional.empty());
    }

    @Override
    public Optional<ClassName> getNestedClassName(URI parentURI, String elementId) {
        logger.entry(parentURI, elementId);
        final var searchURI = buildNestedURI(parentURI, elementId);
        if (this.classNames.containsKey(searchURI)) {
            return logger.exit(Optional.of(this.classNames.get(searchURI)));
        }
        return logger.exit(Optional.empty());
    }

    @Override
    public Optional<TypeName> getNestedSuperclassName(URI parentURI, String elementId) {
        logger.entry(parentURI, elementId);
        final var searchURI = buildNestedURI(parentURI, elementId);
        if (this.superclassNames.containsKey(searchURI)) {
            return logger.exit(Optional.of(this.superclassNames.get(searchURI)));
        }
        return logger.exit(Optional.empty());
    }

    /**
     * Processes a ValueSet configuration entry.
     */
    private void processValueSet(ProcessConfiguration configuration, final ValueSetEnumConfiguration valueSet)
            throws GeneratorException {
        logger.entry(valueSet);
        final var valueSetURI = valueSet.getValueSet();
        validateName(valueSet.getEnumName());
        final var namespace = determineNamespace(configuration, valueSet.getTargetNamespace());
        final var enumName = ClassName.get(namespace, valueSet.getEnumName());
        logger.debug("Storing enum name {} for ValueSet {}", enumName, valueSetURI);
        this.enumNames.put(valueSetURI, enumName);
        logger.exit();
    }

    /**
     * Processes a StructureDefinition configuration entry.
     */
    private void processStructureDefinition(
            ProcessConfiguration configuration, final StructureDefinitionClassConfiguration structureDefinition)
            throws GeneratorException {
        logger.entry(structureDefinition);
        final var structureDefinitionURI = structureDefinition.getStructureDefinition();
        validateName(structureDefinition.getClassName());
        final var namespace = determineNamespace(configuration, structureDefinition.getTargetNamespace());
        final var className = ClassName.get(namespace, structureDefinition.getClassName());
        logger.debug("Storing class name {} for StructureDefinition {}", className, structureDefinitionURI);
        this.classNames.put(structureDefinitionURI, className);
        for (final var nestedClass : structureDefinition.getNestedClasses()) {
            validateName(nestedClass.getClassName());
            final var nestedClassURI = buildNestedURI(structureDefinitionURI, nestedClass.getElementId());
            final var nestedClassName = className.nestedClass(nestedClass.getClassName());
            logger.debug("Storing nested class name {} for StructureDefinition {}", nestedClassName, nestedClassURI);
            this.classNames.put(nestedClassURI, nestedClassName);
            final var nestedSuperClass = nestedClass.getSuperTypeName();
            if (nestedSuperClass.isPresent()) {
                try {
                    logger.debug(
                            "Storing nested superclass name {} for StructureDefinition {}",
                            nestedSuperClass.get(),
                            nestedClassURI);
                    this.superclassNames.put(nestedClassURI, nestedSuperClass.get());
                } catch (final IllegalArgumentException e) {
                    throw logger.throwing(
                            new GeneratorException(String.format("Invalid superclass name %s", nestedSuperClass), e));
                }
            }
        }

        logger.exit();
    }

    /**
     * Determines the namespace (either configured locally or from the global configuration default).
     */
    private String determineNamespace(ProcessConfiguration configuration, final Optional<String> localNamespace)
            throws GeneratorException {
        logger.entry(configuration, localNamespace);
        String namespace;
        if (localNamespace.isPresent()) {
            namespace = localNamespace.get();
        } else {
            final var globalNamespace = configuration.getDefaultNamespace();
            if (globalNamespace.isPresent()) {
                namespace = globalNamespace.get();
            } else {
                throw logger.throwing(
                        new GeneratorException(
                                "The target namespace must be specified (either as global default or for each configuration entry individually."));
            }
        }
        return logger.exit(namespace);
    }

    /**
     * Ensures that the configured name is present and a valid Java class name.
     */
    private void validateName(final String name) throws GeneratorException {
        logger.entry(name);
        if (Strings.isNullOrEmpty(name)) {
            throw logger.throwing(new GeneratorException("The target name must be specified."));
        } else {
            if (!SourceVersion.isName(name)) {
                throw logger.throwing(new GeneratorException(
                        String.format("The target name %s is not a valid Java type name.", name)));
            }
        }
        logger.exit();
    }

    /**
     * Assembles an URI for a nested class consisting of the URI of the StructureDefinition and the Element ID as a fragment (e.g. http://hl7.org/fhir/StructureDefinition/Patient#Patient.name).
     */
    private URI buildNestedURI(URI parentURI, String elementId) {
        logger.entry(parentURI, elementId);
        try {
            return logger.exit(new URI(parentURI.getScheme(), parentURI.getSchemeSpecificPart(), elementId));
        } catch (final URISyntaxException e) {
            throw logger.throwing(new RuntimeException(
                    String.format(
                            "Unable to create URI for nested class of parent URI %s, element ID %s",
                            parentURI, elementId),
                    e));
        }
    }
}
