/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.fhir.elements;

import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.model.IElementDefinitionAccessor;
import java.util.Optional;

/**
 * A node of the element tree, built from the snapshot of a StructureDefinition.
 */
public interface IElementTreeNode {

    /**
     * Returns the tree structure this node belongs to.
     *
     * @return the tree structure this node belongs to
     */
    IElementTree getTree();

    /**
     * Returns the element ID. The ID is returned unchanged, i.e. it includes the
     * slice name if present.
     *
     * @return the element ID
     */
    String getId();

    /**
     * Returns the "local" name of the wrapped element, i.e. the last part of the
     * element ID. Example: "profile" for the node wrapping "Patient.meta.profile".
     *
     * @return the "local" name of the wrapped element
     */
    String getLocalName();

    /**
     * Returns the name of the slice, if the node describes a slice.
     *
     * @return the name of the slice, if the node describes a slice
     */
    Optional<String> getSliceName();

    /**
     * Returns the snapshot ElementDefinition accessor. Since the element tree is
     * built from the snapshot, this accessor is always present.
     *
     * @return the snapshot ElementDefinition accessor
     */
    IElementDefinitionAccessor getSnapshotElement();

    /**
     * Returns the differential ElementDefinition accessor, if present.
     *
     * @return the differential ElementDefinition accessor, if present
     */
    Optional<IElementDefinitionAccessor> getDifferentialElement();

    /**
     * Returns the corresponding snapshot ElementDefinition accessor in the base
     * structure, if present.
     *
     * @return the corresponding snapshot ElementDefinition accessor in the base
     *         structure, if present
     */
    Optional<IElementDefinitionAccessor> getBaseElement();

    /**
     * Returns the {@link ElementBaseRelationship}
     *
     * @return the {@link ElementBaseRelationship}
     */
    ElementBaseRelationship getBaseRelationship();

    /**
     * Returns the parent node of this node, unless the node is the root node of the
     * tree.
     *
     * @return the parent node of this node, unless the node is the root node of the
     *         tree
     */
    Optional<IElementTreeNode> getParent();

    /**
     * Determines whether the node has child nodes beneath it.
     *
     * @return <code>true</code> if the node has child nodes
     * @see #getChildren()
     */
    boolean hasChildren();

    /**
     * Returns a list of the child nodes as defined by the StructureDefinition. This
     * list does <b>not</b> include any slices of the element.
     *
     * @return a list of the child nodes
     */
    ImmutableList<IElementTreeNode> getChildren();

    /**
     * Returns the child with the local name given, if it exists.
     *
     * @param localName The local name of the child
     * @return the child with the local name given, if it exists
     */
    Optional<IElementTreeNode> getChild(String localName);

    /**
     * Determines whether the node has slice nodes beneath it.
     *
     * @return <code>true</code> if the node has slice nodes
     * @see #getSlices()
     */
    boolean hasSlices();

    /**
     * Returns a list of the slices of the node.
     *
     * @return a list of the slices of the node
     */
    ImmutableList<IElementTreeNode> getSlices();

    /**
     * Returns the slice with the name given, if it exists.
     *
     * @param name The slice name
     * @return the slice with the name given, if it exists
     */
    Optional<IElementTreeNode> getSlice(String name);
}
