/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.fhir.elements;

import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.model.IElementDefinitionAccessor;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import lombok.Builder;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.extern.slf4j.XSlf4j;

/**
 * Default implementation of {@link IElementTreeNode}.
 */
@EqualsAndHashCode
@Builder(setterPrefix = "with")
@XSlf4j
public final class ElementTreeNode implements IElementTreeNode {

    @Getter
    private final IElementTree tree;

    @Getter
    private final IElementDefinitionAccessor snapshotElement;

    @Getter
    private final String localName;

    @Getter
    @Builder.Default
    private final Optional<String> sliceName = Optional.empty();

    @Getter
    @Builder.Default
    private final Optional<IElementDefinitionAccessor> differentialElement = Optional.empty();

    @Getter
    @Builder.Default
    private final Optional<IElementDefinitionAccessor> baseElement = Optional.empty();

    @Getter
    private final ElementBaseRelationship baseRelationship;

    @Getter
    @Builder.Default
    private final Optional<IElementTreeNode> parent = Optional.empty();

    private final List<IElementTreeNode> childNodes = new ArrayList<>();
    private final List<IElementTreeNode> sliceNodes = new ArrayList<>();

    @SuppressWarnings("java:S4738") // Java supplier does not support memoization
    @EqualsAndHashCode.Exclude
    private final Supplier<String> idSupplier = Suppliers.memoize(() -> {
        logger.entry();
        final var id = getSnapshotElement().getId();
        if (id.isPresent()) {
            return logger.exit(id.get());
        } else {
            throw logger.throwing(new IllegalStateException("Element.id is missing"));
        }
    });

    @Override
    public String getId() {
        return this.idSupplier.get();
    }

    private enum NodeListKey {
        CHILDREN,
        SLICES
    }

    @EqualsAndHashCode.Exclude
    private final LoadingCache<NodeListKey, ImmutableList<IElementTreeNode>> nodeListCache = CacheBuilder.newBuilder()
            .build(new CacheLoader<NodeListKey, ImmutableList<IElementTreeNode>>() {
                @Override
                public ImmutableList<IElementTreeNode> load(NodeListKey key) throws Exception {
                    logger.entry(key);
                    switch (key) {
                        case CHILDREN:
                            return logger.exit(ImmutableList.copyOf(ElementTreeNode.this.childNodes));
                        case SLICES:
                            return logger.exit(ImmutableList.copyOf(ElementTreeNode.this.sliceNodes));
                        default:
                            throw logger.throwing(new IllegalArgumentException());
                    }
                }
            });

    @Override
    public boolean hasChildren() {
        return !this.childNodes.isEmpty();
    }

    @Override
    public ImmutableList<IElementTreeNode> getChildren() {
        logger.entry();
        try {
            return logger.exit(this.nodeListCache.get(NodeListKey.CHILDREN));
        } catch (final ExecutionException e) {
            throw logger.throwing(new RuntimeException("Unable to create immutable list of children", e));
        }
    }

    @EqualsAndHashCode.Exclude
    private final LoadingCache<String, Optional<IElementTreeNode>> childByLocalNameCache = CacheBuilder.newBuilder()
            .build(new CacheLoader<String, Optional<IElementTreeNode>>() {
                @Override
                public Optional<IElementTreeNode> load(String key) throws Exception {
                    logger.entry(key);
                    return logger.exit(getChildren().stream()
                            .filter(n -> n.getLocalName().equals(key))
                            .findFirst());
                }
            });

    @Override
    public Optional<IElementTreeNode> getChild(String localName) {
        logger.entry(localName);
        try {
            return logger.exit(this.childByLocalNameCache.get(localName));
        } catch (final ExecutionException e) {
            throw logger.throwing(new RuntimeException("Unable to access cache of children by localName", e));
        }
    }

    void addChild(IElementTreeNode node) {
        logger.entry(node);
        this.childNodes.add(node);
        this.nodeListCache.invalidate(NodeListKey.CHILDREN);
        this.childByLocalNameCache.invalidate(node.getLocalName());
        logger.exit();
    }

    @Override
    public boolean hasSlices() {
        return !this.sliceNodes.isEmpty();
    }

    @Override
    public ImmutableList<IElementTreeNode> getSlices() {
        logger.entry();
        try {
            return logger.exit(this.nodeListCache.get(NodeListKey.SLICES));
        } catch (final ExecutionException e) {
            throw logger.throwing(new RuntimeException("Unable to create immutable list of slices", e));
        }
    }

    @EqualsAndHashCode.Exclude
    private final LoadingCache<String, Optional<IElementTreeNode>> sliceByNameCache = CacheBuilder.newBuilder()
            .build(new CacheLoader<String, Optional<IElementTreeNode>>() {
                @Override
                public Optional<IElementTreeNode> load(String key) throws Exception {
                    logger.entry(key);
                    return logger.exit(getSlices().stream()
                            .filter(n -> n.getSliceName().orElse("").equals(key))
                            .findFirst());
                }
            });

    @Override
    public Optional<IElementTreeNode> getSlice(String name) {
        logger.entry(this.localName);
        try {
            return logger.exit(this.sliceByNameCache.get(name));
        } catch (final ExecutionException e) {
            throw logger.throwing(new RuntimeException("Unable to access cache of slices by name", e));
        }
    }

    void addSlice(IElementTreeNode node) {
        logger.entry(node);
        final var nodeSliceName = node.getSliceName()
                .orElseThrow(() -> new IllegalArgumentException("Nodes added as slice must have a slice name set"));
        this.sliceNodes.add(node);
        this.nodeListCache.invalidate(NodeListKey.SLICES);
        this.sliceByNameCache.invalidate(nodeSliceName);
        logger.exit();
    }

    @Override
    public String toString() {
        return "ElementTreeNode [snapshotElement=" + this.snapshotElement + ", localName=" + this.localName
                + ", sliceName=" + this.sliceName + "]";
    }
}
