/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.fhir.elements;

import java.util.ArrayDeque;
import java.util.Deque;
import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * An iterator that performs a <a href=
 * "https://en.wikipedia.org/wiki/Tree_traversal#Pre-order,_NLR">depth-first
 * pre-order</a> traversal of an {@link IElementTree}.
 */
public final class ElementTreeIterator implements Iterator<IElementTreeNode> {

    private final Deque<IElementTreeNode> stack = new ArrayDeque<>();

    /**
     * Constructor used by {@link ElementTree}.
     * @param rootNode the root node to start the iteration from.
     */
    ElementTreeIterator(IElementTreeNode rootNode) {
        this.stack.addFirst(rootNode);
    }

    @Override
    public boolean hasNext() {
        return !this.stack.isEmpty();
    }

    @Override
    public IElementTreeNode next() {
        if (this.stack.isEmpty()) {
            throw new NoSuchElementException();
        }
        final var nextElement = this.stack.removeFirst();
        nextElement.getChildren().reverse().forEach(this.stack::addFirst);
        return nextElement;
    }
}
