/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.fhir.elements;

import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.IAccessorProvider;
import de.fhlintstone.accessors.UnresolvableURIException;
import de.fhlintstone.accessors.implementations.IFrameworkTypeLocator;
import de.fhlintstone.accessors.implementations.ITypeSpecification;
import de.fhlintstone.accessors.model.IElementDefinitionAccessor;
import de.fhlintstone.accessors.model.IStructureDefinitionAccessor;
import de.fhlintstone.fhir.FhirException;
import de.fhlintstone.fhir.IStructureDefinitionIntrospector;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Named;
import lombok.extern.slf4j.XSlf4j;

/**
 * Default implementation of {@link IElementTreeBuilder}.
 */
@Named
@XSlf4j
public class ElementTreeBuilder implements IElementTreeBuilder {

    private final IAccessorProvider accessorProvider;
    private final IStructureDefinitionIntrospector structureDefinitionIntrospector;
    private final IFrameworkTypeLocator frameworkTypeLocator;

    /**
     * Constructor for dependency injection
     *
     * @param accessorProvider                the {@link IAccessorProvider} to use
     * @param structureDefinitionIntrospector the {@link IStructureDefinitionIntrospector} to use
     * @param frameworkTypeLocator            the {@link IFrameworkTypeLocator} to use
     */
    @Inject
    public ElementTreeBuilder(
            IAccessorProvider accessorProvider,
            IStructureDefinitionIntrospector structureDefinitionIntrospector,
            IFrameworkTypeLocator frameworkTypeLocator) {
        super();
        this.accessorProvider = accessorProvider;
        this.structureDefinitionIntrospector = structureDefinitionIntrospector;
        this.frameworkTypeLocator = frameworkTypeLocator;
    }

    @Override
    public IElementTree buildElementTree(IStructureDefinitionAccessor structureDefinition) {
        logger.entry(structureDefinition);
        final var tree = new ElementTree(structureDefinition);
        final var rootNode = buildTreeInternal(tree, structureDefinition, Optional.empty());
        tree.setRootNode(rootNode);
        return logger.exit(tree);
    }

    @Override
    public IElementTree buildElementTree(
            IStructureDefinitionAccessor structureDefinition, IStructureDefinitionAccessor baseStructure) {
        logger.entry(structureDefinition, baseStructure);
        final var tree = new ElementTree(structureDefinition, baseStructure);
        final var rootNode = buildTreeInternal(tree, structureDefinition, Optional.of(baseStructure));
        tree.setRootNode(rootNode);
        return logger.exit(tree);
    }

    private IElementTreeNode buildTreeInternal(
            ElementTree tree,
            IStructureDefinitionAccessor structureDefinition,
            Optional<IStructureDefinitionAccessor> baseStructure) {
        logger.entry();
        logger.debug(
                "Building element tree for StructureDefinition {}",
                structureDefinition.getUrl().orElse("(no URL)"));
        validateStructures(structureDefinition, baseStructure);

        // build some maps (Element ID --> Element) to look up the elements later on
        final var elementLookupTables = new ElementLookupTables(structureDefinition, baseStructure);
        final var snapshotElements =
                structureDefinition.getSnapshot().orElseThrow().getElement();

        ElementTreeNode rootNode = null;
        for (final var snapshotElement : snapshotElements) {

            final var elementId = snapshotElement.getId().orElseThrow();
            final var differentialElement = elementLookupTables.getDifferentialElement(elementId);
            final var baseElement = elementLookupTables.getBaseElement(elementId);
            if (rootNode == null) {
                // first element: create root node
                logger.debug("Creating root node {}", elementId);
                final var baseRelationship = determineBaseRelationship(
                        structureDefinition, elementLookupTables, snapshotElement, Optional.empty());
                rootNode = ElementTreeNode.builder()
                        .withTree(tree)
                        .withSnapshotElement(snapshotElement)
                        // local name and ID are identical for the root element
                        .withLocalName(elementId)
                        .withDifferentialElement(differentialElement)
                        .withBaseElement(baseElement)
                        .withBaseRelationship(baseRelationship)
                        .build();
            } else {
                // remaining element: traverse the tree to locate the parent element, then
                // create the node there
                logger.debug("Creating node {}", elementId);
                final var insertionPoint = determineInsertionPoint(rootNode, snapshotElement);
                final var baseRelationship = determineBaseRelationship(
                        structureDefinition, elementLookupTables, snapshotElement, Optional.of(insertionPoint));
                logger.debug(
                        "New node {} is a {} named {} of node {} with relationship {}",
                        elementId,
                        insertionPoint.mode(),
                        insertionPoint.subId(),
                        insertionPoint.parent().getId(),
                        baseRelationship);
                addElementToTree(
                        tree, insertionPoint, snapshotElement, differentialElement, baseElement, baseRelationship);
            }
        }

        return logger.exit(rootNode);
    }

    private void validateStructures(
            IStructureDefinitionAccessor structureDefinition, Optional<IStructureDefinitionAccessor> baseStructure) {
        logger.entry();

        // current structure: snapshot is required
        if (structureDefinition.getSnapshot().isEmpty()) {
            throw new IllegalArgumentException(String.format(
                    "No snapshot is provided for StructureDefinition %s",
                    structureDefinition
                            .getUrl()
                            .orElse(structureDefinition.getName().orElse("???"))));
        }

        // if a base structure is given, a snapshot is required
        if (baseStructure.isPresent() && baseStructure.get().getSnapshot().isEmpty()) {
            final var baseStructureDefinition = baseStructure.get();
            throw new IllegalArgumentException(String.format(
                    "No snapshot is provided for base StructureDefinition %s",
                    baseStructureDefinition
                            .getUrl()
                            .orElse(baseStructureDefinition.getName().orElse("???"))));
        }

        // if a base structure is given, the differential is required as well
        if (baseStructure.isPresent() && structureDefinition.getDifferential().isEmpty()) {
            throw new IllegalArgumentException(String.format(
                    "No differential is provided for StructureDefinition %s",
                    structureDefinition
                            .getUrl()
                            .orElse(structureDefinition.getName().orElse("???"))));
        }

        logger.exit();
    }

    /**
     * Names the collection a new tree node is inserted into.
     */
    private enum InsertionMode {
        CHILD,
        SLICE;
    }

    /**
     * Information where and how to insert a new node into the element tree.
     */
    private record InsertionPoint(ElementTreeNode parent, String subId, InsertionMode mode) {}

    /**
     * Adds an element to the tree of {@link ElementTreeNode}s.
     */
    private InsertionPoint determineInsertionPoint(
            ElementTreeNode currentNode, IElementDefinitionAccessor snapshotElement) {
        logger.entry(currentNode, snapshotElement);

        // important background information:
        // . https://hl7.org/fhir/R4/elementdefinition.html#id
        // - https://fire.ly/blog/element-identifiers-in-fhir/
        // Assumption: the preferred identifier format
        // "elementName[:sliceName].elementName[:sliceName]..." is used.

        final var currentId = currentNode.getId();
        final var newId = snapshotElement.getId().orElseThrow();
        if (newId.startsWith(currentId)) {
            // cut off the path up to the current element
            var newSubId = newId.substring(currentId.length());

            // evaluate and cut off leading separator
            final var isChild = newSubId.startsWith(".");
            final var isSlice = newSubId.startsWith(":");
            newSubId = newSubId.substring(1);

            // extract the next path element
            newSubId = determineLeadingIdFragment(newSubId);

            InsertionPoint insertionPoint;

            if (isChild) {
                final var child = currentNode.getChild(newSubId);
                if (child.isPresent()) {
                    logger.trace("Handing element {} to existing child {}", newId, newSubId);
                    insertionPoint = determineInsertionPoint((ElementTreeNode) child.get(), snapshotElement);
                } else {
                    logger.trace("Locating element {} as child {} of {}", newId, newSubId, currentId);
                    insertionPoint = new InsertionPoint(currentNode, newSubId, InsertionMode.CHILD);
                }
            } else if (isSlice) {
                final var slice = currentNode.getSlice(newSubId);
                if (slice.isPresent()) {
                    logger.trace("Handing element {} to existing slice {}", newId, newSubId);
                    insertionPoint = determineInsertionPoint((ElementTreeNode) slice.get(), snapshotElement);
                } else {
                    logger.trace("Locating element {} as slice {} of {}", newId, newSubId, currentId);
                    insertionPoint = new InsertionPoint(currentNode, newSubId, InsertionMode.SLICE);
                }
            } else {
                throw logger.throwing(new IllegalStateException(String.format(
                        "Element with id %s arrived at id %s, but can not be identified as either slice or child (unknown separator)",
                        newId, currentId)));
            }
            return logger.exit(insertionPoint);
        } else {
            throw logger.throwing(new IllegalStateException(String.format(
                    "Element with id %s arrived at id %s with completely different path - don't know how to handle this.",
                    newId, currentId)));
        }
    }

    /**
     * Determines the base relationship of the element.
     * @param insertionPoint
     * @see ElementBaseRelationship
     */
    private ElementBaseRelationship determineBaseRelationship(
            final IStructureDefinitionAccessor structureDefinition,
            final ElementLookupTables elementLookupTables,
            final IElementDefinitionAccessor snapshotElement,
            Optional<InsertionPoint> insertionPoint) {
        logger.entry(snapshotElement);
        final var elementId = snapshotElement.getId().orElseThrow();
        final var insertionMode = insertionPoint.map(InsertionPoint::mode).orElse(InsertionMode.CHILD);
        final var parent = insertionPoint.map(InsertionPoint::parent).orElse(null);

        ElementBaseRelationship baseRelationship;
        if (elementLookupTables.hasBaseStructure()) {
            // The element tree is being built using a base tree.
            if (!elementLookupTables.hasBaseElement(elementId)) {
                // The snapshot element is not present in the base tree.
                if (insertionMode == InsertionMode.CHILD) {
                    // This is a child node in the element tree that is not present in the base tree.
                    if (parent == null) {
                        baseRelationship = ElementBaseRelationship.ADDED;
                    } else {
                        baseRelationship = determineBaseRelationshipByParent(
                                elementId, elementLookupTables, parent, structureDefinition);
                    }
                } else {
                    // This is a new slice in the element tree.
                    baseRelationship = ElementBaseRelationship.ADDED;
                }
            } else if (elementLookupTables.hasDifferentialElement(elementId)) {
                // The element is present in the base tree and in the differential list, so we assume that it has been
                // changed.
                baseRelationship = ElementBaseRelationship.CHANGED;
            } else {
                // The element is present in the base tree, but not present in the differential list, so we assume that
                // is unchanged.
                baseRelationship = ElementBaseRelationship.UNCHANGED;
            }
        } else {
            // The element tree is being built without a base tree, so no relationship can be determined.
            baseRelationship = ElementBaseRelationship.NO_BASE;
        }
        return logger.exit(baseRelationship);
    }

    private ElementBaseRelationship determineBaseRelationshipByParent(
            String elementId,
            ElementLookupTables elementLookupTables,
            ElementTreeNode parent,
            IStructureDefinitionAccessor structureDefinition) {
        logger.entry(elementId, parent);
        final var parentTypeStructures = getTypesOfTreeNode(parent, structureDefinition);
        final ElementBaseRelationship baseRelationship;

        if (!parentTypeStructures.isEmpty()) {
            final var parentTypesElementIndex = new ArrayList<String>();
            // for slices on anonymous local types, we need to check against the element lookup tables
            if (parent.getSliceName().isPresent() && isAnonymousLocalType(parentTypeStructures)) {
                final var sliceReplacePattern = Pattern.compile(":\\w*(?=\\.)");
                final var elementIdWithoutSliceNames =
                        sliceReplacePattern.matcher(elementId).replaceAll("");

                if (!elementLookupTables.hasBaseElement(elementIdWithoutSliceNames)) {
                    baseRelationship = ElementBaseRelationship.ADDED;
                } else if (elementLookupTables.hasDifferentialElement(elementId)) {
                    baseRelationship = ElementBaseRelationship.CHANGED;
                } else {
                    baseRelationship = ElementBaseRelationship.UNCHANGED;
                }
                return logger.exit(baseRelationship);
            }

            // The following pattern removes the leading element of the ID, i.e. it turns
            // HumanName.id into .id. This is necessary because when complex types are used,
            // the type name (HumanName) is replaced with the attribute name in the using
            // structure.
            final var idShorteningPattern = Pattern.compile("^\\w+(\\..*)$");
            for (final var structure : parentTypeStructures) {
                final var parentSnapshotElements =
                        structure.getSnapshot().orElseThrow().getElement();
                parentTypesElementIndex.addAll(parentSnapshotElements.stream()
                        .skip(1) // the first element describes the parent type itself
                        .map(element -> element.getId().orElseThrow())
                        .map(id -> {
                            final var elementIdMatcher = idShorteningPattern.matcher(id);
                            return elementIdMatcher.matches() ? elementIdMatcher.group(1) : elementId;
                        })
                        .toList());
            }

            // If the actual parent structure is not the structure the ElementTreeBuilder is
            // working on (e.g. the ETB is working on a logical model named test-model and
            // therefore the parentTypeStructure points to
            // http://hl7.org/fhir/StructureDefinition/Element), we won't find the element
            // by simply looking for the element ID (because test-model.id does not match
            // element.id). We have to remove the name of the original structure from the
            // element ID for this to work.
            final var searchIdMatcher = idShorteningPattern.matcher(elementId);
            final var searchId = searchIdMatcher.matches() ? searchIdMatcher.group(1) : elementId;

            // If the parent element index does not contain the element, it must be new
            if (parentTypesElementIndex.stream().noneMatch(searchId::endsWith)) {
                baseRelationship = ElementBaseRelationship.ADDED;
            } else if (elementLookupTables.hasDifferentialElement(elementId)) {
                baseRelationship = ElementBaseRelationship.CHANGED;
            } else {
                baseRelationship = ElementBaseRelationship.UNCHANGED;
            }
        } else {
            baseRelationship = ElementBaseRelationship.ADDED;
        }

        return logger.exit(baseRelationship);
    }

    private ImmutableList<IStructureDefinitionAccessor> getTypesOfTreeNode(
            ElementTreeNode treeNode, IStructureDefinitionAccessor structureDefinition) {
        logger.entry(treeNode);

        final var snapshotElement = treeNode.getSnapshotElement();
        final var elementId = snapshotElement.getId().orElse("(unknown)");

        final ImmutableCollection<ITypeSpecification> elementTypes;
        try {
            elementTypes = this.structureDefinitionIntrospector.getTypeSpecifications(snapshotElement);
        } catch (final FhirException e) {
            logger.error("Could not determine types of {}", elementId, e);
            return logger.exit(ImmutableList.of());
        }

        List<IStructureDefinitionAccessor> typeStructureDefinitions;
        if (elementTypes.isEmpty()) {
            // Quoting https://hl7.org/fhir/R4/elementdefinition-definitions.html#ElementDefinition.type:
            // "The Type of the element can be left blank in a differential constraint,
            // in which case the type is inherited from the resource."
            // Not sure about the first part of that statement - we're seeing blank types in snapshots as
            // well - but what's important that we have to fall back to the type of the resource...

            // ...but that's technically not correct either. For a StructureDefinition with derivation=constraint,
            // the type refers to the parent (constrained) type, which is what we want for our purposes.
            // For derivation=specialization, however, the type attribute *defines* the new type, so using the
            // structure itself would lead us to miscategorize added fields as existing fields. We have to
            // use the baseDefinition instead.

            final var baseDefinition = structureDefinition.getBaseDefinition();
            if (baseDefinition.isPresent()) {
                try {
                    final var baseDefinitionURI =
                            this.frameworkTypeLocator.makeAbsoluteStructureDefinitionReference(baseDefinition.get());
                    final var baseDefinitionAccessor =
                            this.accessorProvider.provideStructureDefinitionAccessor(baseDefinitionURI);
                    logger.debug(
                            "No type set for element {}, falling back to base definition {}",
                            elementId,
                            baseDefinitionURI);
                    typeStructureDefinitions = List.of(baseDefinitionAccessor);
                } catch (UnresolvableURIException | URISyntaxException e) {
                    logger.warn(
                            "Unable to resolve baseDefinition {}, falling back to {}",
                            baseDefinition.get(),
                            structureDefinition,
                            e);
                    typeStructureDefinitions = List.of(structureDefinition);
                }
            } else {
                logger.debug(
                        "No type set for element {} and no baseDefinition present, falling back to parent resource {}",
                        elementId,
                        structureDefinition);
                typeStructureDefinitions = List.of(structureDefinition);
            }
        } else {
            typeStructureDefinitions = loadStructureDefinitionForTypes(elementTypes);
        }
        return logger.exit(ImmutableList.copyOf(typeStructureDefinitions));
    }

    private List<IStructureDefinitionAccessor> loadStructureDefinitionForTypes(Collection<ITypeSpecification> types) {
        logger.entry(types);
        final var typeStructureDefintions = new ArrayList<IStructureDefinitionAccessor>();

        for (final var type : types) {
            if (type.hasProfiles()) {
                for (final var profile : type.getProfiles()) {
                    try {
                        typeStructureDefintions.add(
                                this.accessorProvider.provideStructureDefinitionAccessor(profile.canonical()));
                    } catch (final IllegalArgumentException e) {
                        logger.warn(
                                "Could not load Structure Definition for profile {}, {}",
                                profile.canonical(),
                                e.getMessage());
                    }
                }
            }
            if (typeStructureDefintions.isEmpty()) {
                // no usable profiles set, use the type code
                try {
                    typeStructureDefintions.add(this.accessorProvider.provideStructureDefinitionAccessor(
                            type.getTypeCode().canonical()));
                } catch (final IllegalArgumentException e) {
                    logger.warn(
                            "Could not load Structure Definition for type {}, {}", type.getTypeCode(), e.getMessage());
                }
            }
        }
        if (typeStructureDefintions.isEmpty()) {
            logger.warn("Type does contain neither a profile or type code, skipping");
        }
        return logger.exit(typeStructureDefintions);
    }

    /**
     * Adds an element to the tree of {@link ElementTreeNode}s.
     */
    private void addElementToTree(
            ElementTree tree,
            InsertionPoint insertionPoint,
            IElementDefinitionAccessor snapshotElement,
            Optional<IElementDefinitionAccessor> differentialElement,
            Optional<IElementDefinitionAccessor> baseElement,
            ElementBaseRelationship baseRelationship) {
        logger.entry(insertionPoint.parent(), snapshotElement);
        final InsertionMode mode = insertionPoint.mode();
        if (mode == InsertionMode.CHILD) {
            logger.trace(
                    "Adding element {} as child {} of {}",
                    snapshotElement.getId().orElseThrow(),
                    insertionPoint.subId(),
                    insertionPoint.parent().getId());
            final var childNode = ElementTreeNode.builder()
                    .withTree(tree)
                    .withSnapshotElement(snapshotElement)
                    .withLocalName(insertionPoint.subId())
                    .withDifferentialElement(differentialElement)
                    .withBaseElement(baseElement)
                    .withBaseRelationship(baseRelationship)
                    .withParent(Optional.of(insertionPoint.parent()))
                    .build();
            insertionPoint.parent().addChild(childNode);
        } else if (mode == InsertionMode.SLICE) {
            logger.trace(
                    "Adding element {} as slice {} of {}",
                    snapshotElement.getId().orElseThrow(),
                    insertionPoint.subId(),
                    insertionPoint.parent().getId());
            final var sliceNode = ElementTreeNode.builder()
                    .withTree(tree)
                    .withSnapshotElement(snapshotElement)
                    .withLocalName(insertionPoint.parent().getLocalName())
                    .withSliceName(Optional.of(insertionPoint.subId()))
                    .withDifferentialElement(differentialElement)
                    .withBaseElement(baseElement)
                    .withBaseRelationship(baseRelationship)
                    .withParent(Optional.of(insertionPoint.parent()))
                    .build();
            insertionPoint.parent().addSlice(sliceNode);
        }
        logger.exit();
    }

    /**
     * Determines the leading element in an element path using the preferred identifier format
     * "elementName[:sliceName].elementName[:sliceName]...".
     */
    private String determineLeadingIdFragment(String currentId) {
        logger.entry(currentId);
        final var dotPosition = currentId.indexOf('.');
        final var colonPosition = currentId.indexOf(':');
        if ((dotPosition < 0) && (colonPosition < 0)) {
            // neither separator found, keep subId as is
        } else if ((dotPosition >= 0) && (colonPosition < 0)) {
            currentId = currentId.substring(0, dotPosition);
        } else if ((dotPosition < 0) && (colonPosition >= 0)) {
            currentId = currentId.substring(0, colonPosition);
        } else {
            // that is, ((dotPosition >= 0) && (colonPosition >= 0))
            currentId = currentId.substring(0, Math.min(dotPosition, colonPosition));
        }
        return logger.exit(currentId);
    }

    private boolean isAnonymousLocalType(List<IStructureDefinitionAccessor> typeStructures) {
        // Getting the Java class name for the HAPI BackboneElement type
        // Comparing it to the BackboneElement type of the type structures, assuming that they are named the same
        final var backboneTypePath =
                this.frameworkTypeLocator.getBackboneElementType().toString().split("\\.");
        final var backboneTypeName = backboneTypePath[backboneTypePath.length - 1];
        // A structure definition is an anonymous local type, if it has only one type and this type is a BackboneElement
        return typeStructures.size() == 1
                && typeStructures.getFirst().getType().isPresent()
                && typeStructures.getFirst().getType().get().equals(backboneTypeName);
    }

    /**
     * Auxiliary class to provide a convenient way to lookup elements in various input structures by element ID.
     */
    @XSlf4j
    private static class ElementLookupTables {

        private final Map<String, IElementDefinitionAccessor> differentialElementIndex;
        private final Map<String, IElementDefinitionAccessor> baseElementIndex;

        /**
         * Creates a new element lookup table collection.
         * @param structureDefinition the current structure the tables are built for
         * @param baseStructure the base structure, if present
         */
        public ElementLookupTables(
                IStructureDefinitionAccessor structureDefinition,
                Optional<IStructureDefinitionAccessor> baseStructure) {
            logger.entry();
            final var snapshotElements =
                    structureDefinition.getSnapshot().orElseThrow().getElement();
            this.differentialElementIndex = baseStructure.isPresent()
                    ? structureDefinition.getDifferential().orElseThrow().getElement().stream()
                            .collect(Collectors.toMap(e -> e.getId().orElseThrow(), e -> e))
                    : new HashMap<>();
            this.baseElementIndex = baseStructure.isPresent()
                    ? baseStructure.orElseThrow().getSnapshot().orElseThrow().getElement().stream()
                            .collect(Collectors.toMap(e -> e.getId().orElseThrow(), e -> e))
                    : new HashMap<>();
            logger.debug(
                    "snapshot: {} entries, differential element index: {} entries, base element index: {} entries",
                    snapshotElements.size(),
                    this.differentialElementIndex.size(),
                    this.baseElementIndex.size());
            logger.exit();
        }

        /**
         * Determines whether a differential element with the given ID exists.
         * @param elementId the element ID
         * @return <code>true</code> if a differential element with the given ID exists
         */
        public boolean hasDifferentialElement(String elementId) {
            return this.differentialElementIndex.containsKey(elementId);
        }

        /**
         * Retrieves the differential element by ID, if it exists.
         * @param elementId the element ID
         * @return an {@link Optional} wrapping the element if it exists, an empty {@link Optional} otherwise
         */
        public Optional<IElementDefinitionAccessor> getDifferentialElement(String elementId) {
            return Optional.ofNullable(this.differentialElementIndex.get(elementId));
        }

        /**
         * Determines whether any base elements are present.
         * @return <code>true</code> if a base element exists
         */
        public boolean hasBaseStructure() {
            return !this.baseElementIndex.isEmpty();
        }

        /**
         * Determines whether a base element with the given ID exists.
         * @param elementId the element ID
         * @return <code>true</code> if a base element with the given ID exists
         */
        public boolean hasBaseElement(String elementId) {
            return this.baseElementIndex.containsKey(elementId);
        }

        /**
         * Retrieves the base element by ID, if it exists.
         * @param elementId the element ID
         * @return an {@link Optional} wrapping the element if it exists, an empty {@link Optional} otherwise
         */
        public Optional<IElementDefinitionAccessor> getBaseElement(String elementId) {
            return Optional.ofNullable(this.baseElementIndex.get(elementId));
        }
    }
}
