/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.fhir.elements;

import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import de.fhlintstone.accessors.model.IElementDefinitionAccessor;
import de.fhlintstone.accessors.model.IStructureDefinitionAccessor;
import java.util.Deque;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.XSlf4j;

/**
 * Default implementation of {@link IElementTree}.
 */
@EqualsAndHashCode
@XSlf4j
public final class ElementTree implements IElementTree {

    @Getter
    private final IStructureDefinitionAccessor structureDefinition;

    @Getter
    private final Optional<IStructureDefinitionAccessor> baseStructureDefinition;

    @Getter
    @Setter(AccessLevel.PACKAGE)
    private IElementTreeNode rootNode;

    /**
     * Constructor to use when no base structure is available.
     *
     * @param structureDefinition the {@link IStructureDefinitionAccessor}
     *                                    described by the tree
     */
    public ElementTree(IStructureDefinitionAccessor structureDefinition) {
        super();
        this.structureDefinition = structureDefinition;
        this.baseStructureDefinition = Optional.empty();
    }

    /**
     * Constructor to use when a base structure is available.
     *
     * @param structureDefinition     the
     *                                        {@link IStructureDefinitionAccessor}
     *                                        described by the tree
     * @param baseStructureDefinition the
     *                                        {@link IStructureDefinitionAccessor}
     *                                        of the base structure
     */
    public ElementTree(
            IStructureDefinitionAccessor structureDefinition, IStructureDefinitionAccessor baseStructureDefinition) {
        super();
        this.structureDefinition = structureDefinition;
        this.baseStructureDefinition = Optional.of(baseStructureDefinition);
    }

    @SuppressWarnings("java:S4738") // Java supplier does not support memoization
    @EqualsAndHashCode.Exclude
    private final Supplier<Map<IElementDefinitionAccessor, IElementTreeNode>> indexByElementSupplier =
            Suppliers.memoize(() -> {
                logger.entry();
                final var result = new HashMap<IElementDefinitionAccessor, IElementTreeNode>();
                visitAllNodes(node -> {
                    result.put(node.getSnapshotElement(), node);
                    final var differentialElement = node.getDifferentialElement();
                    if (differentialElement.isPresent()) {
                        result.put(differentialElement.get(), node);
                    }
                });
                return logger.exit(result);
            });

    @Override
    public IElementTreeNode getNode(IElementDefinitionAccessor element) throws IllegalArgumentException {
        logger.entry(element);
        final var index = this.indexByElementSupplier.get();
        if (index.containsKey(element)) {
            return logger.exit(index.get(element));
        } else {
            throw logger.throwing(new IllegalArgumentException("Element is not part of the element tree"));
        }
    }

    @SuppressWarnings("java:S4738") // Java supplier does not support memoization
    @EqualsAndHashCode.Exclude
    private final Supplier<Map<String, IElementTreeNode>> indexByIdSupplier = Suppliers.memoize(() -> {
        logger.entry();
        final var result = new HashMap<String, IElementTreeNode>();
        visitAllNodes(node -> result.put(node.getId(), node));
        return logger.exit(result);
    });

    @Override
    public Optional<IElementTreeNode> getNode(String id) {
        logger.entry(id);
        final var index = this.indexByIdSupplier.get();
        if (index.containsKey(id)) {
            return logger.exit(Optional.of(index.get(id)));
        } else {
            return logger.exit(Optional.empty());
        }
    }

    void visitAllNodes(Consumer<IElementTreeNode> visitor) {
        logger.entry();
        final Deque<IElementTreeNode> remainingNodes = new LinkedList<>();
        remainingNodes.add(this.rootNode);
        while (!remainingNodes.isEmpty()) {
            final var node = remainingNodes.removeFirst();
            remainingNodes.addAll(node.getChildren());
            remainingNodes.addAll(node.getSlices());
            visitor.accept(node);
        }
        logger.exit();
    }

    @Override
    public String toString() {
        return "ElementTree [structureDefinitionAccessor=" + this.structureDefinition + ", rootNode=" + this.rootNode
                + "]";
    }

    @Override
    public Iterator<IElementTreeNode> iterator() {
        return new ElementTreeIterator(this.rootNode);
    }
}
