/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.fhir.dependencies;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import de.fhlintstone.fhir.dependencies.IDependency.Origin;
import java.net.URI;
import java.util.Optional;

/**
 * A dependency graph of FHIR resources.
 */
public interface IDependencyGraph {

    /**
     * Returns the nodes that make up the dependency graph.
     * @return the nodes that make up the dependency graph
     */
    ImmutableList<IDependencyNode> getNodes();

    /**
     * Returns the node associated with a certain resource URI, if present
     * @param resourceURI the resource URI to search for
     * @return the node associated with the resource
     */
    Optional<IDependencyNode> getNode(URI resourceURI);

    /**
     * Returns the dependencies between the nodes of the graph.
     * @return the dependencies between the nodes of the graph
     */
    ImmutableList<IDependency> getDependencies();

    /**
     * Returns the dependencies (the outgoing relationships) of a node
     * @param dependencyNode the dependent
     * @return the list of dependencies
     */
    ImmutableList<IDependency> getDependencies(IDependencyNode dependencyNode);

    /**
     * Returns the dependents (the incoming relationships) of a node
     * @param dependencyNode the dependency
     * @return the list of dependents
     */
    ImmutableList<IDependency> getDependents(IDependencyNode dependencyNode);

    /**
     * Returns a set of all {@link Origin}s specified by incoming dependency relationships.
     * This answers the question "In which roles is this node referred to by other (dependent) nodes?"
     * @param dependencyNode the node to query
     * @return the set of {@link Origin}s
     */
    ImmutableSet<IDependency.Origin> getDependentOrigins(IDependencyNode dependencyNode);

    /**
     * Returns an ordered list to traverse the dependency graph from dependency to dependents, i.e. in the
     * order implementations would have to be generated.
     * @return the ordered list of nodes
     * @throws IllegalStateException if the list cannot be created, e.g. if the graph is not acyclic
     */
    ImmutableList<IDependencyNode> getOrderedNodeList() throws IllegalStateException;
}
