/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.fhir.dependencies;

import java.util.Optional;

/**
 * A relationship between two {@link IDependencyNode}, representing a dependency.
 */
public interface IDependency {

    /**
     * Describes the origin of the dependency, i.e. which attribute or relationship
     * motivates the dependency.
     */
    enum Origin {
        /**
         * The dependent <a href="https://hl7.org/fhir/structuredefinition-definitions.html">StructureDefinition</a>
         * uses the dependency as a <a href="https://hl7.org/fhir/structuredefinition-definitions.html#StructureDefinition.baseDefinition">baseDefinition</a>.
         */
        BASE_DEFINITION,

        /**
         * The ValueSet of a <a href="https://hl7.org/fhir/elementdefinition-definitions.html#ElementDefinition.binding.valueSet">binding</a>
         * of an <a href="https://hl7.org/fhir/elementdefinition-definitions.html">ElementDefinition</a>.
         */
        BINDING_VALUE_SET,

        /**
         * The <a href="https://hl7.org/fhir/elementdefinition-definitions.html#ElementDefinition.type.code">type code</a>
         * of an <a href="https://hl7.org/fhir/elementdefinition-definitions.html">ElementDefinition</a>.
         */
        TYPE_CODE,

        /**
         * The <a href="https://hl7.org/fhir/elementdefinition-definitions.html#ElementDefinition.type.profile">type profile</a>
         * of an <a href="https://hl7.org/fhir/elementdefinition-definitions.html">ElementDefinition</a>.
         */
        TYPE_PROFILE,
        /**
         * The <a href="https://hl7.org/fhir/elementdefinition-definitions.html#ElementDefinition.type.targetProfile">type target profile</a>
         * of an <a href="https://hl7.org/fhir/elementdefinition-definitions.html">ElementDefinition</a>.
         */
        TYPE_TARGET_PROFILE,

        /**
         * The <a href="https://hl7.org/fhir/valueset-definitions.html#ValueSet.compose.include.system">include CodeSystem</a>
         * or <a href="https://hl7.org/fhir/valueset-definitions.html#ValueSet.compose.exclude.system">exclude CodeSystem</a>
         * of <a href="https://hl7.org/fhir/valueset-definitions.html">ValueSet</a>.
         */
        COMPOSE_CODE_SYSTEM,

        /**
         * The <a href="https://hl7.org/fhir/valueset-definitions.html#ValueSet.compose.include.valueSet">include ValueSet</a>
         * or <a href="https://hl7.org/fhir/valueset-definitions.html#ValueSet.compose.exclude.valueSet">exclude ValueSet</a>
         * of <a href="https://hl7.org/fhir/valueset-definitions.html">ValueSet</a>.
         */
        COMPOSE_VALUE_SET,

        /**
         * @see <a href= "https://hl7.org/fhir/R4/codesystem-definitions.html#CodeSystem.supplements">CodeSystem.supplements</a>
         */
        CODE_SYSTEM_SUPPLEMENTS,

        /**
         * @see <a href="https://hl7.org/fhir/R4/codesystem-definitions.html#CodeSystem.valueSet">CodeSystem.valueSet</a>
         */
        CODE_SYSTEM_VALUE_SET;
    }

    /**
     * Returns the dependent, i.e. the node that depends on the dependency node.
     *
     * @return the dependent
     */
    IDependencyNode getDependent();

    /**
     * Returns the dependency, i.e. the node that the dependent depends upon.
     *
     * @return the dependency
     */
    IDependencyNode getDependency();

    /**
     * Return the origin of the dependency, i.e. which attribute or relationship
     * motivates the dependency.
     *
     * @return the origin of the dependency.
     */
    Origin getOrigin();

    /**
     * Returns the fallback dependency if it exists. This can be set to store an
     * alternate implementation if the actual dependency cannot be satisfied (i.e.
     * use a "raw" Coding if a ValueSet implementation is not available).
     *
     * @return the fallback dependency if it exists
     */
    Optional<IDependencyNode> getFallback();
}
