/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.fhir.dependencies;

import com.google.common.base.Strings;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.accessors.dependencies.IDependencyResourceVisitor;
import de.fhlintstone.accessors.implementations.IFrameworkTypeLocator;
import de.fhlintstone.fhir.IResourceUtilities;
import de.fhlintstone.fhir.dependencies.IDependency.Origin;
import de.fhlintstone.packages.FhirResourceType;
import de.fhlintstone.packages.IPackageRegistry;
import java.net.URI;
import java.util.ArrayList;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import javax.inject.Inject;
import javax.inject.Named;
import lombok.extern.slf4j.XSlf4j;
import org.hl7.fhir.instance.model.api.IBaseResource;

/**
 * Default implementation of {@link IDependencyGraphBuilder}.
 */
@Named
@XSlf4j
public class DependencyGraphBuilder implements IDependencyGraphBuilder {

    private final IPackageRegistry packageRegistry;
    private final IResourceUtilities resourceUtilities;
    private final IFrameworkTypeLocator frameworkTypeLocator;
    private final IDependencyResourceVisitor dependencyResourceVisitor;

    private final List<IDependencyNode> resourceNodes = new ArrayList<>();

    /**
     * Constructor for dependency injection.
     *
     * @param packageRegistry           the {@link IPackageRegistry} to use
     * @param resourceUtilities         the {@link IResourceUtilities} instance to
     *                                  use
     * @param frameworkTypeLocator      the {@link IFrameworkTypeLocator} to use
     * @param dependencyResourceVisitor the {@link IDependencyResourceVisitor} to
     *                                  use
     */
    @Inject
    public DependencyGraphBuilder(
            IPackageRegistry packageRegistry,
            IResourceUtilities resourceUtilities,
            IFrameworkTypeLocator frameworkTypeLocator,
            IDependencyResourceVisitor dependencyResourceVisitor) {
        super();
        this.packageRegistry = packageRegistry;
        this.resourceUtilities = resourceUtilities;
        this.frameworkTypeLocator = frameworkTypeLocator;
        this.dependencyResourceVisitor = dependencyResourceVisitor;
    }

    @Override
    public void addResource(IBaseResource resource) throws DependencyException {
        logger.entry(resource);
        if (this.resourceNodes.stream()
                .anyMatch(n ->
                        n.getResource().isPresent() && n.getResource().get().equals(resource))) {
            throw new DependencyException(
                    String.format("Resource %s has already been added to the dependency graph", resource));
        }
        this.resourceNodes.add(createNode(resource));
        logger.exit();
    }

    @Override
    public boolean containsResource(URI canonicalURI) {
        return this.resourceNodes.stream()
                .anyMatch(n -> n.getResource().isPresent() && n.getResourceURI().equals(canonicalURI));
    }

    @Override
    public IDependencyGraph build() throws DependencyException {
        return new InternalDependencyGraphBuilder().build(this.resourceNodes);
    }

    private IDependencyNode createNode(URI resourceURI) {
        logger.entry(resourceURI);
        final var frameworkType = this.frameworkTypeLocator.determineType(resourceURI);
        return logger.exit(DependencyNode.builder()
                .withResourceURI(resourceURI)
                .withFrameworkType(frameworkType)
                .build());
    }

    private IDependencyNode createNode(IBaseResource resource) throws DependencyException {
        logger.entry(resource);
        try {
            final var uriContents = DependencyGraphBuilder.this.resourceUtilities.readResourceURI(resource);
            if (uriContents.uri().isEmpty()) {
                throw logger.throwing(new DependencyException("Unable to determine URI for resource"));
            }
            final var frameworkType = determineFrameworkType(resource);
            return logger.exit(DependencyNode.builder()
                    .withResource(Optional.of(resource))
                    .withResourceType(Optional.of(FhirResourceType.fromResource(resource)))
                    .withResourceURI(uriContents.uri().orElseThrow())
                    .withFrameworkType(frameworkType)
                    .build());
        } catch (final IllegalArgumentException e) {
            throw logger.throwing(new DependencyException("Unable to create node for resource", e));
        }
    }

    private Optional<TypeName> determineFrameworkType(IBaseResource resource) {
        logger.entry(resource);
        final var uriContents = this.resourceUtilities.readResourceURI(resource);
        final Optional<URI> uri = uriContents.uri();
        if (uri.isPresent()) {
            return this.frameworkTypeLocator.determineType(uri.get());
        }
        return logger.exit(Optional.empty());
    }

    private class InternalDependencyGraphBuilder implements IDependencyCollector {

        private final Map<URI, IDependencyNode> dependencyNodes = new HashMap<>();
        private final Set<IDependency> dependencies = new HashSet<>();

        private final Deque<IDependencyNode> remainingNodes = new LinkedList<>();

        private IDependencyNode currentNode = null;

        public IDependencyGraph build(List<IDependencyNode> initialNodes) throws DependencyException {
            logger.entry();
            logger.debug("Starting dependency resolution with {} initial dependencyNodes", initialNodes.size());
            for (final var initialNode : initialNodes) {
                this.remainingNodes.add(initialNode);
                this.dependencyNodes.put(initialNode.getResourceURI(), initialNode);
            }
            while (!this.remainingNodes.isEmpty()) {
                this.currentNode = this.remainingNodes.removeFirst();
                final URI resourceURI = this.currentNode.getResourceURI();
                logger.debug("Visiting resource {} to determine dependencies", resourceURI);
                final Optional<IBaseResource> resource = this.currentNode.getResource();
                if (resource.isPresent()) {
                    DependencyGraphBuilder.this.dependencyResourceVisitor.visit(resource.get(), this);
                } else {
                    logger.warn("Unable to retrieve resource for URI {}, no dependencies determined", resourceURI);
                }
            }
            return logger.exit(new DependencyGraph(this.dependencyNodes.values(), this.dependencies));
        }

        @Override
        public void collect(String dependency, Origin origin) throws DependencyException {
            collect(dependency, origin, null);
        }

        @Override
        public void collect(String dependency, Origin origin, String fallback) throws DependencyException {
            logger.entry(dependency, origin, fallback);
            if (this.currentNode == null) {
                throw new IllegalStateException("Current node not set (implementation error)");
            }
            logger.debug("Adding dependency {} --> {} ({})", this.currentNode.getResourceURI(), dependency, origin);
            final IDependencyNode dependencyNode = findOrCreateNode(dependency);
            final Optional<IDependencyNode> fallbackNode =
                    Strings.isNullOrEmpty(fallback) ? Optional.empty() : Optional.of(findOrCreateNode(fallback));
            this.dependencies.add(Dependency.builder()
                    .withDependent(this.currentNode)
                    .withDependency(dependencyNode)
                    .withOrigin(origin)
                    .withFallback(fallbackNode)
                    .build());
            logger.exit();
        }

        /**
         * Attempts to find an existing node for the resource URI. Will create a new
         * node if necessary and enqueue it for later visitation.
         *
         * @param resourceURI
         * @return
         * @throws DependencyException
         */
        private IDependencyNode findOrCreateNode(String resourceURI) throws DependencyException {
            logger.entry(resourceURI);

            final var uriContents = DependencyGraphBuilder.this.resourceUtilities.parseResourceURI(resourceURI);
            if (uriContents.uri().isEmpty()) {
                throw logger.throwing(
                        new DependencyException(String.format("Resource URI %s cannot be parsed", resourceURI)));
            }
            final var uri = uriContents.uri().orElseThrow();
            IDependencyNode dependencyNode;
            if (this.dependencyNodes.containsKey(uri)) {
                dependencyNode = this.dependencyNodes.get(uri);
            } else {
                final var dependencyResource = DependencyGraphBuilder.this.packageRegistry.getUniqueResource(uri);
                if (dependencyResource.isEmpty()) {
                    logger.warn("Unable to resolve resource {}", resourceURI);
                    dependencyNode = createNode(uri);
                } else {
                    dependencyNode = createNode(dependencyResource.get());
                }
                logger.debug("Created node for resource {}", uri);
                this.dependencyNodes.put(uri, dependencyNode);

                // do not descend the dependency path any further once we've found a HAPI
                // implementation for the type
                if (dependencyNode.getFrameworkType().isEmpty()) {
                    this.remainingNodes.addLast(dependencyNode);
                }
            }
            return dependencyNode;
        }
    }
}
