/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.fhir.dependencies;

import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Multimap;
import com.google.common.collect.MultimapBuilder;
import de.fhlintstone.fhir.dependencies.IDependency.Origin;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import lombok.EqualsAndHashCode;
import lombok.extern.slf4j.XSlf4j;

/**
 * Default implementation of {@link IDependencyGraph}.
 */
@EqualsAndHashCode
@XSlf4j
public final class DependencyGraph implements IDependencyGraph {

    private final ImmutableList<IDependencyNode> dependencyNodes;
    private final ImmutableList<IDependency> dependencies;

    /**
     * Creates a new {@link DependencyGraph}. Only intended to be used within the
     * package.
     *
     * @param dependencyNodes
     * @param dependencies
     */
    DependencyGraph(Collection<IDependencyNode> dependencyNodes, Collection<IDependency> dependencies) {
        super();
        this.dependencyNodes = ImmutableList.copyOf(dependencyNodes);
        this.dependencies = ImmutableList.copyOf(dependencies);
        checkConsistency();
    }

    @Override
    public ImmutableList<IDependencyNode> getNodes() {
        return this.dependencyNodes;
    }

    @SuppressWarnings("java:S4738") // Java supplier does not support memoization
    @EqualsAndHashCode.Exclude
    private final Supplier<Multimap<URI, IDependencyNode>> indexByResourceURISupplier = Suppliers.memoize(() -> {
        logger.entry();
        final Multimap<URI, IDependencyNode> map =
                MultimapBuilder.hashKeys().arrayListValues().build();
        for (final var node : this.getNodes()) {
            map.put(node.getResourceURI(), node);
        }
        return logger.exit(map);
    });

    @Override
    public Optional<IDependencyNode> getNode(URI resourceURI) {
        logger.entry(resourceURI);
        final var index = this.indexByResourceURISupplier.get();
        if (index.containsKey(resourceURI)) {
            final var entries = index.get(resourceURI);
            switch (entries.size()) {
                case 0:
                    return logger.exit(Optional.empty());
                case 1:
                    return logger.exit(entries.stream().findAny());
                default:
                    throw new IllegalStateException(
                            "Multiple dependencyNodes found for a single resource - something went wrong during the construction of the dependency graph");
            }
        } else {
            return logger.exit(Optional.empty());
        }
    }

    @Override
    public ImmutableList<IDependency> getDependencies() {
        return this.dependencies;
    }

    @SuppressWarnings("java:S4738") // Java supplier does not support memoization
    @EqualsAndHashCode.Exclude
    private final Supplier<ImmutableList<IDependencyNode>> orderedListSupplier = Suppliers.memoize(() -> {
        logger.entry();
        final var result = new ArrayList<IDependencyNode>();

        final var remainingNodes = new ArrayList<IDependencyNode>(this.getNodes());
        final var remainingDependencies = new ArrayList<IDependency>(this.getDependencies());

        while (!remainingNodes.isEmpty()) {
            final var dependentNodes = remainingDependencies.stream()
                    .map(IDependency::getDependent)
                    .collect(Collectors.toSet());
            final var processedNodes = new HashSet<IDependencyNode>();
            for (final var node : remainingNodes) {
                if (!dependentNodes.contains(node)) {
                    remainingDependencies.removeIf(dep -> dep.getDependency() == node);
                    result.add(node);
                    processedNodes.add(node);
                }
            }
            remainingNodes.removeAll(processedNodes);
            if (processedNodes.isEmpty()) {
                // no objects without dependencies were found in the last iteration
                logUnresolvedDependencies(remainingNodes, remainingDependencies);
                throw logger.throwing(new IllegalStateException(
                        "Unable to resolve any more dependencies - is the dependency graph acyclic?"));
            }
        }

        return logger.exit(ImmutableList.copyOf(result));
    });

    private void checkConsistency() {
        // sanity check: all dependencyNodes referred to by dependencies must be part of
        // the graph
        for (final var dep : this.dependencies) {
            if (!this.dependencyNodes.contains(dep.getDependency())) {
                throw new IllegalStateException(
                        String.format("%s contains dependency node that is not in the node list", dep));
            }
            if (!this.dependencyNodes.contains(dep.getDependent())) {
                throw new IllegalStateException(
                        String.format("%s contains dependent node that is not in the node list", dep));
            }
            final Optional<IDependencyNode> fallback = dep.getFallback();
            if (fallback.isPresent() && !this.dependencyNodes.contains(fallback.get())) {
                throw new IllegalStateException(
                        String.format("%s contains fallback node that is not in the node list", dep));
            }
        }
    }

    private void logUnresolvedDependencies(
            final ArrayList<IDependencyNode> remainingNodes, final ArrayList<IDependency> remainingDependencies) {
        logger.entry();
        logger.error("Unable to resolve any more dependencies - is the dependency graph acyclic?");
        logger.info("Remaining dependencyNodes:");
        var maxURILength = 0;
        for (final var node : remainingNodes) {
            logger.info("  {}", node.getResourceURI());
            maxURILength =
                    Math.max(maxURILength, node.getResourceURI().toString().length());
        }
        logger.info("Remaining dependencies:");
        remainingDependencies.sort((o1, o2) -> {
            final var c1 = o1.getDependent()
                    .getResourceURI()
                    .compareTo(o2.getDependent().getResourceURI());
            if (c1 == 0) {
                return o1.getDependency()
                        .getResourceURI()
                        .compareTo(o2.getDependency().getResourceURI());
            } else {
                return c1;
            }
        });
        for (final var dependency : remainingDependencies) {
            logger.info(
                    "  {} --> {}",
                    padString(dependency.getDependent().getResourceURI().toString(), maxURILength),
                    dependency.getDependency().getResourceURI().toString());
        }
        logger.exit();
    }

    @Override
    public ImmutableList<IDependencyNode> getOrderedNodeList() throws IllegalStateException {
        return this.orderedListSupplier.get();
    }

    @EqualsAndHashCode.Exclude
    private final LoadingCache<IDependencyNode, ImmutableList<IDependency>> nodeDependenciesCache =
            CacheBuilder.newBuilder().build(new CacheLoader<>() {
                @Override
                public ImmutableList<IDependency> load(IDependencyNode dependencyNode) throws Exception {
                    return ImmutableList.copyOf(DependencyGraph.this.dependencies.stream()
                            .filter(d -> d.getDependent() == dependencyNode)
                            .toList());
                }
            });

    @Override
    public ImmutableList<IDependency> getDependencies(IDependencyNode dependencyNode) {
        // TODO #7 add unit test
        try {
            return this.nodeDependenciesCache.get(dependencyNode);
        } catch (final ExecutionException e) {
            logger.error("Error determining dependencies of {}", dependencyNode.getResourceURI(), e);
            return ImmutableList.of();
        }
    }

    @EqualsAndHashCode.Exclude
    private final LoadingCache<IDependencyNode, ImmutableList<IDependency>> nodeDependentsCache =
            CacheBuilder.newBuilder().build(new CacheLoader<>() {
                @Override
                public ImmutableList<IDependency> load(IDependencyNode dependencyNode) throws Exception {
                    return ImmutableList.copyOf(DependencyGraph.this.dependencies.stream()
                            .filter(d -> d.getDependency() == dependencyNode)
                            .toList());
                }
            });

    @Override
    public ImmutableList<IDependency> getDependents(IDependencyNode dependencyNode) {
        // TODO #7 add unit test
        try {
            return this.nodeDependentsCache.get(dependencyNode);
        } catch (final ExecutionException e) {
            logger.error("Error determining dependents of {}", dependencyNode.getResourceURI(), e);
            return ImmutableList.of();
        }
    }

    @EqualsAndHashCode.Exclude
    private final LoadingCache<IDependencyNode, ImmutableSet<Origin>> nodeDependentsOriginsCache =
            CacheBuilder.newBuilder().build(new CacheLoader<>() {
                @Override
                public ImmutableSet<Origin> load(IDependencyNode dependencyNode) throws Exception {
                    return ImmutableSet.copyOf(DependencyGraph.this.nodeDependentsCache.get(dependencyNode).stream()
                            .map(d -> d.getOrigin())
                            .collect(Collectors.toSet()));
                }
            });

    @Override
    public ImmutableSet<Origin> getDependentOrigins(IDependencyNode dependencyNode) {
        // TODO #7 add unit test
        try {
            return this.nodeDependentsOriginsCache.get(dependencyNode);
        } catch (final ExecutionException e) {
            logger.error("Error determining origins of {}", dependencyNode.getResourceURI(), e);
            return ImmutableSet.of();
        }
    }

    private String padString(String input, int length) {
        final StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("%");
        stringBuilder.append(length);
        stringBuilder.append(".");
        stringBuilder.append(length);
        stringBuilder.append("s");
        return String.format(stringBuilder.toString(), input);
    }
}
