/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.fhir;

import ca.uhn.fhir.model.api.annotation.Block;
import ca.uhn.fhir.model.api.annotation.DatatypeDef;
import ca.uhn.fhir.model.api.annotation.ResourceDef;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Multimap;
import com.google.common.collect.MultimapBuilder;
import com.palantir.javapoet.ClassName;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.accessors.IAccessorProvider;
import de.fhlintstone.accessors.implementations.IFrameworkTypeLocator;
import de.fhlintstone.accessors.implementations.IMappedType;
import de.fhlintstone.accessors.implementations.ITypeSpecification;
import de.fhlintstone.accessors.implementations.MappedType;
import de.fhlintstone.accessors.implementations.TypeSpecification;
import de.fhlintstone.accessors.model.IBaseAccessor;
import de.fhlintstone.accessors.model.IElementDefinitionAccessor;
import de.fhlintstone.accessors.model.IPropertyAccessor;
import de.fhlintstone.accessors.model.IStructureDefinitionAccessor;
import de.fhlintstone.packages.FhirResourceType;
import de.fhlintstone.packages.IPackageRegistry;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;
import javax.inject.Inject;
import javax.inject.Named;
import lombok.extern.slf4j.XSlf4j;

/**
 * Default implementation of {@link IStructureDefinitionIntrospector}.
 */
@Named
@XSlf4j
public class StructureDefinitionIntrospector implements IStructureDefinitionIntrospector {

    private final IAccessorProvider accessorProvider;
    private final IPackageRegistry packageRegistry;
    private final IFrameworkTypeLocator frameworkTypeLocator;

    /**
     * Constructor for dependency injection.
     *
     * @param accessorProvider     the {@link IAccessorProvider} to use
     * @param packageRegistry      the {@link IPackageRegistry} to use
     * @param frameworkTypeLocator the {@link IFrameworkTypeLocator} to use
     */
    @Inject
    public StructureDefinitionIntrospector(
            IAccessorProvider accessorProvider,
            IPackageRegistry packageRegistry,
            IFrameworkTypeLocator frameworkTypeLocator) {
        super();
        this.accessorProvider = accessorProvider;
        this.packageRegistry = packageRegistry;
        this.frameworkTypeLocator = frameworkTypeLocator;
    }

    @Override
    public ImmutableCollection<ITypeSpecification> getTypeSpecifications(IElementDefinitionAccessor element)
            throws FhirException {
        logger.entry(element);
        final var elementId = element.getId().orElse("(no ID)");
        final var result = new ArrayList<ITypeSpecification>();
        for (final var accessor : element.getType()) {
            try {
                final var typeCode = accessor.getCode()
                        .orElseThrow(() -> logger.throwing(new FhirException(
                                String.format("Invalid ElementDefinition %s: type.code is missing", elementId))));
                final var typeCodeURI = this.frameworkTypeLocator.makeAbsoluteStructureDefinitionReference(typeCode);
                final var builder = TypeSpecification.builder()
                        .withTypeCode(new ITypeSpecification.TypeReference(typeCode, typeCodeURI));
                for (final var profile : accessor.getProfile()) {
                    final var value = profile.getValue();
                    if (value.isPresent()) {
                        final var profileURI =
                                this.frameworkTypeLocator.makeAbsoluteStructureDefinitionReference(value.get());
                        builder.withProfile(value.get(), profileURI);
                    }
                }
                for (final var targetProfile : accessor.getTargetProfile()) {
                    final var value = targetProfile.getValue();
                    if (value.isPresent()) {
                        final var profileURI =
                                this.frameworkTypeLocator.makeAbsoluteStructureDefinitionReference(value.get());
                        builder.withTargetProfile(value.get(), profileURI);
                    }
                }
                result.add(builder.build());
            } catch (final URISyntaxException e) {
                throw logger.throwing(
                        new FhirException(String.format("Invalid ElementDefinition %s: invalid URI", elementId), e));
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    @Override
    public ImmutableCollection<ITypeSpecification> getTypeSpecifications(IPropertyAccessor property)
            throws FhirException {
        logger.entry(property);
        try {
            final var typeCode = property.getTypeCode();
            final var typeCodeURI = this.frameworkTypeLocator.makeAbsoluteStructureDefinitionReference(typeCode);
            final var spec =
                    TypeSpecification.builder().withType(typeCode, typeCodeURI).build();
            return logger.exit(ImmutableList.of(spec));
        } catch (final URISyntaxException e) {
            throw logger.throwing(new FhirException(
                    String.format("Invalid property definition %s: invalid URI", property.getName()), e));
        }
    }

    @Override
    public ImmutableCollection<ITypeSpecification> getTypeSpecifications(IBaseAccessor base) throws FhirException {
        logger.entry(base);
        final var typeCode = base.getFhirType();
        if (typeCode.isPresent()) {
            try {
                final var typeCodeURI =
                        this.frameworkTypeLocator.makeAbsoluteStructureDefinitionReference(typeCode.get());
                final var spec = TypeSpecification.builder()
                        .withType(typeCode.get(), typeCodeURI)
                        .build();
                return logger.exit(ImmutableList.of(spec));
            } catch (final URISyntaxException e) {
                throw logger.throwing(new FhirException(
                        String.format(
                                "Invalid base definition %s: invalid URI",
                                base.getId().orElse("(no id)")),
                        e));
            }
        } else {
            return logger.exit(ImmutableList.of());
        }
    }

    @Override
    public ImmutableCollection<ITypeSpecification> getElementTypeSpecifications(
            IBaseAccessor accessor, String elementID) throws FhirException {
        logger.entry(accessor, elementID);
        if (accessor instanceof final IStructureDefinitionAccessor structureDefinition) {
            final var snapshot = structureDefinition.getSnapshot();
            if (snapshot.isPresent()) {
                final Optional<IElementDefinitionAccessor> element = snapshot.get().getElement().stream()
                        .filter(e -> e.getId().orElse("").equals(elementID))
                        .findAny();
                if (element.isPresent()) {
                    return logger.exit(getTypeSpecifications(element.get()));
                } else {
                    logger.warn(
                            "Element {} not found in StructureDefinition {}",
                            elementID,
                            structureDefinition.getUrl().orElse("?"));
                }
            } else {
                logger.warn(
                        "Missing snapshot for StructureDefinition {}",
                        structureDefinition.getUrl().orElse("?"));
            }
        } else {
            logger.warn(
                    "Method StructureDefinitionIntrospector.getElementTypeURIs called for non-StructureDefinition accessor of type {}",
                    accessor.getFhirType().orElse("?"));
        }
        return logger.exit(ImmutableList.of());
    }

    private final LoadingCache<IStructureDefinitionAccessor, ExtensionType> extensionTypeCache =
            CacheBuilder.newBuilder().build(new CacheLoader<IStructureDefinitionAccessor, ExtensionType>() {
                @Override
                public ExtensionType load(IStructureDefinitionAccessor accessor) throws Exception {
                    logger.entry(accessor);
                    final var url = accessor.getUrl()
                            .orElseThrow(() -> logger.throwing(
                                    new IllegalArgumentException("Unable to determine URL of StructureDefinition")));
                    final var type = accessor.getType()
                            .orElseThrow(() -> logger.throwing(
                                    new IllegalArgumentException("Unable to determine type of StructureDefinition")));
                    if (type.equals("Extension")) {

                        final var snapshot = accessor.getSnapshot();
                        final var differential = accessor.getDifferential();
                        if (snapshot.isPresent()) {
                            return logger.exit(getExtensionTypeFromSnapshot(accessor));
                        } else if (differential.isPresent()) {
                            return logger.exit(getExtensionTypeFromDifferential(accessor));
                        } else {
                            throw new IllegalArgumentException(String.format(
                                    "StructureDefinition %s does not have snapshot or differential view", url));
                        }
                    } else {
                        return logger.exit(ExtensionType.NONE);
                    }
                }

                private ExtensionType getExtensionTypeFromSnapshot(IStructureDefinitionAccessor accessor) {
                    logger.entry(accessor);
                    final var url = accessor.getUrl().orElseThrow();
                    final var elements = accessor.getSnapshot().orElseThrow().getElement();
                    if (elements.isEmpty()) {
                        throw logger.throwing(new IllegalStateException(
                                String.format("StructureDefinition %s has empty snapshot", url)));
                    } else {
                        // https://hl7.org/fhir/elementdefinition.html#min-max:
                        // "In StructureDefinition.snapshot: min and max are always required" -
                        // hence the liberal use of .orElseThrow
                        final var extensionsElement = elements.stream()
                                .filter(e -> e.getId().orElseThrow().equals("Extension.extension"))
                                .findFirst()
                                .orElseThrow(() -> logger.throwing(new IllegalArgumentException(String.format(
                                        "Extension.extension is missing in StructureDefinition %s", url))));
                        final var valueElement = elements.stream()
                                .filter(e -> e.getId().orElseThrow().equals("Extension.value[x]"))
                                .findFirst()
                                .orElseThrow(() -> logger.throwing(new IllegalArgumentException(String.format(
                                        "Extension.value[x] is missing in StructureDefinition %s", url))));
                        final var maxExtensions = extensionsElement.getMax().orElseThrow();
                        final var maxValues = valueElement.getMax().orElseThrow();
                        return mapMaxValuesToExtensionType(url, maxExtensions, maxValues);
                    }
                }

                private ExtensionType getExtensionTypeFromDifferential(IStructureDefinitionAccessor accessor) {
                    logger.entry(accessor);
                    final var url = accessor.getUrl().orElseThrow();
                    final var elements =
                            accessor.getDifferential().orElseThrow().getElement();
                    if (elements.isEmpty()) {
                        throw logger.throwing(new IllegalStateException(
                                String.format("StructureDefinition %s has empty differential", url)));
                    } else {
                        // https://hl7.org/fhir/elementdefinition.html#min-max:
                        // "If there is no StructureDefinition.baseDefinition: min and max are always
                        // required. Otherwise, in StructureDefinition.differential: min and max are
                        // always optional; if they are not present, they default to the min and max
                        // from the base definition."

                        // FIXME #6 implement the min/max access logic documented above
                        // for now, the values from Extension and Element are hard-coded here

                        final var extensionsElement = elements.stream()
                                .filter(e -> e.getId().orElseThrow().equals("Extension.extension"))
                                .findFirst();
                        final var maxExtensions = extensionsElement.isPresent()
                                ? extensionsElement.get().getMax().orElse("*")
                                : "*";

                        final var valueElement = elements.stream()
                                .filter(e -> e.getId().orElseThrow().equals("Extension.value[x]"))
                                .findFirst();
                        final var maxValues = valueElement.isPresent()
                                ? valueElement.get().getMax().orElse("1")
                                : "1";

                        return mapMaxValuesToExtensionType(url, maxExtensions, maxValues);
                    }
                }

                private ExtensionType mapMaxValuesToExtensionType(
                        String url, final String maxExtensions, final String maxValues) {
                    // simple extensions have the value[x] element unrestricted and the extension
                    // element restricted to ..0
                    if (!maxValues.equals("0") && maxExtensions.equals("0")) {
                        return logger.exit(ExtensionType.SIMPLE);
                    }

                    // complex extensions have the value[x] element restricted to ..0 and an
                    // unrestricted extension element
                    if (maxValues.equals("0") && !maxExtensions.equals("0")) {
                        return logger.exit(ExtensionType.COMPLEX);
                    }

                    // any other constellation should not occur
                    throw logger.throwing(new IllegalStateException(
                            String.format("Unable to determine whether extension %s is simple or complex", url)));
                }
            });

    @Override
    public ExtensionType getExtensionType(IStructureDefinitionAccessor structureDefinition) throws FhirException {
        try {
            return this.extensionTypeCache.get(structureDefinition);
        } catch (final ExecutionException e) {
            throw new FhirException(
                    String.format(
                            "Unable to determine extension type of StructureDefinition %s",
                            structureDefinition.getName().orElseThrow()),
                    e);
        }
    }

    @Override
    public Optional<ClassAnnotation> getClassAnnotationForFrameworkType(final TypeName frameworkType) {
        logger.entry(frameworkType);
        Optional<ClassAnnotation> classAnnotation = Optional.empty();
        if (frameworkType instanceof final ClassName frameworkClass) {
            final String frameworkClassName = frameworkClass.reflectionName();
            try {
                final var theClass = Class.forName(frameworkClassName);
                if (theClass.getAnnotation(ResourceDef.class) != null) {
                    classAnnotation = Optional.of(ClassAnnotation.RESOURCE);
                } else if (theClass.getAnnotation(DatatypeDef.class) != null) {
                    classAnnotation = Optional.of(ClassAnnotation.DATATYPE);
                } else if (theClass.getAnnotation(Block.class) != null) {
                    classAnnotation = Optional.of(ClassAnnotation.BLOCK);
                }
            } catch (final ClassNotFoundException e) {
                logger.warn("Unable to load framework class {}, assuming no annotation present", frameworkClassName, e);
            }
        } else {
            logger.warn("Unable to handle framework type {}, assuming no annotation present", frameworkType);
        }
        return logger.exit(classAnnotation);
    }

    @Override
    public Optional<ClassAnnotation> getClassAnnotation(IStructureDefinitionAccessor structureDefinition)
            throws FhirException {
        logger.entry(structureDefinition);
        switch (getExtensionType(structureDefinition)) {
            case COMPLEX:
                // complex extensions with generated classes have to be generated as Block
                // see #171: when tagged with @Datatype, an extraneous element is introduced
                return logger.exit(Optional.of(ClassAnnotation.BLOCK));
            case SIMPLE:
                // no class is generated for simple extensions, so no extensions are required either
                return logger.exit(Optional.empty());
            default: // covers ExtensionType.NONE
                // Inherit the annotation type from the parent type. In order to do so, we need
                // to trace back (possibly through multiple levels of derivation) to its
                // framework type.
                final var result = seachBaseStructureHierarchy(structureDefinition, sd -> {
                    final var uri = URI.create(sd.getUrl().orElseThrow());
                    final var frameworkType = this.frameworkTypeLocator.determineType(uri);
                    if (frameworkType.isPresent()) {
                        // If the implementation is a descendant of BackboneElement, we have to place a
                        // @Block annotation on it to have it serialized correctly
                        if (frameworkType.get().equals(this.frameworkTypeLocator.getBackboneElementType())) {
                            return Optional.of(ClassAnnotation.BLOCK);
                        } else {
                            return getClassAnnotationForFrameworkType(frameworkType.get());
                        }
                    } else {
                        return Optional.empty();
                    }
                });
                return logger.exit(result);
        }
    }

    @Override
    public Optional<TypeName> getFrameworkSuperclass(IStructureDefinitionAccessor structureDefinition)
            throws FhirException {
        logger.entry(structureDefinition);
        final var result = seachBaseStructureHierarchy(structureDefinition, sd -> {
            final var uri = URI.create(sd.getUrl().orElseThrow());
            return this.frameworkTypeLocator.determineType(uri);
        });
        return logger.exit(result);
    }

    @Override
    public ImmutableCollection<IMappedType> getFrameworkSuperclasses(Collection<ITypeSpecification> typeSpecifications)
            throws FhirException {
        logger.entry(typeSpecifications);
        final var result = new ArrayList<IMappedType>();

        // check all profile and type code references
        for (final var typeSpecification : typeSpecifications) {
            var superclassFound = false;
            // first check the profiles (which are ranked equally)
            final Multimap<TypeName, ITypeSpecification.TypeReference> profilesForTargetClass =
                    MultimapBuilder.linkedHashKeys().arrayListValues().build();
            for (final var profile : typeSpecification.getProfiles()) {
                final var profileAccessor =
                        this.accessorProvider.provideStructureDefinitionAccessor(profile.canonical());
                final var superclass = getFrameworkSuperclass(profileAccessor);
                if (superclass.isPresent()) {
                    logger.debug("Framework superclass {} found for profile {}", superclass.get(), profile.canonical());
                    profilesForTargetClass.put(superclass.get(), profile);
                    superclassFound = true;
                }
            }
            if (superclassFound) {
                for (final var superclass : profilesForTargetClass.keySet()) {
                    final var builder = MappedType.builder()
                            .withTypeCode(typeSpecification.getTypeCode().code())
                            .withTypeCodeURI(typeSpecification.getTypeCode().canonical());
                    for (final var profile : profilesForTargetClass.get(superclass)) {
                        builder.withProfile(Optional.of(profile.code()))
                                .withProfileURI(Optional.of(profile.canonical()));
                    }
                    result.add(builder.withType(superclass).build());
                }

            } else {
                // if no type was identified using the profiles, check the type code
                final var typeAccessor = this.accessorProvider.provideStructureDefinitionAccessor(
                        typeSpecification.getTypeCode().canonical());
                final var superclass = getFrameworkSuperclass(typeAccessor);
                if (superclass.isPresent()) {
                    logger.debug(
                            "Framework superclass {} found for type code {}",
                            superclass.get(),
                            typeSpecification.getTypeCode().canonical());
                    result.addAll(MappedType.forTypeSpecification(typeSpecification, superclass.get()));
                }
            }
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    /**
     * Traverse the derivation tree of a StructureDefinition upwards (following the baseDefinition towards Element) and apply an operation to each StructureDefinition in turn.
     * If the operation returns a value, that value is returned, otherwise the traversal is continued until no further baseDefinition can be found.
     */
    private <T> Optional<T> seachBaseStructureHierarchy(
            IStructureDefinitionAccessor structureDefinition,
            Function<IStructureDefinitionAccessor, Optional<T>> operation)
            throws FhirException {
        logger.entry(structureDefinition);
        var currentStructureDefinition = Optional.of(structureDefinition);
        while (currentStructureDefinition.isPresent()) {
            final var structureName = currentStructureDefinition.get().getName().orElse("(no name)");
            logger.debug("Checking StructureDefinition {}", structureName);
            final var result = operation.apply(currentStructureDefinition.get());
            if (result.isPresent()) {
                return logger.exit(result);
            } else {
                final var baseDefinition = currentStructureDefinition.get().getBaseDefinition();
                if (baseDefinition.isEmpty()) {
                    logger.debug("StructureDefinition {} does not have a baseDefinition", structureName);
                    return logger.exit(Optional.empty());
                } else {
                    final var baseDefinitionURI = URI.create(baseDefinition.get());
                    final var baseResource = this.packageRegistry.getUniqueResource(
                            FhirResourceType.STRUCTURE_DEFINITION, baseDefinitionURI);
                    if (baseResource.isPresent()) {
                        final var baseAccessor =
                                this.accessorProvider.provideStructureDefinitionAccessor(baseResource.get());
                        currentStructureDefinition = Optional.of(baseAccessor);
                    } else {
                        logger.warn(
                                "baseDefinition {} of StructureDefinition {} cannot be resolved",
                                baseDefinitionURI,
                                structureName);
                        currentStructureDefinition = Optional.empty();
                    }
                }
            }
        }
        return logger.exit(Optional.empty());
    }
}
