/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.fhir;

import ca.uhn.fhir.context.FhirVersionEnum;
import com.google.common.base.Strings;
import de.fhlintstone.accessors.UnsupportedFHIRVersionException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Optional;
import javax.inject.Named;
import lombok.extern.slf4j.XSlf4j;
import org.hl7.fhir.instance.model.api.IBaseResource;

/**
 * Auxiliary methods to work with {@link IBaseResource}s.
 */
@Named
@XSlf4j
public class ResourceUtilities implements IResourceUtilities {

    /**
     * Reads the URI and version information of a resource.
     *
     * @param resource the {@link IBaseResource} to examine
     * @return a {@link IResourceUtilities.ResourceURIContents} object containing
     *         the URI and version information, if present
     */
    @Override
    public ResourceURIContents readResourceURI(IBaseResource resource) {
        logger.entry(resource);
        final var uriString = readResourceURL(resource);
        final var resourceVersion = readResourceVersion(resource);
        if (uriString.isEmpty()) {
            return logger.exit(new ResourceURIContents(Optional.empty(), Optional.empty()));
        } else {
            final var uriStringWithoutPotentialVersion = uriString.get().split("\\|")[0];
            final var resultUri = uriStringtoURI(uriStringWithoutPotentialVersion);
            return logger.exit(new ResourceURIContents(Optional.of(resultUri), resourceVersion));
        }
    }

    /**
     * Read the URL string of a resource (which might contain a |x.y.z version
     * suffix!).
     *
     * @param resource the {@link IBaseResource} to examine
     * @return the URL string associated with the resource
     */
    @Override
    public Optional<String> readResourceURL(IBaseResource resource) {
        logger.entry(resource);
        return logger.exit(extractMetadataResourceProperty(resource, metadataResource -> {
            if (metadataResource instanceof org.hl7.fhir.r4.model.MetadataResource r4Meta) {
                return r4Meta.getUrl();
            }
            if (metadataResource instanceof org.hl7.fhir.r4b.model.MetadataResource r4bMeta) {
                return r4bMeta.getUrl();
            }
            if (metadataResource instanceof org.hl7.fhir.r5.model.MetadataResource r5Meta) {
                return r5Meta.getUrl();
            }
            return null;
        }));
    }

    @Override
    public ResourceURIContents parseResourceURI(String resourceURL) {
        logger.entry(resourceURL);
        Optional<URI> resultURI = Optional.empty();
        Optional<String> resultVersion = Optional.empty();
        var uriString = resourceURL;
        // split version suffix if present
        final var pipeIndex = uriString.indexOf("|");
        if (pipeIndex >= 0) {
            resultVersion = Optional.of(uriString.substring(pipeIndex + 1));
            uriString = uriString.substring(0, pipeIndex);
        }

        // parse the remaining URI
        if (!uriString.isEmpty()) {
            resultURI = Optional.of(uriStringtoURI(uriString));
        }
        return logger.exit(new ResourceURIContents(resultURI, resultVersion));
    }

    @Override
    public URI determineURIFromTypeCode(String typeCode) {
        // TODO #13 add unit tests for this method
        logger.entry(typeCode);
        String url = "";
        if (typeCode.startsWith("http://hl7.org/fhirpath/")) {
            // see https://build.fhir.org/fhirpath.html#types
            switch (typeCode) {
                case "http://hl7.org/fhirpath/System.Boolean" ->
                    url = "http://hl7.org/fhir/StructureDefinition/boolean";
                case "http://hl7.org/fhirpath/System.String" -> url = "http://hl7.org/fhir/StructureDefinition/string";
                case "http://hl7.org/fhirpath/System.Integer" ->
                    url = "http://hl7.org/fhir/StructureDefinition/integer";
                case "http://hl7.org/fhirpath/System.Long" -> url = "http://hl7.org/fhir/StructureDefinition/integer64";
                case "http://hl7.org/fhirpath/System.Decimal" ->
                    url = "http://hl7.org/fhir/StructureDefinition/decimal";
                case "http://hl7.org/fhirpath/System.DateTime" ->
                    url = "http://hl7.org/fhir/StructureDefinition/dateTime";
                case "http://hl7.org/fhirpath/System.Time" -> url = "http://hl7.org/fhir/StructureDefinition/time";
                case "http://hl7.org/fhirpath/System.Quantity" ->
                    url = "http://hl7.org/fhir/StructureDefinition/Quantity";
                default ->
                    throw logger.throwing(
                            new IllegalArgumentException(String.format("Unknown FHIRpath type %s", typeCode)));
            }
        } else if (!typeCode.startsWith("http")) {
            url = "http://hl7.org/fhir/StructureDefinition/" + typeCode;
        } else {
            url = typeCode;
        }
        final var result = URI.create(url);
        return logger.exit(result);
    }

    private Optional<String> readResourceVersion(IBaseResource resource) {
        logger.entry(resource);
        return logger.exit(extractMetadataResourceProperty(resource, metadataResource -> {
            if (metadataResource instanceof org.hl7.fhir.r4.model.MetadataResource r4Meta) {
                return r4Meta.getVersion();
            }
            if (metadataResource instanceof org.hl7.fhir.r4b.model.MetadataResource r4bMeta) {
                return r4bMeta.getVersion();
            }
            if (metadataResource instanceof org.hl7.fhir.r5.model.MetadataResource r5Meta) {
                return r5Meta.getVersion();
            }
            return null;
        }));
    }

    private Optional<String> extractMetadataResourceProperty(
            IBaseResource resource, java.util.function.Function<Object, String> propertyExtractor) {
        logger.entry(resource);
        String propertyValue = "";
        final FhirVersionEnum fhirVersion = resource.getStructureFhirVersionEnum();
        if (fhirVersion == null) {
            throw logger.throwing(new IllegalArgumentException("No FHIR version reported for resource"));
        } else {
            switch (fhirVersion) {
                case R4 -> {
                    if (resource instanceof final org.hl7.fhir.r4.model.MetadataResource metadataResource) {
                        propertyValue = propertyExtractor.apply(metadataResource);
                    }
                }
                case R4B -> {
                    if (resource instanceof final org.hl7.fhir.r4b.model.MetadataResource metadataResource) {
                        propertyValue = propertyExtractor.apply(metadataResource);
                    }
                }
                case R5 -> {
                    if (resource instanceof final org.hl7.fhir.r5.model.MetadataResource metadataResource) {
                        propertyValue = propertyExtractor.apply(metadataResource);
                    }
                }
                default -> throw logger.throwing(new UnsupportedFHIRVersionException(fhirVersion));
            }
        }
        if (Strings.isNullOrEmpty(propertyValue)) {
            return logger.exit(Optional.empty());
        } else {
            return logger.exit(Optional.of(propertyValue));
        }
    }

    private URI uriStringtoURI(String uriString) {
        try {
            return new URI(uriString);
        } catch (final URISyntaxException e) {
            throw logger.throwing(
                    new IllegalArgumentException(String.format("Malformed resource URI: %s", uriString), e));
        }
    }
}
