/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.fhir;

import com.google.common.collect.ImmutableCollection;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.accessors.implementations.IMappedType;
import de.fhlintstone.accessors.implementations.ITypeSpecification;
import de.fhlintstone.accessors.model.IBaseAccessor;
import de.fhlintstone.accessors.model.IElementDefinitionAccessor;
import de.fhlintstone.accessors.model.IPropertyAccessor;
import de.fhlintstone.accessors.model.IStructureDefinitionAccessor;
import java.util.Collection;
import java.util.Optional;

/**
 * Provides several complex operations to gather information on FHIR
 * StructureDefinitions.
 */
public interface IStructureDefinitionIntrospector {

    /**
     * Determines the type specifications contained in an
     * <a href="https://hl7.org/fhir/R4/elementdefinition.html#ElementDefinition">ElementDefinition</a>.
     *
     * @param element an accessor for the ElementDefinition
     * @return a list of type specifications corresponding to the contents of the ElementDefinition
     * @throws FhirException in case of an invalid ElementDefinition
     */
    ImmutableCollection<ITypeSpecification> getTypeSpecifications(IElementDefinitionAccessor element)
            throws FhirException;

    /**
     * Determines the type specifications contained in a property.
     * @see IPropertyAccessor
     *
     * @param property an accessor for the property
     * @return a list of type specifications corresponding to the contents of the property
     * @throws FhirException in case of an invalid property
     */
    ImmutableCollection<ITypeSpecification> getTypeSpecifications(IPropertyAccessor property) throws FhirException;

    /**
     * Determines the type specifications describing a base element.
     * @see IBaseAccessor
     *
     * @param base an accessor for the element
     * @return a list of type specifications corresponding to the contents of the element
     * @throws FhirException in case of an invalid property
     */
    ImmutableCollection<ITypeSpecification> getTypeSpecifications(IBaseAccessor base) throws FhirException;

    /**
     * Returns the type specifications for a specific element ID, assuming that the accessor refers to a StructureDefinition.
     *
     * @param accessor the accessor to examine
     * @param elementID the element ID to extract the type for
     * @return the type specifications for the specific element ID
     * @throws FhirException
     */
    ImmutableCollection<ITypeSpecification> getElementTypeSpecifications(IBaseAccessor accessor, String elementID)
            throws FhirException;

    /**
     * Determines the {@link ExtensionType} of a StructureDefinition.
     *
     * @param structureDefinition an accessor for the StructureDefinition to examine
     * @return the {@link ExtensionType} of the StructureDefinition
     * @throws FhirException
     */
    ExtensionType getExtensionType(IStructureDefinitionAccessor structureDefinition) throws FhirException;

    /**
     * Determines the {@link ClassAnnotation} from an existing framework type.
     *
     * @param frameworkType the type to examine
     * @return the {@link ClassAnnotation}, if any
     */
    Optional<ClassAnnotation> getClassAnnotationForFrameworkType(TypeName frameworkType);

    /**
     * Determines the {@link ClassAnnotation} used for a FHIR StructureDefinition.
     *
     * @param structureDefinition an accessor for the StructureDefinition to examine
     * @return the {@link ClassAnnotation}, if any
     * @throws FhirException
     */
    Optional<ClassAnnotation> getClassAnnotation(IStructureDefinitionAccessor structureDefinition) throws FhirException;

    /**
     * Identifies the first (as in, most specialized) base definition that is represented by a HAPI framework type. If
     * the StructureDefinition specified as a parameter is represented by a framework type, that type is returned and no
     * superclass determination takes place.
     *
     * @param structureDefinition the StructureDefinition to examine
     * @return the {@link TypeName} of the framework class, if identifiable
     * @throws FhirException
     */
    Optional<TypeName> getFrameworkSuperclass(IStructureDefinitionAccessor structureDefinition) throws FhirException;

    /**
     * Tries to determine the possible framework superclasses from a set of type specifications. Only the first (as in,
     * most specialized) framework classes are returned. Since the type specifications might point to different type
     * codes, the method returns a list of mapped types. Note that the type specifications might point to generated or
     * framework classes.
     *
     * @see #getFrameworkSuperclass(IStructureDefinitionAccessor)
     * @param typeSpecifications the type specifications to examine
     * @return the list of possible framework superclasses, if any could be found
     * @throws FhirException
     */
    ImmutableCollection<IMappedType> getFrameworkSuperclasses(Collection<ITypeSpecification> typeSpecifications)
            throws FhirException;
}
