/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.fhir;

import de.fhlintstone.accessors.implementations.IFrameworkTypeLocator;
import java.net.URI;
import java.util.Optional;
import org.hl7.fhir.instance.model.api.IBaseResource;

/**
 * Auxiliary methods to work with {@link IBaseResource}s.
 */
public interface IResourceUtilities {

    /**
     * Structure used to transfer the values extracted from the resource URI (may
     * also contain a version).
     *
     * @param uri     the URI specified by the resource, if any
     * @param version the version suffix contained in the URI, if any
     */
    public record ResourceURIContents(Optional<URI> uri, Optional<String> version) {}

    /**
     * Reads the URI and version information of a resource.
     *
     * @param resource the {@link IBaseResource} to examine
     * @return a {@link ResourceURIContents} object containing the URI and version
     *         information, if present
     */
    ResourceURIContents readResourceURI(IBaseResource resource);

    /**
     * Parses the URL string of a resource (which might contain a |x.y.z version
     * suffix) and extracts the URI and version information..
     *
     * @param resourceURL the string to parse
     * @return a {@link ResourceURIContents} object containing the URI and version
     *         information, if present
     */
    ResourceURIContents parseResourceURI(String resourceURL);

    /**
     * Read the URL string of a resource (which might contain a |x.y.z version
     * suffix!).
     *
     * @param resource the {@link IBaseResource} to examine
     * @return the URL string associated with the resource
     */
    Optional<String> readResourceURL(IBaseResource resource);

    /**
     * Attempts to determine the URI to access a StructureDefinition from the type
     * code. This method will resolve FHIR types without the URL prefix (like
     * boolean) and exchange the FHRIpath types.
     *
     * <b>Caution:</b> It can not be assumed that the URI is correct in all cases,
     * nor that the corresponding resource can be located and loaded in all cases.
     *
     * @param typeCode
     * @return the URI determined from the type code
     * @deprecated should be merged into {@link IFrameworkTypeLocator} methods makeAbsolute...Reference
     */
    @Deprecated
    URI determineURIFromTypeCode(String typeCode);
}
