/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.fhir;

import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.model.IConceptDefinitionComponentAccessor;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import lombok.extern.slf4j.XSlf4j;

/**
 * Provides utilitiy functions for the handling of FHIR resources
 */
@XSlf4j
public final class FhirUtilities {

    private FhirUtilities() {}

    /**
     * Flatten a hierarchical concept tree (usually from a code system).
     * @param concepts A list of concepts on the first level of the hierachy to flatten
     * @return A flattened list of all concepts in the hierarchy
     */
    public static ImmutableList<IConceptDefinitionComponentAccessor> flattenConceptHierarchy(
            List<IConceptDefinitionComponentAccessor> concepts) {
        logger.entry(concepts);
        final var result = new ArrayList<IConceptDefinitionComponentAccessor>();
        final var queue = new ArrayDeque<>(concepts);

        while (!queue.isEmpty()) {
            final var current = queue.pop();
            result.add(current);
            queue.addAll(current.getChildConcept());
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    /**
     * Calculates the parent id of a given element id.
     * @param elementId The given element id.
     * @return The parent id of the given element id or empty if the provided element id doesn't have a parent
     */
    public static Optional<String> getParentId(String elementId) {
        logger.entry(elementId);
        final var childIndex = elementId.lastIndexOf(".");
        final var sliceIndex = elementId.lastIndexOf(":");
        final var lastElementIndex = Math.max(childIndex, sliceIndex);

        if (lastElementIndex <= 0) {
            return logger.exit(Optional.empty());
        }

        return logger.exit(Optional.of(elementId.substring(0, lastElementIndex)));
    }
}
