/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.model;

import de.fhlintstone.accessors.IAccessorCache;
import java.util.Optional;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import org.hl7.fhir.r4b.model.ValueSet;

/**
 * Implementation of {@link IValueSetAccessor} for FHIR version R4v. i.e.
 * wrapping {@link ValueSet}.
 */
@EqualsAndHashCode(callSuper = true)
public final class ValueSetAccessorR4B extends CanonicalResourceAccessorR4B implements IValueSetAccessor {

    @Getter(AccessLevel.PRIVATE)
    private final ValueSet valueSet;

    /**
     * Casting constructor.
     *
     * @param cache the cache this accessor is managed by
     * @param valueSet the ValueSet instance to wrap
     */
    public ValueSetAccessorR4B(IAccessorCache cache, ValueSet valueSet) {
        super(cache, valueSet);
        this.valueSet = valueSet;
    }

    @Override
    public Optional<String> getName() {
        if (this.valueSet.hasName()) {
            return Optional.of(this.valueSet.getName());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getUrl() {
        if (this.valueSet.hasUrl()) {
            return Optional.of(this.valueSet.getUrl());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getDescription() {
        if (this.valueSet.hasDescription()) {
            return Optional.of(this.valueSet.getDescription());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<IValueSetComposeComponentAccessor> getCompose() {
        if (this.valueSet.hasCompose()) {
            return Optional.of(
                    getCache().getAccessor(this.valueSet.getCompose(), IValueSetComposeComponentAccessor.class));
        } else {
            return Optional.empty();
        }
    }
}
