/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.model;

import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.IAccessorCache;
import java.util.Optional;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import org.hl7.fhir.r5.model.StructureDefinition;

/**
 * Implementation of {@link IStructureDefinitionAccessor} for FHIR version R5.
 * i.e. wrapping {@link StructureDefinition}.
 */
@EqualsAndHashCode(callSuper = true)
public final class StructureDefinitionAccessorR5 extends CanonicalResourceAccessorR5
        implements IStructureDefinitionAccessor {

    @Getter(AccessLevel.PRIVATE)
    private final StructureDefinition structureDefinition;

    /**
     * Creates a new wrapper.
     *
     * @param cache the cache this accessor is managed by
     * @param structureDefinition the StructureDefinition to wrap
     */
    public StructureDefinitionAccessorR5(IAccessorCache cache, StructureDefinition structureDefinition) {
        super(cache, structureDefinition);
        this.structureDefinition = structureDefinition;
    }

    @Override
    public Optional<String> getName() {
        if (this.structureDefinition.hasName()) {
            return Optional.of(this.structureDefinition.getName());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getUrl() {
        if (this.structureDefinition.hasUrl()) {
            return Optional.of(this.structureDefinition.getUrl());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getVersionedUrl() {
        if (this.structureDefinition.hasUrl()) {
            return Optional.of(this.structureDefinition.getVersionedUrl());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getDescription() {
        if (this.structureDefinition.hasDescription()) {
            return Optional.of(this.structureDefinition.getDescription());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getType() {
        if (this.structureDefinition.hasType()) {
            return Optional.of(this.structureDefinition.getType());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getBaseDefinition() {
        if (this.structureDefinition.hasBaseDefinition()) {
            return Optional.of(this.structureDefinition.getBaseDefinition());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<IStructureDefinitionSnapshotComponentAccessor> getSnapshot() {
        if (this.structureDefinition.hasSnapshot()) {
            return Optional.of(getCache()
                    .getAccessor(
                            this.structureDefinition.getSnapshot(),
                            IStructureDefinitionSnapshotComponentAccessor.class));
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<IStructureDefinitionDifferentialComponentAccessor> getDifferential() {
        if (this.structureDefinition.hasDifferential()) {
            return Optional.of(getCache()
                    .getAccessor(
                            this.structureDefinition.getDifferential(),
                            IStructureDefinitionDifferentialComponentAccessor.class));
        } else {
            return Optional.empty();
        }
    }

    @Override
    public ImmutableList<IExtensionAccessor> getExtensions() {
        return this.structureDefinition.getExtension().stream()
                .map(component -> getCache().getAccessor(component, IExtensionAccessor.class))
                .collect(ImmutableList.toImmutableList());
    }

    @Override
    public ImmutableList<IExtensionAccessor> getModifierExtensions() {
        return this.structureDefinition.getModifierExtension().stream()
                .map(component -> getCache().getAccessor(component, IExtensionAccessor.class))
                .collect(ImmutableList.toImmutableList());
    }
}
