/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.model;

import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.Accessor;
import de.fhlintstone.accessors.IAccessorCache;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import org.hl7.fhir.r4.model.Property;

/**
 * Implementation of {@link IBaseAccessor} for FHIR release R4.
 */
@EqualsAndHashCode(callSuper = true)
public final class PropertyAccessorR4 extends Accessor implements IPropertyAccessor {

    @Getter(AccessLevel.PRIVATE)
    private final Property property;

    /**
     * Creates a new wrapper.
     *
     * @param cache the cache this accessor is managed by
     * @param property the FHIR Property to wrap
     */
    public PropertyAccessorR4(IAccessorCache cache, Property property) {
        super(cache);
        this.property = property;
    }

    @Override
    public String getName() {
        return this.property.getName();
    }

    @Override
    public String getTypeCode() {
        return this.property.getTypeCode();
    }

    @Override
    public String getDefinition() {
        return this.property.getDefinition();
    }

    @Override
    public int getMinCardinality() {
        return this.property.getMinCardinality();
    }

    @Override
    public int getMaxCardinality() {
        return this.property.getMaxCardinality();
    }

    @Override
    public ImmutableList<IBaseAccessor> getValues() {
        return this.property.getValues().stream()
                .map(value -> getCache().getAccessor(value, IBaseAccessor.class))
                .collect(ImmutableList.toImmutableList());
    }

    @Override
    public boolean hasValues() {
        return this.property.hasValues();
    }

    @Override
    public boolean isList() {
        return this.property.isList();
    }

    @Override
    public String toString() {
        return this.property.toString();
    }
}
