/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.model;

import de.fhlintstone.accessors.IAccessorCache;
import de.fhlintstone.accessors.UnsupportedTypeException;
import java.util.Optional;
import lombok.EqualsAndHashCode;
import lombok.extern.slf4j.XSlf4j;
import org.hl7.fhir.r4b.model.Base64BinaryType;
import org.hl7.fhir.r4b.model.BooleanType;
import org.hl7.fhir.r4b.model.CanonicalType;
import org.hl7.fhir.r4b.model.CodeType;
import org.hl7.fhir.r4b.model.DateTimeType;
import org.hl7.fhir.r4b.model.DateType;
import org.hl7.fhir.r4b.model.DecimalType;
import org.hl7.fhir.r4b.model.IdType;
import org.hl7.fhir.r4b.model.InstantType;
import org.hl7.fhir.r4b.model.Integer64Type;
import org.hl7.fhir.r4b.model.IntegerType;
import org.hl7.fhir.r4b.model.MarkdownType;
import org.hl7.fhir.r4b.model.OidType;
import org.hl7.fhir.r4b.model.PositiveIntType;
import org.hl7.fhir.r4b.model.PrimitiveType;
import org.hl7.fhir.r4b.model.StringType;
import org.hl7.fhir.r4b.model.TimeType;
import org.hl7.fhir.r4b.model.UnsignedIntType;
import org.hl7.fhir.r4b.model.UriType;
import org.hl7.fhir.r4b.model.UrlType;
import org.hl7.fhir.r4b.model.UuidType;

/**
 * Implementation of {@link IPrimitiveTypeAccessor} for FHIR release R4B.
 * @param <T> the type parameter of the wrapped PrimitiveType
 */
@EqualsAndHashCode(callSuper = true)
@XSlf4j
public class PrimitiveTypeAccessorR4B<T> extends TypeAccessorR4B implements IPrimitiveTypeAccessor {

    private final PrimitiveType<T> primitiveType;

    /**
     * Creates a new wrapper.
     *
     * @param cache the cache this accessor is managed by
     * @param type the FHIR PrimitiveType to wrap
     */
    public PrimitiveTypeAccessorR4B(IAccessorCache cache, PrimitiveType<T> type) {
        super(cache, type);
        this.primitiveType = type;
    }

    @Override
    @SuppressWarnings("java:S1541") // this method is not complex, just long
    public boolean isType(PrimitiveDatatype checkType) {
        logger.entry(checkType);
        boolean result;
        switch (checkType) {
            case BASE64BINARY -> result = this.primitiveType instanceof Base64BinaryType;
            case BOOLEAN -> result = this.primitiveType instanceof BooleanType;
            case CANONICAL -> result = this.primitiveType instanceof CanonicalType;
            case CODE -> result = this.primitiveType instanceof CodeType;
            case DATE -> result = this.primitiveType instanceof DateType;
            case DATETIME -> result = this.primitiveType instanceof DateTimeType;
            case DECIMAL -> result = this.primitiveType instanceof DecimalType;
            case ID -> result = this.primitiveType instanceof IdType;
            case INSTANT -> result = this.primitiveType instanceof InstantType;
            case INTEGER64 -> result = this.primitiveType instanceof Integer64Type;
            case INTEGER -> result = this.primitiveType instanceof IntegerType;
            case MARKDOWN -> result = this.primitiveType instanceof MarkdownType;
            case OID -> result = this.primitiveType instanceof OidType;
            case POSITIVEINT -> result = this.primitiveType instanceof PositiveIntType;
            case STRING -> result = this.primitiveType instanceof StringType;
            case TIME -> result = this.primitiveType instanceof TimeType;
            case UNSIGNEDINT -> result = this.primitiveType instanceof UnsignedIntType;
            case URI -> result = this.primitiveType instanceof UriType;
            case URL -> result = this.primitiveType instanceof UrlType;
            case UUID -> result = this.primitiveType instanceof UuidType;
            default -> throw new IllegalArgumentException(
                    String.format("Type %s is currently not supported.", checkType));
        }
        return logger.exit(result);
    }

    @Override
    @SuppressWarnings("java:S1541") // this method is not really complex, just long due to the number of types
    public Optional<StringifiedValue> getStringifiedValue() throws UnsupportedTypeException {
        logger.entry();

        // shortcut if no value exists
        if (this.primitiveType.isEmpty()) {
            return logger.exit(Optional.empty());
        }

        // currently, all types use the same setter method
        final var value = StringifiedValue.builder().withSetterMethod("setValue");
        switch (this.primitiveType) {
            case final BooleanType booleanType -> value.withDatatype(PrimitiveDatatype.BOOLEAN)
                    .withValue(booleanType.getValue().toString())
                    .withLiteral(true);
            case final CanonicalType canonicalType -> value.withDatatype(PrimitiveDatatype.CANONICAL)
                    .withValue(canonicalType.getValue())
                    .withLiteral(false);
            case final CodeType codeType -> value.withDatatype(PrimitiveDatatype.CODE)
                    .withValue(codeType.getValue())
                    .withLiteral(false);
            case final DecimalType decimalType -> value.withDatatype(PrimitiveDatatype.DECIMAL)
                    .withValue(decimalType.getValue().toString())
                    .withLiteral(true);
            case final IdType idType -> value.withDatatype(PrimitiveDatatype.ID)
                    .withValue(idType.getValue())
                    .withLiteral(false);
            case final MarkdownType markdownType -> value.withDatatype(PrimitiveDatatype.MARKDOWN)
                    .withValue(markdownType.getValue())
                    .withLiteral(false);
            case final OidType oidType -> value.withDatatype(PrimitiveDatatype.OID)
                    .withValue(oidType.getValue())
                    .withLiteral(false);
            case final PositiveIntType positiveIntType -> value.withDatatype(PrimitiveDatatype.POSITIVEINT)
                    .withValue(positiveIntType.getValue().toString())
                    .withLiteral(true);
            case final UnsignedIntType unsignedIntType -> value.withDatatype(PrimitiveDatatype.UNSIGNEDINT)
                    .withValue(unsignedIntType.getValue().toString())
                    .withLiteral(true);
            case final UrlType urlType -> value.withDatatype(PrimitiveDatatype.URL)
                    .withValue(urlType.getValue())
                    .withLiteral(false);
            case final UuidType uuidType -> value.withDatatype(PrimitiveDatatype.UUID)
                    .withValue(uuidType.getValue())
                    .withLiteral(false);
            case final StringType stringType -> value.withDatatype(PrimitiveDatatype.STRING)
                    .withValue(stringType.getValue())
                    .withLiteral(false);
            case final Integer64Type integer64Type -> value.withDatatype(PrimitiveDatatype.INTEGER64)
                    .withValue(integer64Type.getValue().toString())
                    .withLiteral(true);
            case final IntegerType integerType -> value.withDatatype(PrimitiveDatatype.INTEGER)
                    .withValue(integerType.getValue().toString())
                    .withLiteral(true);
            case final UriType uriType -> value.withDatatype(PrimitiveDatatype.URI)
                    .withValue(uriType.getValue())
                    .withLiteral(false);
            default -> throw logger.throwing(UnsupportedTypeException.forStringConversion(this.primitiveType));
        }
        return Optional.ofNullable(value.build());
    }
}
