/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.model;

/**
 * This is a list of the <a href="https://build.fhir.org/datatypes.html#primitive">primitive FHIR data types</a>
 * that are currently supported by the {@link ITypeAccessor}.
 */
public enum PrimitiveDatatype {

    /**
     * A stream of bytes, base64 encoded (RFC 4648). base64Binary content does not
     * include any whitespace or line feeds, but reading applications should ignore
     * whitespace characters (per RFC 4648).
     *
     * @see <a href="https://build.fhir.org/datatypes.html#base64Binary">base64Binary</a>
     */
    BASE64BINARY,

    /**
     * true | false
     *
     * @see <a href="https://build.fhir.org/datatypes.html#boolean">boolean</a>
     */
    BOOLEAN,

    /**
     * A URI that refers to a resource by its canonical URL (resources with a url
     * property). The canonical type differs from a uri in that it has special
     * meaning in this specification, and in that it may have a version appended,
     * separated by a vertical bar (|). Note that the type canonical is not used for
     * the actual canonical URLs that are the target of these references, but for
     * the URIs that refer to them, and may have the version suffix in them. Like
     * other URIs, elements of type canonical may also have #fragment references.
     * Unlike other URIs, canonical URLs are never relative - they are either
     * absolute URIs, or fragment identifiers.
     *
     * @see <a href="https://build.fhir.org/datatypes.html#canonical">canonical</a>
     */
    CANONICAL,

    /**
     * Indicates that the value is taken from a set of controlled strings defined
     * elsewhere (see Using codes for further discussion). Technically, a code is
     * restricted to a string which has at least one character and no leading or
     * trailing whitespace, and where there is no whitespace other than single
     * spaces in the contents.
     *
     * @see <a href="https://build.fhir.org/datatypes.html#code">code</a>
     */
    CODE,

    /**
     * A date, or partial date (e.g. just year or year + month) as used in human
     * communication. The format is a subset of [ISO8601]: YYYY, YYYY-MM, or
     * YYYY-MM-DD, e.g. 2018, 1973-06, or 1905-08-23. There SHALL be no timezone
     * offset. Dates SHALL be valid dates.
     *
     * @see <a href="https://build.fhir.org/datatypes.html#date">date</a>
     */
    DATE,

    /**
     * A date, date-time or partial date (e.g. just year or year + month) as used in
     * human communication. The format is a subset of [ISO8601]: YYYY, YYYY-MM,
     * YYYY-MM-DD or YYYY-MM-DDThh:mm:ss+zz:zz, e.g. 2018, 1973-06, 1905-08-23,
     * 2015-02-07T13:28:17-05:00 or 2017-01-01T00:00:00.000Z. If hours and minutes
     * are specified, a timezone offset SHALL be populated. Actual timezone codes
     * can be sent using the Timezone Code extension, if desired. Seconds must be
     * provided due to schema type constraints but may be zero-filled and may be
     * ignored at receiver discretion. Milliseconds are optionally allowed. Dates
     * SHALL be valid dates. The time "24:00" is not allowed. Leap Seconds are
     * allowed - see below
     *
     * @see <a href="https://build.fhir.org/datatypes.html#dateTime">dateTime</a>
     */
    DATETIME,

    /**
     * Rational numbers that have a decimal representation. Decimals in FHIR cannot
     * have more than 18 digits and a decimal point. See below about the precision
     * of the number.
     *
     * @see <a href="https://build.fhir.org/datatypes.html#decimal">decimal</a>
     */
    DECIMAL,

    /**
     * Any combination of upper- or lower-case ASCII letters ('A'..'Z', and
     * 'a'..'z', numerals ('0'..'9'), '-' and '.', with a length limit of 64
     * characters. (This might be an integer, an un-prefixed OID, UUID or any other
     * identifier pattern that meets these constraints.)
     *
     * @see <a href="https://build.fhir.org/datatypes.html#id">id</a>
     */
    ID,

    /**
     * An instant in time in a format that is a subset of [ISO8601]:
     * YYYY-MM-DDThh:mm:ss.sss+zz:zz (e.g. 2015-02-07T13:28:17.239+02:00 or
     * 2017-01-01T00:00:00Z). The time SHALL specified at least to the second and
     * SHALL include a timezone offset. Actual timezone codes can be sent using the
     * [[[http://hl7.org/fhir/StructureDefinition/timezone extension]]], if desired.
     * Sub-millisecond precision is optionally allowed. Note: This is intended for
     * when precisely observed times are required (typically system logs etc.), and
     * not human-reported times - for those, use date or dateTime (which can be as
     * precise as instant, but is not required to be). instant is a more constrained
     * dateTime
     *
     * @see <a href="https://build.fhir.org/datatypes.html#instant">instant</a>
     */
    INSTANT,

    /**
     * A signed integer in the range -9,223,372,036,854,775,808 to
     * +9,223,372,036,854,775,807 (64-bit). This type is defined to allow for
     * record/time counters that can get very large.
     *
     * @see <a href="https://build.fhir.org/datatypes.html#integer64">integer64</a>
     */
    INTEGER64,

    /**
     * A signed integer in the range −2,147,483,648..2,147,483,647 (32-bit; for
     * larger values, use decimal).
     *
     * Note that this also comprises the primitive types {@link #UNSIGNEDINT} and
     * {@link #POSITIVEINT}.
     *
     * @see <a href="https://build.fhir.org/datatypes.html#integer">integer</a>
     */
    INTEGER,

    /**
     * A FHIR string (see below) that may contain markdown syntax for optional
     * processing by a markdown presentation engine, in the GFM extension of
     * CommonMark format (see below).
     *
     * @see <a href="https://build.fhir.org/datatypes.html#markdown">markdown</a>
     */
    MARKDOWN,

    /**
     * An OID represented as a URI (RFC 3001); e.g. urn:oid:1.2.3.4.5
     *
     * @see <a href="https://build.fhir.org/datatypes.html#oid">oid</a>
     */
    OID,

    /**
     * Any positive integer in the range 1..2,147,483,647.
     *
     * @see <a href=
     *      "https://build.fhir.org/datatypes.html#positiveInt">positiveInt</a>
     */
    POSITIVEINT,

    /**
     * A sequence of Unicode characters.
     *
     * Note that this also comprises the primitive types {@link #CODE}, {@link #ID}
     * and {@link #MARKDOWN}.
     *
     * @see <a href="https://build.fhir.org/datatypes.html#string">string</a>
     */
    STRING,

    /**
     * A time during the day, in the format hh:mm:ss (a subset of [ISO8601]). There
     * is no date specified. Seconds must be provided due to schema type constraints
     * but may be zero-filled and may be ignored at receiver discretion. The time
     * "24:00" SHALL NOT be used. A timezone offset SHALL NOT be present. Times can
     * be converted to a Duration since midnight.
     *
     * @see <a href="https://build.fhir.org/datatypes.html#time">time</a>
     */
    TIME,

    /**
     * Any non-negative integer in the range 0..2,147,483,647.
     *
     * @see <a href=
     *      "https://build.fhir.org/datatypes.html#unsignedInt">unsignedInt</a>
     */
    UNSIGNEDINT,

    /**
     * A Uniform Resource Identifier Reference (RFC 3986). Note: URIs are case
     * sensitive. For UUID (urn:uuid:53fefa32-fcbb-4ff8-8a92-55ee120877b7) use all
     * lowercase.
     *
     * Note that this also comprises the types {@link #URL}, {@link #CANONICAL},
     * {@link #UUID} and {@link #OID}.
     *
     * @see <a href="https://build.fhir.org/datatypes.html#uri">uri</a>
     */
    URI,

    /**
     * A Uniform Resource Locator (RFC 1738). Note URLs are accessed directly using
     * the specified protocol. Common URL protocols are http{s}:, ftp:, mailto: and
     * mllp:, though many others are defined.
     *
     * @see <a href="https://build.fhir.org/datatypes.html#url">url</a>
     */
    URL,

    /**
     * A UUID (aka GUID) represented as a URI (RFC 4122); e.g.
     * urn:uuid:c757873d-ec9a-4326-a141-556f43239520.
     *
     * @see <a href="https://build.fhir.org/datatypes.html#uuid">uuid</a>
     */
    UUID
}
