/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.model;

import com.google.common.collect.ImmutableList;
import java.util.Optional;

/**
 * Release-independent wrapper for a FHIR StructureDefinition.
 *
 * Note that in R4, the StructureDefinition is derived from MetadataResource while in
 * R4B and R5, it is derived from CanonicalResource.
 *
 * This interface and the corresponding implementations are used to make as much
 * of the rest of the core library release-independent, i.e. shield it from the
 * various HAPI FHIR release package differences.
 *
 * @see org.hl7.fhir.r4.model.StructureDefinition
 * @see org.hl7.fhir.r4b.model.StructureDefinition
 * @see org.hl7.fhir.r5.model.StructureDefinition
 */
public interface IStructureDefinitionAccessor extends IMetadataResourceAccessor {

    /**
     * Returns the name of the StructureDefinition, if set.
     *
     * @return the name of the StructureDefinition, if set
     * @see org.hl7.fhir.r4.model.StructureDefinition#hasName()
     * @see org.hl7.fhir.r4.model.StructureDefinition#getName()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#hasName()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#getName()
     * @see org.hl7.fhir.r5.model.StructureDefinition#hasName()
     * @see org.hl7.fhir.r5.model.StructureDefinition#getName()
     */
    Optional<String> getName();

    /**
     * Returns the URL of the StructureDefinition, if set.
     *
     * @return the URL of the StructureDefinition, if set
     * @see org.hl7.fhir.r4.model.StructureDefinition#hasUrl()
     * @see org.hl7.fhir.r4.model.StructureDefinition#getUrl()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#hasUrl()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#getUrl()
     * @see org.hl7.fhir.r5.model.StructureDefinition#hasUrl()
     * @see org.hl7.fhir.r5.model.StructureDefinition#getUrl()
     */
    Optional<String> getUrl();

    /**
     * Returns the versioned URL of the StructureDefinition, if set.
     *
     * @return The URL of the Structure Definition including the version suffix, if set
     * @see org.hl7.fhir.r4.model.StructureDefinition#hasUrl()
     * @see org.hl7.fhir.r4.model.StructureDefinition#getVersionedUrl()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#hasUrl()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#getVersionedUrl()
     * @see org.hl7.fhir.r5.model.StructureDefinition#hasUrl()
     * @see org.hl7.fhir.r5.model.StructureDefinition#getVersionedUrl()
     */
    Optional<String> getVersionedUrl();

    /**
     * Returns the description of the StructureDefinition, if set.
     *
     * @return the description of the StructureDefinition, if set
     * @see org.hl7.fhir.r4.model.StructureDefinition#hasDescription()
     * @see org.hl7.fhir.r4.model.StructureDefinition#getDescription()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#hasDescription()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#getDescription()
     * @see org.hl7.fhir.r5.model.StructureDefinition#hasDescription()
     * @see org.hl7.fhir.r5.model.StructureDefinition#getDescription()
     */
    Optional<String> getDescription();

    /**
     * Returns the type of the StructureDefinition, if set.
     *
     * @return the type of the StructureDefinition, if set
     * @see org.hl7.fhir.r4.model.StructureDefinition#hasType()
     * @see org.hl7.fhir.r4.model.StructureDefinition#getType()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#hasType()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#getType()
     * @see org.hl7.fhir.r5.model.StructureDefinition#hasType()
     * @see org.hl7.fhir.r5.model.StructureDefinition#getType()
     */
    Optional<String> getType();

    /**
     * Returns the baseDefinition of the StructureDefinition, if set.
     *
     * @return the baseDefinition of the StructureDefinition, if set
     * @see org.hl7.fhir.r4.model.StructureDefinition#hasBaseDefinition()
     * @see org.hl7.fhir.r4.model.StructureDefinition#getBaseDefinition()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#hasBaseDefinition()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#getBaseDefinition()
     * @see org.hl7.fhir.r5.model.StructureDefinition#hasBaseDefinition()
     * @see org.hl7.fhir.r5.model.StructureDefinition#getBaseDefinition()
     */
    Optional<String> getBaseDefinition();

    /**
     * Returns the snapshot structure of the StructureDefinition, if set.
     *
     * @return the snapshot structure of the StructureDefinition, if set
     * @see org.hl7.fhir.r4.model.StructureDefinition#hasSnapshot()
     * @see org.hl7.fhir.r4.model.StructureDefinition#getSnapshot()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#hasSnapshot()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#getSnapshot()
     * @see org.hl7.fhir.r5.model.StructureDefinition#hasSnapshot()
     * @see org.hl7.fhir.r5.model.StructureDefinition#getSnapshot()
     */
    Optional<IStructureDefinitionSnapshotComponentAccessor> getSnapshot();

    /**
     * Returns the differential structure of the StructureDefinition, if set.
     *
     * @return the differential structure of the StructureDefinition, if set
     * @see org.hl7.fhir.r4.model.StructureDefinition#hasDifferential()
     * @see org.hl7.fhir.r4.model.StructureDefinition#getDifferential()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#hasDifferential()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#getDifferential()
     * @see org.hl7.fhir.r5.model.StructureDefinition#hasDifferential()
     * @see org.hl7.fhir.r5.model.StructureDefinition#getDifferential()
     */
    Optional<IStructureDefinitionDifferentialComponentAccessor> getDifferential();

    /**
     * Returns the Extensions of the StructureDefinition, if any.
     *
     * @return the Extensions of the StructureDefinition, if any
     * @see org.hl7.fhir.r4.model.StructureDefinition#hasExtension()
     * @see org.hl7.fhir.r4.model.StructureDefinition#getExtension()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#hasExtension()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#getExtension()
     * @see org.hl7.fhir.r5.model.StructureDefinition#hasExtension()
     * @see org.hl7.fhir.r5.model.StructureDefinition#getExtension()
     */
    ImmutableList<IExtensionAccessor> getExtensions();

    /**
     * Returns the Modifier Extensions of the StructureDefinition, if any.
     *
     * @return the Modifier Extensions of the StructureDefinition, if any
     * @see org.hl7.fhir.r4.model.StructureDefinition#hasModifierExtension()
     * @see org.hl7.fhir.r4.model.StructureDefinition#getModifierExtension()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#hasModifierExtension()
     * @see org.hl7.fhir.r4b.model.StructureDefinition#getModifierExtension()
     * @see org.hl7.fhir.r5.model.StructureDefinition#hasModifierExtension()
     * @see org.hl7.fhir.r5.model.StructureDefinition#getModifierExtension()
     */
    ImmutableList<IExtensionAccessor> getModifierExtensions();
}
