/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.model;

import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.IAccessor;

/**
 * Release-independent wrapper for the HAPI FHIR general Property model type.
 *
 * This interface and the corresponding implementations are used to make as
 * much of the rest of the core library release-independent, i.e. shield it
 * from the various HAPI FHIR release package differences.
 *
 * @see org.hl7.fhir.r4.model.Property
 * @see org.hl7.fhir.r4b.model.Property
 * @see org.hl7.fhir.r5.model.Property
 */
public interface IPropertyAccessor extends IAccessor {

    /**
     * Returns the name of this property in the FHIR Specification.
     *
     * @return the name of this property in the FHIR Specification
     */
    String getName();

    /**
     * Returns the stated type in the FHIR specification.
     *
     * @return the stated type in the FHIR specification
     */
    String getTypeCode();

    /**
     * Returns the definition of this element in the FHIR specification.
     *
     * @return the definition of this element in the FHIR specification
     */
    String getDefinition();

    /**
     * Returns the minimum cardinality for this element.
     *
     * @return the minimum cardinality for this element
     */
    int getMinCardinality();

    /**
     * Returns the maximum cardinality for this element.
     *
     * @return the maximum cardinality for this element
     */
    int getMaxCardinality();

    /**
     * Returns the actual values.
     *
     * @return the actual values
     */
    ImmutableList<IBaseAccessor> getValues();

    /**
     * Determines whether the property has actual values.
     *
     * @return <code>true</code> if the property has actual values
     */
    boolean hasValues();

    /**
     * Determines whether the property is a list, i.e. has a maxCardinality greater
     * than one.
     *
     * @return <code>true</code> if the property is a list
     */
    boolean isList();
}
