/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.model;

import com.google.common.collect.ImmutableList;
import java.util.Optional;

/**
 * Release-independent wrapper for a FHIR CodeSystem.
 *
 * Note that in R4, the CodeSystem is derived from MetadataResource while in
 * R4B and R5, it is derived from CanonicalResource.
 *
 * This interface and the corresponding implementations are used to make as
 * much of the rest of the core library release-independent, i.e. shield it
 * from the various HAPI FHIR release package differences.
 *
 * @see org.hl7.fhir.r4.model.CodeSystem
 * @see org.hl7.fhir.r4b.model.CodeSystem
 * @see org.hl7.fhir.r5.model.CodeSystem
 */
public interface ICodeSystemAccessor extends IMetadataResourceAccessor {

    /**
     * A natural language name identifying the code system. This name should be usable as an identifier for the
     * module by machine processing applications such as code generation.
     * @return the name of the CodeSystem, if set
     * @see org.hl7.fhir.r4.model.ValueSet#hasName()
     * @see org.hl7.fhir.r4.model.ValueSet#getName()
     * @see org.hl7.fhir.r4b.model.ValueSet#hasName()
     * @see org.hl7.fhir.r4b.model.ValueSet#getName()
     * @see org.hl7.fhir.r5.model.ValueSet#hasName()
     * @see org.hl7.fhir.r5.model.ValueSet#getName()
     */
    Optional<String> getName();

    /**
     * An absolute URI that is used to identify this code system when it is referenced in a specification, model,
     * design or an instance; also called its canonical identifier.
     * @return the URL of the CodeSystem, if set
     * @see org.hl7.fhir.r4.model.ValueSet#hasUrl()
     * @see org.hl7.fhir.r4.model.ValueSet#getUrl()
     * @see org.hl7.fhir.r4b.model.ValueSet#hasUrl()
     * @see org.hl7.fhir.r4b.model.ValueSet#getUrl()
     * @see org.hl7.fhir.r5.model.ValueSet#hasUrl()
     * @see org.hl7.fhir.r5.model.ValueSet#getUrl()
     */
    Optional<String> getUrl();

    /**
     * A free text natural language description of the code system from a consumer's perspective.
     * @return the description of the CodeSystem, if set
     * @see org.hl7.fhir.r4.model.ValueSet#hasDescription()
     * @see org.hl7.fhir.r4.model.ValueSet#getDescription()
     * @see org.hl7.fhir.r4b.model.ValueSet#hasDescription()
     * @see org.hl7.fhir.r4b.model.ValueSet#getDescription()
     * @see org.hl7.fhir.r5.model.ValueSet#hasDescription()
     * @see org.hl7.fhir.r5.model.ValueSet#getDescription()
     */
    Optional<String> getDescription();

    /**
     * Concepts that are in the code system. The concept definitions are inherently hierarchical, but the definitions must be consulted to determine what the meanings of the hierarchical relationships are.
     * @return the concepts of the code system, if any
     * @see org.hl7.fhir.r4.model.CodeSystem#hasConcept()
     * @see org.hl7.fhir.r4.model.CodeSystem#getConcept()
     * @see org.hl7.fhir.r4b.model.CodeSystem#hasConcept()
     * @see org.hl7.fhir.r4b.model.CodeSystem#getConcept()
     * @see org.hl7.fhir.r5.model.CodeSystem#hasConcept()
     * @see org.hl7.fhir.r5.model.CodeSystem#getConcept()
     */
    ImmutableList<IConceptDefinitionComponentAccessor> getConcept();
}
