/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.model;

/**
 * This type represents the known values in
 * <ul>
 * <li>{@link org.hl7.fhir.r4.model.CodeSystem.FilterOperator}</li>
 * <li>{@link org.hl7.fhir.r4b.model.Enumerations.FilterOperator}</li>
 * <li>{@link org.hl7.fhir.r5.model.Enumerations.FilterOperator}</li>
 * </ul>
 */
public enum FilterOperator {
    /**
     * The specified property of the code equals the provided value.
     *
     * @see <a href="https://www.hl7.org/fhir/codesystem-filter-operator.html#filter-operator-.61">&equals;</a>
     */
    EQUAL,

    /**
     * Includes all concept ids that have a transitive is-a relationship with the concept Id provided as the value,
     * including the provided concept itself (include descendant codes and self).
     *
     * @see <a href="https://www.hl7.org/fhir/codesystem-filter-operator.html#filter-operator-is-a">is-a</a>
     */
    ISA,

    /**
     * ncludes all concept ids that have a transitive is-a relationship with the concept Id provided as the value,
     * excluding the provided concept itself (i.e. include descendant codes only).
     *
     * @see <a href="https://www.hl7.org/fhir/codesystem-filter-operator.html#filter-operator-descendent-of">descendent-of</a>
     */
    DESCENDENTOF,

    /**
     * The specified property of the code does not have an is-a relationship with the provided value.
     *
     * @see <a href="https://www.hl7.org/fhir/codesystem-filter-operator.html#filter-operator-is-not-a">is-not-a</a>
     */
    ISNOTA,

    /**
     * The specified property of the code matches the regex specified in the provided value.
     *
     * @see <a href="https://www.hl7.org/fhir/codesystem-filter-operator.html#filter-operator-regex">regex</a>
     */
    REGEX,

    /**
     * The specified property of the code is in the set of codes or concepts specified in the provided value
     * (comma-separated list).
     *
     * @see <a href="https://www.hl7.org/fhir/codesystem-filter-operator.html#filter-operator-in">in</a>
     */
    IN,

    /**
     * The specified property of the code is not in the set of codes or concepts specified in the provided value
     * (comma-separated list).
     *
     * @see <a href="https://www.hl7.org/fhir/codesystem-filter-operator.html#filter-operator-not-in">not-in</a>
     */
    NOTIN,

    /**
     * Includes all concept ids that have a transitive is-a relationship from the concept Id provided as the value,
     * including the provided concept itself (i.e. include ancestor codes and self).
     *
     * @see <a href="https://www.hl7.org/fhir/codesystem-filter-operator.html#filter-operator-generalizes">generalizes</a>
     */
    GENERALIZES,

    /**
     * Only concepts with a direct hierarchical relationship to the index code and no other concepts.
     * This does not include the index code in the output.
     *
     * @see <a href="https://www.hl7.org/fhir/codesystem-filter-operator.html#filter-operator-child-of">child-of</a>
     */
    CHILDOF,

    /**
     * Includes concept ids that have a transitive is-a relationship with the concept Id provided as the value,
     * but which do not have any concept ids with transitive is-a relationships with themselves.
     *
     * @see <a href="https://www.hl7.org/fhir/codesystem-filter-operator.html#filter-operator-descendent-leaf">descendent-leaf</a>
     */
    DESCENDENTLEAF,

    /**
     * The specified property of the code has at least one value (if the specified value is true; if the specified
     * value is false, then matches when the specified property of the code has no values).
     *
     * @see <a href="https://www.hl7.org/fhir/codesystem-filter-operator.html#filter-operator-exists">exists</a>
     */
    EXISTS;

    /**
     * Determines the enum value from the string representation
     * @param value the string representation of the resource type
     * @return the corresponding enum representation
     */
    public static FilterOperator fromString(String value) {
        return switch (value) {
            case "=" -> EQUAL;
            case "is-a" -> ISA;
            case "descendent-of" -> DESCENDENTOF;
            case "is-not-a" -> ISNOTA;
            case "regex" -> REGEX;
            case "in" -> IN;
            case "not-in" -> NOTIN;
            case "generalizes" -> GENERALIZES;
            case "child-of" -> CHILDOF;
            case "descendent-leaf" -> DESCENDENTLEAF;
            case "exists" -> EXISTS;
            default -> throw new IllegalArgumentException(String.format("Unknown FilterOperator %s", value));
        };
    }

    /**
     * Determines the string representation of an enum value
     * @return the corresponding string representation
     */
    @Override
    public String toString() {
        return switch (this) {
            case EQUAL -> "equal";
            case ISA -> "is-a";
            case DESCENDENTOF -> "descendent-of";
            case ISNOTA -> "is-not-a";
            case REGEX -> "regex";
            case IN -> "in";
            case NOTIN -> "not-in";
            case GENERALIZES -> "generalizes";
            case EXISTS -> "child-of";
            case CHILDOF -> "descendent-leaf";
            case DESCENDENTLEAF -> "exists";
        };
    }
}
