/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.model;

import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.IAccessorCache;
import java.util.Optional;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import org.hl7.fhir.r4b.model.ElementDefinition;

/**
 * Implementation of {@link IElementDefinitionAccessor} for FHIR Release R4B.
 */
@EqualsAndHashCode(callSuper = true)
public final class ElementDefinitionAccessorR4B extends BackboneTypeAccessorR4B implements IElementDefinitionAccessor {

    @Getter(AccessLevel.PRIVATE)
    private final ElementDefinition elementDefinition;

    /**
     * Creates a new wrapper.
     *
     * @param cache the cache this accessor is managed by
     * @param elementDefinition the ElementDefinition instance to wrap
     */
    public ElementDefinitionAccessorR4B(IAccessorCache cache, ElementDefinition elementDefinition) {
        super(cache, elementDefinition);
        this.elementDefinition = elementDefinition;
    }

    @Override
    public Optional<String> getId() {
        if (this.elementDefinition.hasId()) {
            return Optional.of(this.elementDefinition.getId());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getPath() {
        if (this.elementDefinition.hasPath()) {
            return Optional.of(this.elementDefinition.getPath());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<Integer> getMin() {
        if (this.elementDefinition.hasMin()) {
            return Optional.of(this.elementDefinition.getMin());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getMax() {
        if (this.elementDefinition.hasMax()) {
            return Optional.of(this.elementDefinition.getMax());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getDefinition() {
        if (this.elementDefinition.hasDefinition()) {
            return Optional.of(this.elementDefinition.getDefinition());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getShort() {
        if (this.elementDefinition.hasShort()) {
            return Optional.of(this.elementDefinition.getShort());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<ITypeAccessor> getFixed() {
        if (this.elementDefinition.hasFixed()) {
            return Optional.of(getCache().getAccessor(this.elementDefinition.getFixed(), ITypeAccessor.class));
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<ITypeAccessor> getPattern() {
        if (this.elementDefinition.hasPattern()) {
            return Optional.of(getCache().getAccessor(this.elementDefinition.getPattern(), ITypeAccessor.class));
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<IElementDefinitionBindingComponentAccessor> getBinding() {
        if (this.elementDefinition.hasBinding()) {
            return Optional.of(getCache()
                    .getAccessor(
                            this.elementDefinition.getBinding(), IElementDefinitionBindingComponentAccessor.class));
        } else {
            return Optional.empty();
        }
    }

    @Override
    public ImmutableList<IElementTypeRefComponentAccessor> getType() {
        return this.elementDefinition.getType().stream()
                .map(component -> getCache().getAccessor(component, IElementTypeRefComponentAccessor.class))
                .collect(ImmutableList.toImmutableList());
    }
}
