/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.model;

import de.fhlintstone.accessors.IAccessorCache;
import java.util.Optional;
import lombok.EqualsAndHashCode;
import org.hl7.fhir.r4b.model.ValueSet.ConceptSetFilterComponent;

/**
 * Implementation of {@link IConceptSetFilterComponentAccessor} for FHIR version
 * R4B. i.e. wrapping {@link ConceptSetFilterComponent}.
 */
@EqualsAndHashCode(callSuper = true)
public final class ConceptSetFilterComponentAccessorR4B extends BackboneElementAccessorR4B
        implements IConceptSetFilterComponentAccessor {

    private final ConceptSetFilterComponent filter;

    /**
     * Creates a new wrapper
     *
     * @param cache the cache this accessor is managed by
     * @param filter the {@link ConceptSetFilterComponent} to wrap
     */
    public ConceptSetFilterComponentAccessorR4B(IAccessorCache cache, ConceptSetFilterComponent filter) {
        super(cache, filter);
        this.filter = filter;
    }

    @Override
    public Optional<FilterOperator> getOp() {
        if (this.filter.hasOp()) {
            return switch (this.filter.getOp()) {
                case EQUAL -> Optional.of(FilterOperator.EQUAL);
                case ISA -> Optional.of(FilterOperator.ISA);
                case DESCENDENTOF -> Optional.of(FilterOperator.DESCENDENTOF);
                case ISNOTA -> Optional.of(FilterOperator.ISNOTA);
                case REGEX -> Optional.of(FilterOperator.REGEX);
                case IN -> Optional.of(FilterOperator.IN);
                case NOTIN -> Optional.of(FilterOperator.NOTIN);
                case GENERALIZES -> Optional.of(FilterOperator.GENERALIZES);
                case EXISTS -> Optional.of(FilterOperator.EXISTS);
                case NULL -> Optional.empty();
            };
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getProperty() {
        if (this.filter.hasProperty()) {
            return Optional.of(this.filter.getProperty());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getValue() {
        if (this.filter.hasValue()) {
            return Optional.of(this.filter.getValue());
        } else {
            return Optional.empty();
        }
    }
}
