/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.model;

import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.IAccessorCache;
import java.util.Optional;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import org.hl7.fhir.r5.model.ValueSet.ConceptSetComponent;

/**
 * Implementation of {@link IConceptSetComponentAccessor} for FHIR version R4.
 * i.e. wrapping {@link ConceptSetComponent}.
 */
@EqualsAndHashCode(callSuper = true)
public final class ConceptSetComponentAccessorR5 extends BackboneElementAccessorR5
        implements IConceptSetComponentAccessor {

    @Getter(AccessLevel.PRIVATE)
    private final ConceptSetComponent component;

    /**
     * Creates a new wrapper.
     *
     * @param cache the cache this accessor is managed by
     * @param component the {@link ConceptSetComponent} to wrap
     */
    public ConceptSetComponentAccessorR5(IAccessorCache cache, ConceptSetComponent component) {
        super(cache, component);
        this.component = component;
    }

    @Override
    public ImmutableList<IConceptReferenceComponentAccessor> getConcept() {
        return this.component.getConcept().stream()
                .map(concept -> getCache().getAccessor(concept, IConceptReferenceComponentAccessor.class))
                .collect(ImmutableList.toImmutableList());
    }

    @Override
    public ImmutableList<IConceptSetFilterComponentAccessor> getFilter() {
        return this.component.getFilter().stream()
                .map(filter -> getCache().getAccessor(filter, IConceptSetFilterComponentAccessor.class))
                .collect(ImmutableList.toImmutableList());
    }

    @Override
    public ImmutableList<ICanonicalTypeAccessor> getValueSet() {
        return this.component.getValueSet().stream()
                .map(filter -> getCache().getAccessor(filter, ICanonicalTypeAccessor.class))
                .collect(ImmutableList.toImmutableList());
    }

    @Override
    public Optional<String> getSystem() {
        if (this.component.hasSystem()) {
            return Optional.of(this.component.getSystem());
        } else {
            return Optional.empty();
        }
    }
}
